/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright (C) 2019-2022 Red Hat, Inc. Daniel Bristot de Oliveira  <bristot@kernel.org>
 *
 * Deterministic automata helper functions, to be used with the automata
 * models in C generated by the dot2k tool.
 */

/*
 * DECLARE_AUTOMATA_HELPERS - define a set of helper functions for automata
 *
 * Define a set of helper functions for automata. The 'name' argument is used
 * as suffix for the functions and data. These functions will handle automaton
 * with data type 'type'.
 */
#define DECLARE_AUTOMATA_HELPERS(name, type)					\
										\
/*										\
 * model_get_state_name_##name - return the (string) name of the given state	\
 */ 										\
static char *model_get_state_name_##name(enum states_##name state)		\
{										\
	if ((state < 0) || (state >= state_max_##name))				\
		return "INVALID";						\
										\
	return automaton_##name.state_names[state];				\
}										\
										\
/*										\
 * model_get_event_name_##name - return the (string) name of the given event	\
 */										\
static char *model_get_event_name_##name(enum events_##name event)		\
{										\
	if ((event < 0) || (event >= event_max_##name))				\
		return "INVALID";						\
										\
	return automaton_##name.event_names[event];				\
}										\
										\
/*										\
 * model_get_initial_state_##name - return the automaton's initial state		\
 */										\
static inline type model_get_initial_state_##name(void)				\
{										\
	return automaton_##name.initial_state;					\
}										\
										\
/*										\
 * model_get_next_state_##name - process an automaton event occurrence		\
 *										\
 * Given the current state (curr_state) and the event (event), returns		\
 * the next state, or INVALID_STATE in case of error.				\
 */										\
static inline type model_get_next_state_##name(enum states_##name curr_state,	\
					       enum events_##name event)	\
{										\
	if ((curr_state < 0) || (curr_state >= state_max_##name))		\
		return INVALID_STATE;						\
										\
	if ((event < 0) || (event >= event_max_##name))				\
		return INVALID_STATE;						\
										\
	return automaton_##name.function[curr_state][event];			\
}										\
										\
/*										\
 * model_is_final_state_##name - check if the given state is a final state	\
 */										\
static inline bool model_is_final_state_##name(enum states_##name state)	\
{										\
	if ((state < 0) || (state >= state_max_##name))				\
		return 0;							\
										\
	return automaton_##name.final_states[state];				\
}
