/* Alex Dickinson
   Procedures for setting and resetting UNIX tty characteristics.
   Interesting functions are:
      savetty();
      restoretty();
      echoon();
      echooff();
      singlecharon();
      singlecharoff();
      buffercount();
      suspend();
   A side effect of calling savetty() is to set up signal handling to reset
   the terminal characteristics appropriately for the various interrupt
   signals.
*/

#include <stdio.h>
#include <sgtty.h>
#include <signal.h>

struct sgttyb initchars; /* store initial terminal characteristics */

/* Save the original tty characteristics and set up the signalling. */
savetty()
{
  ioctl(1, TIOCGETP, &initchars);
  setsignals();
}

/* Restore the original tty characteristics. */
restoretty()
{
  ioctl(1, TIOCSETN, &initchars);
}

/* Set driver to read characters as they are typed without waiting for a
   terminator. Echo remains unchanged. */
singlecharon()
{
  struct sgttyb s;
  ioctl(1, TIOCGETP, &s);
  s.sg_flags |= CBREAK;
  ioctl(1, TIOCSETN, &s);
  fflush(stdin);
}

/* Turn off single character read mode. */
singlecharoff()
{
  struct sgttyb s;
  ioctl(1, TIOCGETP, &s);
  s.sg_flags &= ~CBREAK;
  ioctl(1, TIOCSETN, &s);
}

/* Turn character echoing on. */
echoon()
{
  struct sgttyb s;
  ioctl(1, TIOCGETP, &s);
  s.sg_flags |= ECHO;
  ioctl(1, TIOCSETN, &s);
}

/* Turn character echoing off. */
echooff()
{
  struct sgttyb s;
  ioctl(1, TIOCGETP, &s);
  s.sg_flags &= ~ECHO;
  ioctl(1, TIOCSETN, &s);
 }

/* Return the number of characters currently in the input buffer. */
int buffercount()
{
  long count;
  ioctl(1, FIONREAD, &count);
  return(count);
}

/* Yukko */
realungetc(ch, filed)
     char ch;
     int filed;
{
  ioctl(filed, TIOCSTI, &ch);
}

/* Catch signals from tty.
   If sig is an interrupt, put 0 and terminator into the buffer.
   Otherwise it was a suspend, so we put 1 and terminator into buffer. */
handleint(sig)
     int sig;
{
  fflush(stdin);
  if (sig == SIGINT)
    {
      realungetc(0, 0);
      realungetc(13, 0);
    }
  else
    {
      realungetc(1, 0);
      realungetc(13, 0);
    }
}

/* Signal initialization. */
setsignals()
{
  signal(SIGINT, handleint);
  signal(SIGTSTP, handleint);
}

/* Suspend the process */
suspend()
{
  signal(SIGTSTP, SIG_DFL);
  kill(0, SIGTSTP);
  /* resumed again, goody! */
  setsignals();
}


/* Dummy "implementation block" required by linker. */
unixio__init()
{}

