/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "cpptcl_init.cc"
 *					created: 25/4/96 {5:57:13 pm} 
 *				  last update: 09/06/98 {14:14:36 PM} 
 *	Author:	Vince Darley
 *	E-mail:	<mailto:darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www: <http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */

#include "cpptclInt.h"
#include "cpptcl_init.h"
#include "cpptcl_info.h"
#include <assert.h>
#include "cpptcl_metaobject.h"
#include "cpptcl_members.h"

#ifndef CPP_NO_TEST
#include "cpptcl_test.h"
#endif

static cpptcl_info* cpptcl_controller=0;

extern "C" int Cpptcl_Init(Tcl_Interp* interp) {
    return Cpptcl_PkgInit(interp,NULL);
}

/* Hash table initialisation flag */
static bool cpptcl_hash_table_ready; 

int Cpptcl_Init(tcl_obj& interp) {
#ifdef USE_TCL_STUBS
	Tcl_InitStubs(interp,TCL_VERSION,0);
#endif
    /* Create hash table on first call */
    if (!cpptcl_hash_table_ready) {
	cpptcl_hash_table_ready = true;
	meta_object::register_tcl_types();
    }
    	
    if(!cpptcl_controller)
	cpptcl_controller = new cpptcl_info(interp,"cpptcl");
    if(!tcl_base::metaobject)
	tcl_base::update_metaobject(interp);

#ifndef CPP_NO_TEST
    Cpptcl_Object(cpp_test,tcl_base);
#endif
    	
    interp.PkgProvide("Cpptcl",CPPTCL_VERSION);
    if(interp.MakeNamespace("::cpp") == TCL_ERROR)
	return TCL_ERROR;
    return interp.PackageLibraryInit("::cpp::library","CPPTCL_LIBRARY","cpptcl",
				     "cpptcl.tcl",CPPTCL_VERSION,"Cpptcl");
}

int Cpptcl_PkgInit(Tcl_Interp* interp, Cpptcl_PackageInitProc cpx_iproc) {
    tcl_obj& tcl_ = cpptcl_create_stream(interp);
    // make sure Cpptcl is initialised before calling the package proc
    // but don't initialise Cpptcl twice if we're actually starting up
    // for the first time.
    if (Cpptcl_Init(tcl_) == TCL_ERROR) return TCL_ERROR;
    
    if (cpx_iproc && ((*cpx_iproc)(tcl_) == TCL_ERROR)) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

typedef struct tcl_obj_link  {
    tcl_obj* stream;
    tcl_obj_link* next;
    int refs;
    tcl_obj_link(tcl_obj* s){ stream = s; next = 0; refs=0;}
    ~tcl_obj_link(void) {}
    const tcl_obj_link& operator= (tcl_obj_link& t) {
	stream = t.stream;
	next = t.next;
	refs = t.refs;
	return *this;
    }
	
} tcl_obj_link;

tcl_obj* cpptcl_obj_memory(Tcl_Interp* interp, bool add_not_delete);
void cpptcl_delete_stream(tcl_obj& i);

/** 
 * -------------------------------------------------------------------------
 *	 
 * "cpptcl_create_stream" --
 *  
 *  Wraps a tcl interpreter (type Tcl_Interp) inside a tcl_obj.  Checks to 
 *  see if there already exists a stream with that interpreter, and if so 
 *  returns the already-created tcl_obj.  This is done because confusion 
 *  could arise if two tcl_objs manipulated the same interpreter (probably 
 *  hard to do with the current setup, but easy in a multi-threaded scenario).
 *         
 * Results:
 *  Returns a valid tcl_obj which wraps the given Tcl_Interp
 *  
 * Side effects:
 *  May create a new tcl_obj, and may add to its static list of known
 *  interpreter/stream pairs.
 * -------------------------------------------------------------------------
 */
tcl_obj& cpptcl_create_stream(Tcl_Interp* interp) {
    return *(cpptcl_obj_memory(interp,true));
}

void cpptcl_delete_stream(tcl_obj& i) {
    (void) cpptcl_obj_memory(i.interpreter(),false);
}

tcl_obj* cpptcl_obj_memory(Tcl_Interp* interp, bool add_not_delete) {
    static tcl_obj_link all_tcl_objs = tcl_obj_link(0);
    tcl_obj_link *i;
    tcl_obj_link *j=0;
    for (i = &all_tcl_objs; i->stream !=0; i = i->next) {
	if(i->stream->interpreter() == interp) {
	    j = i;
	    break;
	}
    }
    if(add_not_delete) {
	// we're creating a stream
	if(j) {
	    j->refs++;
	} else {
	    // 'i' is at the end of the list
	    j = i;
	    j->stream = new tcl_obj(interp);
	    j->next = new tcl_obj_link(0);
	}
	return j->stream;
    } else {
	// we delete an old stream
	assert(j!=0);
	if(j->refs) {
	    j->refs--;
	    return 0;
			
	} else {
	    // delete the interp too
	    if(j == &all_tcl_objs) {
		all_tcl_objs = *(all_tcl_objs.next);
		delete j;
		return 0;
	    } else {
		for (i = &all_tcl_objs; i->stream !=0; i = i->next) {
		    if(i->next == j) {
			i->next = j->next;
			delete j;
			return 0;
		    }
		}
		// error
		assert(0==1);
		return 0;
	    }
	}
    }
}





