/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* MESSAGE.C --- Plain text messages with formating. */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdarg.h>
#include <graphics.h>
#include "window.h"
#include "resource.h"
#include "key.h"

#define MSG_X_MARGIN 8
#define MSG_Y_MARGIN 6
/* There is no check for this. Best we
   can do is die when it's exceeded. */
#define MAX_MESSAGE_SIZE 2048
#define MAX_MESSAGE_LINES 60

void handle_press(ENV env);
static MESSAGE_BUTTON button_tags[] = { mYES, mNO, mOK, mCANCEL, };

#if sizeof button_tags / sizeof button_tags[0] != mN
#error Message button tables inconsistent
#endif

DefBitmapButton(msg_yes,    yes,    'y', 2, 2, Enabled, Closure(handle_press, &button_tags[mYES]))
DefBitmapButton(msg_no,     no,     'n', 2, 2, Enabled, Closure(handle_press, &button_tags[mNO]))
DefBitmapButton(msg_ok,     ok,     'k', 2, 2, Enabled, Closure(handle_press, &button_tags[mOK]))
DefBitmapButton(msg_cancel, cancel, ESC, 2, 2, Enabled, Closure(handle_press, &button_tags[mCANCEL]))

/* Text string store is allocated dynamically. */
static TEXT_REC msg_text[1] = {{ _W_, NULL, 0, CENTER_TEXT }};

/* Order of buttons here must agree with MESSAGE_BUTTON enumeration. 
   Coordinates of buttons are computed dynamically. */

BeginDefDialog(msg_dialog)
  DefDialogItem(ButtonItem, msg_yes,    0, 0)
  DefDialogItem(ButtonItem, msg_no,     0, 0)
  DefDialogItem(ButtonItem, msg_ok,     0, 0)
  DefDialogItem(ButtonItem, msg_cancel, 0, 0)
  DefDialogItem(TextItem,   msg_text,   MSG_X_MARGIN, MSG_Y_MARGIN)
EndDefDialog(msg_dialog, NULL, Enabled)

static void handle_press(ENV env)
{
  stop_modal_dialog(msg_dialog, *(int*)env);
}

/* A compact way to expand tab escapes in place:
   Find the next tab escape. Call recursively to move
   the text after the escape forward out of the way,
   move the text at `from' forward to `to', fill 
   with spaces up to the tab, and return.  The
   recursion is grounded when we find no tab escape. */

LOCAL(char *) expand(char *line_start, char *from, char *to)
{
  int tab, n_spaces;
  char ch, *p, *end;

  /* Find end of area we're expanding. */
  for (p = from; (ch = *p) != TAB_ESC_CHAR; ++p) {

    /* Base case: no tab escape. */
    if (ch == '\0') {
      memmove(to, from, p - from + 1);
      return p;
    }
    else if (ch == '\n')
      line_start = p + 1;
  }

  /* Tab escape is now at p. Get the tab position. */
  tab = strtol(p + 1, &end, 10);

  assert(tab >= 0);
  if (*end == TAB_ESC_CHAR)
    ++end;

  /* Compute (possibly negative) number of spaces to tab. */
  n_spaces = tab - (p - line_start);

  /* Move the area after the tab command by recurring. */
  end = expand(line_start, end, p + n_spaces);

  /* Now it's safe to move the text before 
     the tab escape and insert spaces. */
  memmove(to, from, p - from);
  if (n_spaces > 0)
    memset(p, ' ', n_spaces);
  return end;
}

/* Pop up a message window and block waiting for a button
   press. Several buttons are available and any or all may
   be enabled.  The format string is as for printf with
   augmentations:

     1. If the first two chars are @< or @>, then multiple
     line messages are left or right justified respectively.
     By default lines are each centered in the box.  This
     has no effect on single line messages.

     2. If @n or @n@ appear anywhere in the text (where n
     is an integer), then spaces are inserted to tab to
     position n of the line, with 0 being the first position.
     Thus with left-justification you can create tabular messages. 
     The second form is useful when a number follows the 
     tab escape in the output message.  Tab spaces are inserted
     _after_ printf expansion, so don't say @30%d unless you
     _want_ the tab position to be 30ddd where ddd is 
     the printf expansion of %d (probably _not_ the case!).
     Use @30@%d instead.  Also, it's ok to use tab escapes to
     `back up,' but the message will show the text _left_ 
     of the tab escape.

   Note: As we are using strtok to find lines, multiple 
   adjacent \n's are all treated as one.  To get blank lines
   in the message, use a newline followed by a space. 

   Note: This function doesn't know about tab characters. */

MESSAGE_BUTTON message(unsigned buttons, WINDOW parent, char *fmt, ... )
{
  va_list ap;
  int i, n, x;
  unsigned b;
  char *buf, *p, **text;
  struct pointtype size;
  MESSAGE_BUTTON pressed;

  /* Look for @justification escape. */
  msg_text[0].justify = CENTER_TEXT;
  if (fmt[0] == TAB_ESC_CHAR)
    switch(fmt[1]) {
      case '<':
        msg_text[0].justify = LEFT_TEXT;
        fmt += 2;
        break;
      case '>':
        msg_text[0].justify = RIGHT_TEXT;
        fmt += 2;
        break;
    }

  /* Get a big buffer for the message. */
  buf = malloc(MAX_MESSAGE_SIZE);

  /* Expand the printf escapes. */
  va_start(ap, fmt);
  vsprintf(buf, fmt, ap);
  va_end(ap);

  /* Expand @n tab escapes. */
  n = expand(buf, buf, buf) - buf + 1;
  assert(n < MAX_MESSAGE_SIZE);

  /* Can't free correctly in this case because
     text[0] must point to the start of the buffer
     and strtok won't provide this. */
  assert(buf[0] != '\n');

  /* Trim the buffer to the actual message size. */
  buf = realloc(buf, n);

  /* Get big buffer for pointers to lines. */
  text = malloc(MAX_MESSAGE_LINES * sizeof(char*));

  /* Set up array of pointers to lines. We use library
     function strtok to get stretches of text between
     newlines and set newlines to \0 as required. */
  for (n = 0, p = strtok(buf, "\n"); p != NULL; ++n, p = strtok(NULL, "\n")) {
    assert(n < MAX_MESSAGE_LINES);
    text[n] = p;
  }

  /* Trim line buffer to size. */
  msg_text->lines = realloc(text, n * sizeof(char*));

  /* Fill in text structure with number of lines. */
  msg_text->n_lines = n;


  /* Set message window geometry. */
  size = text_size(msg_text);
  x = MSG_X_MARGIN;

  for (i = 0, b = 1; i < mN; ++i, b <<= 1) {
    DIALOG_ITEM di = &msg_dialog->items[i];

    /* If user requested button, enable it and
       account for its size in next button position. */
    if (b & buttons) {
      di->y = size.y + 2*MSG_Y_MARGIN;
      di->x = x;
      x += button_width((BUTTON)di->item) + MSG_X_MARGIN;
      di->status |= bit(diENABLED);
    }
    else
      di->status &= notbit(diENABLED);
  }

  open_dialog(msg_dialog, CENTER_WINDOW, CENTER_WINDOW, parent);

  pressed = run_modal_dialog(msg_dialog);

  free(msg_text->lines[0]);
  free(msg_text->lines);
  destroy_window(&msg_dialog->window);
  return pressed;
}
