package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`%CP3V0``````````/``+@(+`@(H`$(```#,`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```RR8!``,``(``
M`"`````````0```````````0````````$``````````````0```````!`+T!
M````$`$`]`H````@`0!0*P```-```-0!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MV!(!`'`"```````````````````````````````````N=&5X=````/A!````
M$````$(````$``````````````````!@``!@+F1A=&$```"``````&`````"
M````1@``````````````````0```P"YR9&%T80``+$4```!P````1@```$@`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#4`0```-`````"````D```````````````
M````0```0"YX9&%T80``T`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``"]`0`````!```"````E```````````````````0```0"YI9&%T80``
M]`H````0`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"B``````````````````!```#`+G)E;&]C```P`````%`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0V%/P``
MZ.`Y``!%,<`QTC')Z/0Y``!%,<`QTC')Z/@Y``!%,<`QTC')Z/PY``!%,<`Q
MTC')2(/$*.G\.0``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.D;-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%WJ```$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'HPC@``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQW8GP``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.@3-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GH%S<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ'8W``!(B?%)B<;H:S<``$&-
M3`8"2&/)Z$XV``!)B?%-B>!(B>I)B<9(B<'H^C8``$R)\4B)^NC_-@``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z'(W``!,B?'HJC4``$R)Z>B"-@``2(7`
M#X5K____9@\?A```````3(GIZ%@U``!,B>'HD#8``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z/PT``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=;YX``$B+`TB)A"2H````,<!(C50D($F)S.AE
M-0``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'HJC0`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^AQ-```D%6X
M.``!`%=64^C2,P``2"G$2(LUZ)T``#'20;@"````2(L&2(F$)"@``0`QP(G-
M2(U<)"#HQS0``$B->/](B?I(P?H_2,'J,$B-!!</M\!(*=!(*<</B(4````/
M'T0``$4QP$B)^HGIZ),T``!!N`@``0!(B=J)Z>@3-0``2)A(@_@'=DM(C40#
M^$@YV')!2(T5&DP``.L1#Q^$``````!(@^@!2#G8<B=(BPI(.0AU[T@IV$@!
M^$B+E"0H``$`2"L6=25(@<0X``$`6UY?7<-(@>\```$`2('_``#__W6`2,?`
M_____^O.Z((S``"0D$%4N#`0``!55U93Z.`R``!(*<1(BS7VG```0;C_#P``
M2(L&2(F$)"@0```QP$B-7"0@28G,2(G52(T-=UL``$B)VNAY-```A<!^18U0
M_TAC^H!\/"``=7"%TG0T28GXNB\```!(B=GHU#,``$B%P'0?2(L-B)P``.AC
M,@``28GY28G83(GB2(L(Z-HQ``#K&DB+#6F<``#H1#(``$F)Z$R)XDB+".C&
M,0``2(N$)"@0``!(*P9U%DB!Q#`0``!;7E]=05S##Q\`2&/XZX_HKC(``)!F
M9BX/'X0``````&:005=!5D%505155U932(/L.$B+'1F<``!(BP-(B40D*#'`
M,<"`.CU(B=8/E,!(B<](`<9(B?'H)C0``$B-5"0D2(GY2(G%Z`;[__])B<1(
MA<`/A/H```!(B<'H`C0``$@YQ0^&R0```$QC?"0D3(LMW9L``$G!YP,/M@>$
MP`^$C@$``#P]#X2&`0``2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+
M90!(C4PH`NB<,@``30'\28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<
MZQYF#Q^$```````/M@%(@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V
M#`:(3`(!2(/``83)=?!(BT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#
MD#'`9@\?1```#[84!D&(%`1(@\`!A-)U\.O&9@\?1```3(LM^9H``$V+=0!)
M@SX`#X2R````3(GP13'D#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!
MX@.+!3_9``"%P'53QP4QV0```0```$B)T>BI,0``2(G!2(7`#X1=____36/\
M3(GR2<'G`TV)^.B[,0``28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^
M__\/'P!,B?'H`#(``$F)10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````
M,<#IC_[__[H0````13'DZ6?____H=S```)!F#Q]$``!32(/L,$B+#>29``#H
MOR\``$R-#5A;``!,C04A+P``2(L(2(T5C%@``$B)P^A'+P``3(T-F5@``,=$
M)"@`````3(T%*1$``$R)3"0@2(L+2(T5?%@``.@5+P``D$B#Q#!;PV9F+@\?
MA```````#Q\`2(/L*$B+#769``#H4"\``$R-#>E:``!,C06R+@``2(L(2(T5
M'5@``$B#Q"CIURX``&9F+@\?A```````#Q]``#'`PV9F+@\?A```````9I!!
M5[C8$```059!54%455=64^@*+P``2"G$3(LM()D``$F+10!(B80DR!```#'`
M2(UL)$!(B<Y)B=1(C0WI5P``2(GJZ!CX__^Z+P```$B)\4B)P^CH,```2(7`
M=#M(B?)(C0W+5P``Z*3\__](B?!(BY0DR!```$DK50`/A7P!``!(@<38$```
M6UY?74%<05U!7D%?PP\?`$R)X4B-/9U7``!,C26?5P``Z+(P``!(B?I(B<'H
M]S```$F)QTB%P'2<08`_``^$UP````\?1```2(7;=!-,B?I(B=GH<#```(7`
M#X2:````3(GYZ(`P``!)C50'_TF)QDDYUW(/ZQV0Q@(`2(/J`4DYUW0%@#HO
M=.],B?GH5S```$F)QDB)\>A,,```28U$!@%(/?\/```/ARC___](B70D,#'2
M3(VT),````!,C0WV5@``3(ED)"A,B?%!N``0``!,B7PD(.@]+@``2(GJ3(GQ
MZ,(O``"%P'40BT0D4"4`\```/0"```!T3DB)^C')Z"0P``!)B<=(A<`/A,7^
M__]!@#\`#X4N____2(7;=`6`.RYT%$&^`0```$R-/6Y6``#I7O___V:0@'L!
M`'2\Z^0/'X0``````+H!````3(GQZ.,M``"%P'6A3(GR2(T-1U8``.@@^___
M3(GQZ&@O``#I<O[__^BN+0``D&9F+@\?A```````9I!!5$B#[""Z+P```$F)
MS.AM+P``2(U0`4B%P$P/1>),B>%(@\0@05SI)"\```\?0`!!5;@X$```0516
M4^C0+```2"G$2(L=YI8``$B+`TB)A"0H$```,<!)B<U(A<D/A)T```"`.0`/
MA)0```#H[RX``$R-0`%(B<9)@?@`$```#X>S````3(UD)"!!N0`0``!,B>I,
MB>'HY2P``$F-1#3_3#G@=Q?K&F8/'X0``````,8``$B#Z`%,.>!T!8`X+W3O
MNB\```!,B>'HLBX``$B%P'0M3#G@=UA,B>'H<"X``$B+E"0H$```2"L3=4](
M@<0X$```6UY!7$%=PP\?1```2(N$)"@0``!(*P-U+TB-#0-5``!(@<0X$```
M6UY!7$%=Z2HN``!F+@\?A```````Q@``ZZ,/'P`QP.NDZ%\L``"09F8N#Q^$
M```````/'P!!5T%6055!5%575E-(@^PX2(LMR94``$B+->*5``!(C1W%5```
M2(M%`$B)1"0H,<!,C60D)$R)XDB)V>C`]/__2(7`=#M(8U0D)$B+!DB-!-!(
MBU`(2(D02(72=-@/'P!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z(7T__](A<!U
MQ4B+-7F5``!(C1UE5```3(GB2(G9Z&?T__](A<!T0DAC5"0D2(L&2(T$T$B+
M4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z"7T
M__](A<!UODB+-1F5``!(C1T-5```3(GB2(G9Z`?T__](A<!T0DAC5"0D2(L&
M2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB
M2(G9Z,7S__](A<!UODB+-;F4``!(C1VV4P``3(GB2(G9Z*?S__](A<!T0DAC
M5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%
MTG7P3(GB2(G9Z&7S__](A<!UODB+'5F4``!,C2U=4P``3(GB3(GIZ$?S__](
MA<!T0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`
M"$B)$$B%TG7P3(GB3(GIZ`7S__](A<!UODB+-?F3``!(C1T-4P``3(GB2(G9
MZ.?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(
MBU`02(/`"$B)$$B%TG7P3(GB2(G9Z*7R__](A<!UODB+'9F3``!,C35;4@``
M3(GB3(GQZ(?R__](A<!T0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GQZ$7R__](A<!UODB+-3F3``!(
MC1U94@``3(GB2(G9Z"?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z.7Q__](A<!UODB+
M-=F2``!,C3T#4@``3(GB3(GYZ,?Q__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(
MB1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GYZ(7Q__](
MA<!UODB+/7F2``!(C36M40``3(GB2(GQZ&?Q__](A<!T0DAC5"0D2(L'2(T$
MT$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQ
MZ"7Q__](A<!UODB+/1F2``!(C37D4```3(GB2(GQZ`?Q__](A<!T0DAC5"0D
M2(L'2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P
M3(GB2(GQZ,7P__](A<!UODR)XDB-#?M0``#HL?#__TB)PDB%P'0(3(GYZ%'U
M__],B>)(C0WM4```Z)+P__](B<)(A<!T#$B-#>M0``#H+O7__TR)XDB-#>=0
M``#H;_#__TB)PDB%P'0R3(GQZ`_U__](BT0D*$@K10!U0$B-%>%0``!,B>E(
M@\0X6UY?74%<05U!7D%?Z>7T__],B>)(C0VN4```Z";P__](B<)(A<!TOTB)
MV>C&]/__Z[7H7R<``)!F9BX/'X0```````\?`%-(@^PP2(L=U)```$B-#110
M``!(BP-(B40D*#'`2(U4)"3HV>___TB%P'0L#[80@/HP#Y3`A-(/E,()T(/P
M`0^VP$B+5"0H2"L3=1!(@\0P6\-F#Q]$```QP.OFZ.\F``"09F8N#Q^$````
M```/'P!(@^PHZ%<G``")P>A`)P``2(7`=`-(BP!(@\0HPP\?`$%6055!5%93
M2(/L0$B+-3V0``!,C2WL3P``2(L&2(E$)#@QP$F)S$B-5"0T3(GIZ#SO__](
MB<-(A<!T!8`X`'4O2(M$)#A(*P8/A:$```!,B>),B>E(@\1`6UY!7$%=05[I
MNO/__V8N#Q^$``````!,B>)(B<'H-2@``$B%P'0@2(M$)#A(*P9U9DB#Q$!;
M7D%<05U!7L-F#Q^$``````!,B>'HV"<``$B)V4F)QNC-)P``28U,!@+HLR8`
M`$B)7"0@38G@3(T-@TX``$B-%7Y.``!(B<%)B<;H4B<``$B+1"0X2"L&=0A,
MB?+I8O___^C+)0``D&8N#Q^$``````!!5T%6055!5%575E-(@>R8`0``2(L]
M-H\``$R--1A.``!(BP=(B80DB`$``$B-!01.``!(B<Y,C6PD4$R)\4C'A"3@
M`````````$R)ZDC'A"0X`0```````$C'A"0``0```````$B)A"38````2(T%
MJ$X``$B)A"08`0``2(T%H$X``$B)A"0@`0``2(T%F4X``$B)A"0H`0``2(T%
MCTX``$B)A"0P`0``2(T%A$X``$B)A"3X````Z+/M__])B<1(A<!T"8`X``^%
M@@0``.AM)0``B<'H5B4``$B%P`^$[0,``$B+*$B%[0^$X0,``$B)Z>B))@``
M2(U,``'H;R4``$0/MD4`2(G#183`=#-)B=],C24E3@``#Q^$``````!%#[;`
M3(GY3(GB2(/%`>C])0``1`^V10!)@\<"183`==](C:PD$`$``$B-#8A-``#K
M%V8/'T0``$B+30A(@\4(2(7)#X0'`0``3(GJZ/_L__])B<1(A<!TWX`X`'3:
M2(G!Z$KO__^%P'3.3(GAZ-XE``!(B<=(A<`/A-(```!FD$B)^>C8)0``2(G9
M2(G%Z,TE``!$C:0%!`0``$UCY$R)X>BJ)```2(E<)"A(C15P30``28GX28G'
M2(T%;$T``$R-#7),``!(B40D($R)^>@])0``NL`!``!,B?GHL"0``(/X_P^$
M?P(``$R)ZDR)^>A,)```B<.%P`^$P@```$B+'D4QY.C7(@``38GY2(T574T`
M`$B+2!A)B=CHP2,``$B+/0J-``!(BX0DB`$``$@K!P^%5`0``$R)X$B!Q)@!
M``!;7E]=05Q!74%>05_#9I`Q[4R-)9Y,``#K'`\?1```2&/%3(NDQ-````!-
MA>1T0$&`/"0`=#E,B>&#Q0'H+N[__X7`=-I,B>'HPB0``$B)QTB%P`^%YO[_
M_TACQ4R+I,30````387D=<9F#Q]$```Q_^G)_O__9@\?A```````BT0D8"4`
M\```/0!````/A2K___^+;"1HZ$,C```YQ0^%&?___XM$)&`E_P$``#W``0``
M#X4%____3(GAZ$\C``!,B>I(C0W;2P``28G$Z#WK__](B<)(A<`/A%$"``!(
MBP[HR?+__TB)Q4B%P`^$/0(``$B)Z>@E)```28G`@_@'?CU(F$B-%7=,``!,
MB40D2$B-3`7XZ.4C``"%P'4A3(M$)$A!@_@(#X3B`@``08U`]TB8@'P%`"\/
MA-$"``"02(TU\$H``$R)ZDB)\>B^ZO__2(7`#X3E`0``#[8`A,`/A-H!```\
M,`^$T@$``$B)\4B-%2Q+``!(C35"2@``Z#SO__],C2UM2@``2(TM(TP``.@I
M(@``2(ET)"A-B?A,B>&)1"0@3(T-2TH``$B-%?-+``#H%R,``.LU#Q]$``#H
M6R$``(,X$74XZ/$A``"#PP%(B70D,$V)Z8E<)"A-B?A(B>I,B>&)1"0@Z.`B
M``"ZP`$``$R)X>A3(@``@_C_=+Y,B?GHIB$``$R)\4R)XNBK[O__3(GAZ(/Z
M___IO?W__V8/'T0``.CS(```@S@1#X1S_?__Z.4@``!(BQY%,>2+*.A8(```
M38GY28G82(T5FTH``$B+2!B);"0@Z#XA``#I>/W__V8/'X0``````#';2(T-
M,DH``.L5#Q]$``!(8\-(BXS$\````$B%R70Y3(GJ@\,!Z'7I__](A<!TX(`X
M`'3;2(G!Z&,B``!(B<5(A<`/A=;[__](8\-(BXS$\````$B%R77'N0T```!(
MC2W;20``Z#8A``!!N%,```!(B</IQOO__P\?A```````2(G!Z*CY__](BSWQ
MB0``2(N$)(@!``!(*P</A3L!``!,B>%(@<28`0``6UY?74%<05U!7D%?Z>4A
M```/'T0``$B++NF[_?__#Q^$``````"Z```!`$B)Z>@3(0``08G%A<`/A!;^
M__^)P>B1Z___2(G%2(7`#X@#_O__13'`2(U0]D2)Z>AV(```2(V4)$H!``!!
MN`8```!$B>GH\"```(&\)$H!````0T%##X7-_?__9H&\)$X!``!(10^%O?W_
M_TB-5<Y%,<!$B>GH,"```$B-G"10`0``1(GI0;@H````2(G:Z*<@``!(B5PD
M($V)^$R)X4B-!=9'``!,C0T&2```QH0D>`$```!(B40D*$B-%9))``#HQ2``
M`.GR_?__2(MV"$B%]@^$(_W__TB)\>C[(```2(/X`P^&$?W__TB-3`;\2(T5
M44D``.C`(```A<!(#T3NZ?7\___H$!\``)!F9BX/'X0```````\?0`!!5%=6
M4TB#[#A(BPUPB```2(L]>8@``$B+!TB)1"0H,<!,C60D).@U'@``2(T-B4<`
M`$B)PTB+``^VD+````"(D+$```!,B>+H8^?__TB%P'0)@#@Q#X15`0``Z%#R
M__](BPM!N0,```!!N`$```!(C17P2```Z)T=``!(B<9(A<!T)TB+0!!(BPM,
MC0772```2(L0Z%<=``!(BT802(L0]D(.0`^%-@$``$B+"T&Y`P```$&X`0``
M`$B-%:U(``#H4QT``$B)QDB%P'0J3(GB2(T-JT8``.C4YO__28G`2(M&$$V%
MP`^$U````$B+$$B+"^CY'```2(L+0;@!````2(T5:$@``.@4'0``2(L+3(T%
M;D@``$B)PNC2'```2(L#2(T-448``$R)XL:`L0````#H>>;__TB)!?K%``!(
MB<%(A<!T.KK``0``Z*`>``"-4`&#^@%V"8/X$0^%C````$B+1"0H2"L'#X6>
M````2(/$.%M>7T%<PP\?@`````!(BPVYQ0``Z+3W__](B06EQ0``2(G!2(7`
M=:M(C0WN1P``Z($<``!F#Q^$``````"`>`$`#X2F_O__Z9S^__^02(L5><4`
M`$B+"$B+$NBNZ?__Z2#___]F#Q^$``````!(BPOH*!P``.F]_O__08G!2(L%
M2<4``$R+!3K%``!(C0VS1P``2(L0Z!L<``#H[AP``)!F9BX/'X0``````&:0
M055!5%-(@^PP2(L=8(8``$B+`TB)1"0H,<!)B<WH3N___[HO````2(G!28G$
MZ(X>``!(C5`!2(7`3`]%XDR)X>A+'@``2(U4)"1(C0UF10``28G$Z#?E__](
MA<!T(@^V$(32#Y3`@/HP#Y3""-!U$$V%[70+08!]``!U'`\?0`!(BT0D*$@K
M`W5"2(/$,%M!7$%=PP\?0`!!N`0```!(C17&10``3(GAZ`,>``"%P'7/2(M$
M)"A(*P-U$4R)Z4B#Q#!;05Q!7>FTY?__Z`\<``"0D)"0D)"0D)"0D)"0D)!!
M5T%6055!5%575E-(@>RX`@``2(L]=H4``$B+!TB)A"2H`@``,<!(B<I,C5DX
M2(U,)"!(C9PDH````$F)RF:008L#38L+28/"$$F#PP@/R(G`38G(28E"\$R)
MR$G!Z"!(P>@H2<'I."4`_P``3`G(38G!2<'@"$G!X1A!@>```/\`18G)3`G(
M3`G`28E"^$DYVG6L3(V)``(```\?1```2(M!:$@S04!(@\$(2#-!"$@S0?A,
MC00`2,'H'TP)P(/@_TB)07A,.<EUU4R+&DB+<A!(BUH@2(MZ&$R)V$R)V4B+
M:@A,BU0D*$C!Z1M(P>`%2`G(`T0D($Z-E!>9>8):C8P8F7F"6DB)\$@Q^$@A
MZ$@Q^$2-#`%(B>E(B>A(P>@"2,'A'DR)RT@)P4B)\$C!ZQM!B<A,,<!,(=A(
M,?!,`=!-B<I)P>(%20G:1HTD$$R)V$V)VDG!Z@)(P>`>38GE3`G03(M4)#!)
MP>T;2#'!B<-,(<E.C906F7F"6DPQP4P!T4V)XDG!X@5-">I,BVPD.$$!RDR)
MR4G!Z0)(P>$>18G23XV$*)EY@EI,"<E-B=5(,<A!B<E,(>!)P>T;2#'83`'`
M38G02<'@!4T)Z$R+;"1`00'`3(G@2<'L`DC!X!Y%B<!*C9PKF7F"6DP)X$V)
MQ4@QP4G![1M!B<1,(=%,,<E(`=E,B<-(P>,%3`GK3(ML)$@!RTR)T4G!Z@)(
MP>$>B=M/C8PIF7F"6DP)T4F)W4@QR$G![1M!B<I,(<!,,>!,`<A)B=E)P>$%
M30GI3(ML)%!!`<%,B<!)P>@"2,'@'D6)R4^-I"R9>8):3`G`38G-2#'!2<'M
M&T&)P$@AV4PQT4P!X4V)S$G!Y`5-">Q,BVPD6$$!S$B)V4C!ZP)(P>$>18GD
M3XV4*IEY@EI("=E-B>5(,<A)P>T;B<M,(<A,,<!,`=!-B>))P>(%30GJ3(ML
M)&!!`<),B<A)P>D"2,'@'D6)TD^-A"B9>8):3`G(38G52#'!2<'M&T&)P4PA
MX4@QV4P!P4V)T$G!X`5-">A,BVPD:$$!R$R)X4G![`)(P>$>18G`2HV<*YEY
M@EI,">%-B<5(,<A)P>T;08G,3"'03#'(2`'83(G#2,'C!4P)ZTR+;"1P`<-,
MB=!)P>H"2,'@'HG;3XV,*9EY@EI,"=!)B=U(,<%)P>T;08G"3"'!3#'A3`')
M28G92<'A!4T)Z4R+;"1X00')3(G!2<'H`DC!X1Y%B<E/C:0LF7F"6DP)P4V)
MS4@QR$G![1M!B<A((=A,,=!,`>!-B<Q)P>0%30GL3(NL)(````!!`<1(B=A(
MP>`>2,'K`D6)Y$^-E"J9>8):2`G838GE2#'!2<'M&XG#3"')3#'!3`'138GB
M2<'B!4T)ZDR+K"2(````00'*3(G)2<'I`DC!X1Y%B=)/C80HF7F"6DP)R4V)
MU4@QR$G![1M!B<E,(>!(,=A,`<!-B=!)P>`%30GH3(NL))````!!`<!,B>!)
MP>P"2,'@'D6)P$J-G"N9>8):3`G@38G%2#'!2<'M&T&)Q$PAT4PQR4@!V4R)
MPTC!XP5,">M,BZPDF`````'+3(G12,'A'DG!Z@*)VT^-C"F9>8):3`G128G=
M2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-">E,BZPDH````$$!P4R)P$G!
MZ`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;08G`2"'93#'13`'A38G,
M2<'D!4T)[$:--"%(B=E(P>L"2,'A'D@)V4B+G"2H````2#'(08G,3"'(38V4
M&IEY@EI,B?-,,<!(P>L;3`'038GR2<'B!4D)VDB+G"2P````00'"3(G(2,'@
M'DG!Z0)%B=)-C808F7F"6DP)R$R)TT@QP4C!ZQM!B<%,(?%,,>%)`<A,B=%(
MP>$%2`G93(GS2<'N`DC!XQZ)R4P)\TD!R$B+C"2X````2#'818G%3"'028V,
M#)EY@EI-B>Q,,<A)P>P;2`'(3(GI2,'A!4P)X8G)2`'(3(G12<'J`DC!X1Y!
MB<1,"=%)B=J)VTDQRD4QT$R+E"3`````3XV,$:'KV6Y-B>)-`<%-B>!)P>H;
M2<'@!4T)T$6)P$T!P4V)Z$G!X!Y)P>T"18G*2`.<),@```!-">A)B<V)R4@#
MC"30````33'%1#'H38G52(V<`Z'KV6Y,B=!)P>T;2,'@!4P)Z(G`2`'#3(G@
M2<'L`DC!X!Y!B=U,">!-B<1%B<!,`X0DV````$DQQ$4QX4V)[$Z-C`FAZ]EN
M3(GI2<'L&TC!X05,">&)R4D!R4R)T4G!Z@)(P>$>18G,3`G128G"B<!),<I!
M,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!X!Y$
MB=--">A)B<U(`X0DX````(G)33'%2`.,).@```!%,>E)B=U.C8P(H>O9;DB)
MV$G![1M(P>`%3`GHB<!)`<%,B>!)P>P"2,'@'D6)S4P)X$V)Q$6)P$P#A"3P
M````23'$13'B38GL3HV4$:'KV6Y,B>E)P>P;2,'A!4P)X8G)20'*2(G92,'K
M`DC!X1Y%B=1("=E(B<.)P$@#A"3X````2#'+03'93(GC3XV,"*'KV6Y-B>!(
MP>L;2<'@!4D)V$6)P$T!P4V)Z$G![0))P>`>1(G+30GH28G-B<E-,<5(`XPD
M``$``$4QZDF)W4Z-E!"AZ]EN2(G82<'M&TC!X`5,">B)P$D!PDR)X$G![`)(
MP>`>18G53`G@38G$18G`3`.$)`@!``!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G!
M[!M(P>$%3`GAB<E)`<E(B=E(P>L"2,'A'D6)S$@)V4B)PXG`2`.$)!`!``!(
M,<M!,=I,B>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!
MX!Y$B=--">A)B<V)R4TQQ44QZ4F)W4Z-C`BAZ]EN2(G82,'@!4G![1M(`XPD
M&`$``$P)Z(G`20'!3(G@2<'L`DC!X!Y%B<U,">!-B<1%B<!,`X0D(`$``$DQ
MQ$4QXDV)[$Z-E!&AZ]EN3(GI2<'L&TC!X05,">&)R4D!RDB)V4C!ZP)(P>$>
M18G42`G92(G#B<!(`X0D*`$``$@QRT$QV4R)XT^-C`BAZ]EN38G@2,'K&TG!
MX`5)"=A%B<!-`<%-B>A)P>T"2<'@'D2)RTT)Z$F)S8G)33'%13'J28G=3HV4
M$*'KV6Y(B=A)P>T;2,'@!4P)Z$@#C"0P`0``B<!)`<),B>!)P>P"2,'@'D6)
MU4P)X$V)Q$6)P$P#A"0X`0``23'$13'A38GL3HV,":'KV6Y,B>E)P>P;2,'A
M!4P)X8G)20')2(G92,'K`DC!X1Y%B<Q("=E(B<.)P$@#A"1``0``2#'+03':
M3(GC3XV4$*'KV6Y-B>!(P>L;2<'@!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G3
M30GH28G-B<E-,<5%,>E)B=U.C8P(H>O9;DB)V$G![1M(P>`%3`GHB<!)`<%,
MB>!(P>`>2<'L`D6)S4@#C"1(`0``3`G@38G$18G`3`.$)%`!``!),<1%,>).
MC:01H>O9;DR)Z4V)ZDG!ZAM(P>$%3`G1B<E)`<Q(B=E(P>L"2,'A'D6)XD@)
MV4B)PXG`2`.$)%@!``!(,<M!,=E,B=-/C8P(H>O9;DV)T$C!ZQM)P>`%20G8
M3(GK2<'M`DC!XQY%`<%)B<B)R4P)ZT6)R4DQV$F)WD&)W44QX$V)S$Z-A`"A
MZ]EN3(G(2<'L&TC!X`5,">!%C20`38G02<'J`KC<O!N/2<'@'DT)T$T)QD6)
MPDPATTTASDD)WDB+G"1@`0``2`'#2`'93(GC3`'Q38GF2,'C!4G![AM,"?,!
MRTR)R4G!Z0)(P>$>B=M,"<E)"<A)B<Y!B<E-(>!-(=9-"<9,BX0D:`$``$D!
MP$T!Z$F)W4T!\$F)WDG!Y05)P>X;30GU28G.10'%38G@2<'L`DG!X!Y%B>U-
M">!%B<1-"<9)(=Y,(>%,"?%,B[0D<`$``$D!QDT!UDV)ZDP!\4V)[DG!X@5)
MP>X;30GR00'*2(G92,'K`DC!X1Y%B=)("=E)"<A)B<Z)RTTAZ$TAYDT)QDR+
MA"1X`0``20'`30'(38G130'P38G62<'A!4G![AM-"?%)B<Y%`<%-B>A)P>T"
M2<'@'D6)R4T)Z$6)Q4T)QDTAUDPAZ4P)\4R+M"2``0``20'&30'F38G,3`'Q
M38G.2<'D!4G![AM-"?1!`<Q,B=%)P>H"2,'A'D6)Y$P)T4D)R$F)SD&)RDTA
MR$TA[DT)QDR+A"2(`0``20'`20'83(GC30'P38GF2,'C!4G![AM,"?-)B<Y$
M`<--B<A)P>D"2<'@'HG;30G(18G!30G&32'F3"')3`GQ3(NT))`!``!)`<9-
M`>Y)B=U,`?%)B=Y)P>4%2<'N&TT)]4$!S4R)X4C!X1Y)P>P"18GM3`GA20G(
M28G.08G,22'832'.30G&3(N$))@!``!)`<!-`=!-B>I-`?!-B>Y)P>(%2<'N
M&TT)\DF)SD4!PDF)V$C!ZP))P>`>18G220G81(G#30G&32'N2"'93`GQ3(NT
M)*`!``!)`<9-`<Y-B=%,`?%-B=9)P>$%2<'N&TT)\4$!R4R)Z4G![0)(P>$>
M18G)3`GI20G(28G.08G-32'022'>30G&3(N$)*@!``!)`<!-`>!-B<Q-`?!-
MB<Y)P>0%2<'N&TT)]$F)SD4!Q$V)T$G!Z@))P>`>18GD30G018G"30G&32'.
M3"'13`GQ3(NT)+`!``!)`<9)`=Y,B>-,`?%-B>9(P>,%2<'N&TP)\P'+3(G)
M2<'I`DC!X1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+A"2X`0``20'`30'H
M28G=30'P28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"2<'@'D6)[4T)X$6)
MQ$T)QDDAWDPAX4P)\4R+M"3``0``20'&30'638GJ3`'Q38GN2<'B!4G![AM-
M"?)!`<I(B=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H32'F30G&3(N$),@!
M``!)`<!-`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!P4V)Z$G![0))P>`>
M18G)30GH18G%30G&32'63"'I3`GQ3(NT)-`!``!)`<9-`>9-B<Q,`?%-B<Y)
MP>0%2<'N&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(28G.08G*32'(32'N
M30G&3(N$)-@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N&TP)\TF)SD0!PTV)
MR$G!Z0))P>`>B=M-"<A-"<9%B<%-(>9,(<E,"?%,B[0DX`$``$D!QDT![DF)
MW4P!\4F)WDG!Y05)P>X;30GU00'-3(GA2<'L`DC!X1Y%B>U,">%)"<A)B<Q!
MB<])(=A-(<Q-"<1,BX0DZ`$``$D!P$T!T$V)ZDT!X$V)[$G!X@5)P>P;30GB
M10'"28G82,'K`DG!X!Y%B=))"=A(B<M%B<1,"<-,(>M,(>%("=E(BYPD\`$`
M`$@!PT@#A"3X`0``20'93(G33`'X3(N\)!@"``!)`<E,B=%(P>L;2,'A!4@)
MV8G)20')3(GI2<'M`DC!X1Y$B<M,">E)"<A)B<U!B<Y-(=!-(>5-"<5-C40%
M`$B)V$F)W4G![1M(P>`%3`GHB<!)`<!,B=!)P>H"2,'@'D6)Q4D)PKC6P6+*
M3#'11#')3(N,)``"``!)`<%-`>%-B>Q,`<E-B>E)P>P;2<'A!4T)X4F)W$6)
MR4P!R4G!Y!Y)`<=(P>L"08G)3`GC38G418G223'<13'$3(N$)`@"``!)`<!-
M`?!-B<Y-`>!-B<Q)P>X;2<'D!4T)]$V)[DG![0))P>8>18GD30GU3(NT)!`"
M``!-`>!%B<1)`<9-`?))B=Y-,>Y!,<Y,B>%-`?)-B>9(P>$%2<'N&TP)\8G)
M20'*3(G)2<'I`DC!X1Y%B=9)"<F)V4R)ZTPQRTP!^4R+O"0@`@``1#'#3(T$
M&4R)\4R)\TC!X05(P>L;20''2`G9B<E)`<A,B>%)P>P"2,'A'D2)PTP)X46)
M[$V)S46)R4DQS4T!_$R+O"0X`@``13'53XT4+$F)W$F)W4G![1M)P>0%30GL
M38GU2<'N`DG!Y1Y%B>1-">Y,BZPD*`(``$T!XD6)U$D!Q4T!Z4F)S8G)33'U
M13'%38G@30'I38GE2<'@!4G![1M-">A%B<!-`<%)B=A(P>L"2<'@'D6)S4D)
MV$B+G"0P`@``2`'#2`'93(GS3#'#1#'320''3(T4&4R)Z4R)ZTC!ZQM(P>$%
M2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QD6)P$TQYDP!^4R+O"1(
M`@``13'.3HT,,4B)V4F)WDD!QTG![AM(P>$%3`GQB<E)`<E,B>E)P>T"2,'A
M'D6)SDP)Z4R+K"1``@``20'%30'H38GE23'-13'53XT4*$V)\$V)]4G![1M)
MP>`%30GH18G`30'"28G82,'K`DG!X!Y%B=5,"<-%B>!)B<R)R4T!^$DQW$R+
MO"18`@``13',3XT,($V)Z$V)[$D!QTG![!M)P>`%30G@18G`30'!38GP2<'N
M`DG!X!Y%B<Q-"?!,B[0D4`(``$D!QDP!\4F)WDTQQD4QUDZ-%#%,B>%-B>9)
MP>X;2,'A!4P)\8G)20'*3(GI2<'M`DC!X1Y%B=9)"<V)V4R)PT6)P$PQZTP!
M^4R+O"1H`@``1#'+3(T,&4R)\4R)\TC!ZQM(P>$%2`G9B<E)`<E,B>%)P>P"
M2,'A'D2)RTP)X4R+I"1@`@``20'$30'@38GL20''23',13'43XT4($F)V$F)
MW$G![!M)P>`%30G@18G`30'"38GP2<'N`DG!X!Y%B=1-"<9%B>A)B<V)R4TQ
M]4T!^$R+O"1X`@``13'-3XT,*$V)X$V)Y4G![1M)P>`%30GH18G`30'!28G8
M2,'K`DG!X!Y%B<U)"=A(BYPD<`(``$@!PT@!V4R)\TPQPT0QTTR-%!E,B>E,
MB>M(P>L;2,'A!4@)V8G)20'*3(GA2<'L`DC!X1Y$B=-)"<Q$B?%-B<9)`<=-
M,>9,`?E%B<!,B[PDB`(``$4QSDF)V4D!SDB)V4G!Z1M)`<=(P>$%3`G)B<E)
M`<Y,B>E)P>T"2,'A'D6)\4P)Z4R+K"2``@``20'%30'H38GE23'-13'53XT4
M*$V)R$V)S4G![1M)P>`%30GH18G`30'"28G82,'K`DG!X!Y%B=5,"<-%B>!)
MB<R)R4DQW$T!^$4Q]$V)[DT!X$V)[$G![AM)P>0%30GT38G.18GD30'@2<'F
M'DG!Z0)%B<1-"?%,B[0DD`(``$D!QD@#A"28`@``3`'Q28G>3`'8B=M-,<Y(
M`=A%,=9-B>),`?%-B>9)P>(%2<'N&TT)\D0!T4V)ZDG![0))P>(>B<E-"=5-
MB<I$`TH@33'J00']3(E*($4QT$F)RDR):AA,`<!)B<A)P>H;`>E)P>`%2(E*
M"$T)T$0!P$B)`DR)X$G![`)(P>`>3`G@`?!(BS5(;P``2(E"$$B+A"2H`@``
M2"L&=11(@<2X`@``6UY?74%<05U!7D%?P^B.!0``D&9F+@\?A```````9I!(
M@^PHN8````#H(@8``+J)J\WON?[<NIA(B5`(2('JF<GZ*TC'``$C16=(B4@0
M2,=`&'94,A!(B5`@2,=`*`````!(QT`P`````,=`>`````!(@\0HPV8N#Q^$
M``````!!54%455=64TB#["A)8\!,BT$H2(G/2(G32(M7,$F)Q$B-=SA!C0S`
M3#G!2(E/*$AC3WA(@](`2,'H'4@!T$B)1S"%R71(O4````!(B=HIS40YY4$/
M3^Q(`?%,8^U-B>CHBP4``(M'>`'HB4=X@_A`=`Y(@\0H6UY?74%<05W#D$B)
M^4$I[$P!Z^B2Z/__08/\/P^.V````$&#[$!%B>5!P>T&08UM`4C!Y09(`=UF
MD/,/;P-(B?E(@\-`#Q$&\P]O2]`/$4X0\P]O4^`/$58@\P]O6_`/$5XPZ$+H
M__](.>MUS4'!Y09%*>Q)8\1(@_@(<EU(BU4`2(U."$B#X?A(B5<X2(M4!?A(
MB50&^$@ISD@!\$@I]4B#X/A(@_@(<AA(@^#X,=),BT05`$R)!!%(@\((2#G"
M<NY$B6=X2(/$*%M>7UU!7$%=PP\?@`````"H!'4D2(7`=-\/ME4`B%<XJ`)T
MU`^W5`7^9HE4!O[KR$B)W>ER____BU4`B5<XBU0%_(E4!OSKL`\?A```````
M05155U932(/L($B+0BB+<BB+>C!(P>@##\Z#X#\/STB)RXU(`4B828G4QD0"
M.(!(C6HX2&/!2`'H@_DX#XY.`0``0;A`````02G(=!`QTHG1@\(!Q@0(`$0Y
MPG+R3(GAZ"OG__])QT0D.`````!)QT0D0`````!(QT40`````$C'11@`````
M2,=%(`````!(QT4H`````$C'13``````08E\)'!,B>%!B70D=.C?YO__28L$
M)$R)X4C!Z!B(`TF+!"1(P>@0B$,!28L$)(AC`DF+!"2(0P-)BT0D"$C!Z!B(
M0P1)BT0D"$C!Z!"(0P5)BT0D"(AC!DF+1"0(B$,'28M$)!!(P>@8B$,(28M$
M)!!(P>@0B$,)28M$)!"(8PI)BT0D$(A#"TF+1"082,'H&(A##$F+1"082,'H
M$(A##4F+1"08B&,.28M$)!B(0P])BT0D($C!Z!B(0Q!)BT0D($C!Z!"(0Q%)
MBT0D((AC$DF+1"0@B$,32(/$(%M>7UU!7.E:`@``9BX/'X0``````+HX````
M13')*<J#^@AR28G12,<``````$C'1`CX`````$B-2`A(@^'X2"G(`<*#XOB#
M^@@/@M7^__^#XO@QP$&)P(/`"$Z)#`$YT'+RZ;W^__]F#Q]$``#VP@1U(X72
M#X2J_O__Q@``]L("#X2>_O__,<EFB4P0_NF2_O__#Q\`QP``````QT00_```
M``#I?/[__Y"0D)"0D)"0D)"0D)#_);+.``"0D/\EHLX``)"0_R62S@``D)#_
M)8+.``"0D/\E<LX``)"0_R5BS@``D)#_)5+.``"0D/\E.LX``)"0_R4JS@``
MD)#_)1K.``"0D/\E"LX``)"0_R7ZS0``D)#_)>K-``"0D/\EVLT``)"0_R7*
MS0``D)#_);K-``"0D/\EJLT``)"0_R6:S0``D)#_)8K-``"0D/\E>LT``)"0
M_R72RP``D)"0D)"0D)"0D/\ENLP``)"0D)"0D)"0D)#_)3K-``"0D`\?A```
M````45!(/0`0``!(C4PD&'(92('I`!```$B#"0!(+0`0``!(/0`0``!WYT@I
MP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E4LL``)"0D)"0D)"0D)#_)4K+``"0
MD)"0D)"0D)"0_R5*RP``D)"0D)"0D)"0D/\E0LL``)"0D)"0D)"0D)#_)3K+
M``"0D)"0D)"0D)"0_R4RRP``D)"0D)"0D)"0D/\E0LL``)"0D)"0D)"0D)#_
M)4++``"0D)"0D)"0D)"0_R5:RP``D)"0D)"0D)"0D/\E4LL``)"0D)"0D)"0
MD)#_)4K+``"0D)"0D)"0D)"0_R5"RP``D)"0D)"0D)"0D/\E.LL``)"0D)"0
MD)"0D)#_)3++``"0D)"0D)"0D)"0_R4JRP``D)"0D)"0D)"0D/\E(LL``)"0
MD)"0D)"0D)#_)1K+``"0D)"0D)"0D)"0_R42RP``D)"0D)"0D)"0D/\E"LL`
M`)"0D)"0D)"0D)#_)0++``"0D)"0D)"0D)"0_R7ZR@``D)"0D)"0D)"0D/\E
M\LH``)"0D)"0D)"0D)#_)>K*``"0D)"0D)"0D)"0_R7RR@``D)"0D)"0D)"0
MD/\EZLH``)"0D)"0D)"0D)#_)>+*``"0D)"0D)"0D)"0_R7:R@``D)"0D)"0
MD)"0D/\ETLH``)"0D)"0D)"0D)#_)<K*``"0D)"0D)"0D)"0_R7"R@``D)"0
MD)"0D)"0D/\ENLH``)"0D)"0D)"0D)#_);+*``"0D)"0D)"0D)"0_R6JR@``
MD)"0D)"0D)"0D/\EHLH``)"0D)"0D)"0D)#_)9K*``"0D)"0D)"0D)"0_R62
MR@``D)"0D)"0D)"0D/\EBLH``)"0D)"0D)"0D)#_)8+*``"0D)"0D)"0D)"0
M_R5ZR@``D)"0D)"0D)"0D$B#["@QTNA5````_Q47R0``D)"0D)"0D)"0D)"0
MD)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0D)"0D)"0
MD)"0D)"0PY"0D)"0D)"0D)"0D)"0D%932(/L*$B)SDB)T[D(````Z+H$``!(
MA=L/A!$"``!(N@````!9`0``2(ES0$B--8S___](C0U%!P``2(F3\````$B-
M%4\'``!(B5-02(L5?,@``$B)LX````!(C35N____2(E+2$B-#7/___](B5,8
M2(T56*4``$B)LX@```!(C35J____2(F+D````$BY2`$``+P+``!(B;.8````
M2(TUB_W__TB)4SA(C14@_?__2(E+"$B-#17^__](B7,@2(TUZ@,``$B)4RA(
MC14/!```2,<#`````,=#$`8```!(B4LP2(ES>$B)DQ@!``!(A<`/A+T```!(
MBX#@````2(7`#X2M````BQ73I```A=(/A9\```!(@SW3%`````^$#0(``$B#
M/<T4````#X1O`@``2(,]QQ0````/A%$"``!(@SW!%`````^$,P(``$B#/;L4
M````#X05`@``2(,]M10````/A/<!``!(@SVO%`````^$V0$``$B#/:D4````
M#X2[`0``9@]O!6,4```/$0!F#V\-:!0```\12!!F#V\5;!0```\14"!F#V\=
M<!0```\16#`QR>AU`P``,<E(B8/H````2(T%Y1,``$B)0UA(C05:%```2(E#
M8$B-!<^C``!(B4-H2(T%I*4``$B)0W!(C06I:```2(F#(`$``$B-!==H``!(
MB8,H`0``2(T%G;/__TB)@S`!``#H@0(``+@!````2(/$*%M>PP\?1```2(EP
M0$B--87]__](N@````!9`0``2(T-3`4``$B)L(````!(C35V_?__2(F0\```
M`$B-%1@%``!(B;"(````2(TU:OW__TB)4$A(BQ5/Q@``2(E(4$B+B$`!``!(
MB;"0````2(TU5OW__TB)L)@```!(C34HHP``2(D*2(T5?OO__TB-#1?[__](
MB7`X2+Y(`0``O`L``$B)<`A(C37^^___2(E0($B-%=,!``!(B4@H2(T-^`$`
M`$C'``````#'0!`&````2(EP,$B)4'A(B8@8`0``2(7`#X2F_O__2(G#2(N`
MX````$B%P`^%YOW__^F._O__9BX/'X0``````$B+$$B)%;82``#IY/W__Y!(
MBU`X2(D5W1(``.DU_O__2(M0,$B)%<42``#I%_[__TB+4"A(B16M$@``Z?G]
M__](BU`@2(D5E1(``.G;_?__2(M0&$B)%7T2``#IO?W__TB+4!!(B15E$@``
MZ9_]__](BU`(2(D531(``.F!_?__3(L%.:(``(L5.Z(``$B+#3RB``#IQP``
M``\?@`````!(@^PX@_H"='=W5872#X1]````,<!-A<!(C17!____2(D-"J(`
M``^4P$R)!?"A``#'!>ZA```!````B070H0``Z-L```!(@_C_2(D%R*$```^5
MP`^VP$B#Q#C##Q]$``"X`0```(/Z`W7LN@,```!(@\0XZ4@````/'X0`````
M`+H"````2(/$..DR````9I`QTN@I````A<!TNTB+#7:A``")1"0LZ$4```"+
M1"0L2,<%7J$``/_____KFI"0D)"X`0```,.0D)"0D)"0D)"0PY"0D)"0D)"0
MD)"0D)"0D/\E*L0``)"0D)"0D)"0D)#_)2K$``"0D)"0D)"0D)"0_R4BQ```
MD)"0D)"0D)"0D/\EHL0``)"0D)"0D)"0D)!!54%42(/L*$R-+1&A``!)B<Q(
MB=%,B>KH\_K__TR)ZDR)X4B#Q"A!7$%=Z0````#_)=K#``"0D)"0D)"0D)"0
M_R6RQ0``D)`/'X0``````$%505164TB#[%B)C"2`````2(E4)#A,B40D,.CP
M]___2(U4)#A(BQVD80``3(U$)#!(C8PD@````$B+`TB)1"1(,<#HV/;__TB+
M%6%A``!(BPU*80``3(T%$_?__^@N]___2(L%)V$``$R+)5!A``"`.`!U-NB.
M]O__2(D%!Z```$B)P4B%P`^$A@$``.AN]O__3(LE)V$``$R)X>C_]O__2(L`
MQH`/"@```$R)X>CM]O__2(L`@(AB!````XN$)(````"-2`2-<`-(8\E(P>$#
MZ&+V__],BU0D.$2+A"2`````28G!2(D%DY\``$F+`DF)`4B-!3\B``!)B4$(
M2(L%BP\``$F)01!(C04L(@``28E!&$&#^`$/CNL```!)@\((08/``[H$````
M#Q]$``!)BP)(8\J#P@%)@\((28D$R40YPG7J2&/22(L-.)\``$&)\$G'!-$`
M````2(T5;L;__TC'1"0@`````.AX]?__08G$A<!T3DB+#0J?``!,C6PD1.AP
M]?__3(GJ2(T->1\``.A9O___2(7`=#I(BPWEG@``Z$CU___H:_7__TB+1"1(
M2"L#=69$B>!(@\186UY!7$%=PTB+#;R>``#H#_7__T&)Q.NA2(L-FYX``$B%
MR70'Z'G9___KLTB-#<D>``!,B>KH^+[__TB)!7F>``!(B<%(A<!TE>O9N@0`
M``#I,____[D!````Z%/V___H'O;__Y"0D)"0D)"0D)"0D)"0Z8N^__^0D)"0
MD)"0D)"0D/__________P%%```$``````````````/__________````````
M```````````````````````````````````*4$%2+G!M"@``````````X')`
M``$`````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````)7,O)7,`+W!R;V,O<V5L9B]E
M>&4`1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R``!);G1E<FYA;',Z.E!!
M4CHZ0D]/5``N`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"5S)7,E<P`O`%!%
M4DPU3$E"`%!%4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%
M1`!005)?4U!!5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!0
M05)?1TQ/0D%,7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`
M4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`,0!0051(`%5315(`
M4UE35$5-`"]T;7``5$U01$E2`%1%35!$25(`5$5-4`!435``55-%4DY!344`
M)3`R>``E<R5S)7,E<P!P87(M`"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U
M8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I
M=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E
M(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*`'!A<FPN97AE`"YP87(`
M`$-!0TA%`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U)7,`)7,E<W1E;7`M
M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D524TE/3@`Q+C`U-P``
M``````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``````````
M)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S
M(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS:2YC`````````&5V
M86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P
M.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A
M<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C
M:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M
M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A
M8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P
M-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@
M+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H
M=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@
M)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O
M<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'
M24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU
M;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N
M86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[
M4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;
M4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ
M9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P
M87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F
M)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M
M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N
M86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)
M3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R
M:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@
M9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@
M73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N
M90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@
M)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^
M)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U
M;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]
M"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE
M9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R
M7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T
M:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT[
M"G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S
M*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O
M;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@
M34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I
M.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA
M9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI
M.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I
M=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?
M1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#
M04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@
M;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M
M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O
M9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO
M=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD[
M"G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D
M;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@[
M"FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H
M:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@
M/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN
M72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO
M?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B
M=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E
M9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I
M*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L
M(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D
M9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]
M("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB
M+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U
M=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD36]D=6QE0V%C
M:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN
M86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]
M*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@
M>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO
M9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO
M9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L
M:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O
M)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/
M.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL
M92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB
M:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K
M*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?
M<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN
M9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L
M("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@
M)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z
M(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I
M<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P
M.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@
M*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI
M('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"
M4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U
M:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D
M9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@
M)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FEF("@D8G5F(&YE("102UI)
M4%]-04=)0RD@>PIO=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E<B!&24Q%('-E
M8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#(#L*?0HD
M<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@
M<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L
M92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?
M4D554T5].R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET>2!P
M<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A
M<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@
M("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P
M87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A
M;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6
M>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R
M;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%2
M1U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@
M;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D
M,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS
M:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS
M:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B
M)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*
M)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T
M(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB
M+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T[
M"GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H
M:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E
M;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!
M4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8[
M"B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*
M?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*
M>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM
M>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR
M87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R
M(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV
M93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@
M)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R
M<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E;65T85]P
M87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E
M;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD
M;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N
M*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L
M("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@
M(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@
M<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D
M4$M:25!?34%'24,["F-L;W-E("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*
M;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%
M050H*2!\($E/.CI&:6QE.CI/7U)$5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#
M*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T
M(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L
M;V%D97(["FEF("AD969I;F5D("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*
M<F5A9"!?1D@L("1L;V%D97(L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@
M96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II
M9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A
M?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!
M4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0HD
M9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO
M861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!0
M05(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN
M:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![
M("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D
M7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#
M;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L
M('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK
M(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E
M24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@
M("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X[
M"F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I
M;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ
M.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@
M)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D
M:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@
M/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O
M6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q
M?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@
M*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L
M("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE
M.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I
M;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R
M:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F
M:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D
M9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO
M8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O
M<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@
M/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L
M>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%2
M7U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI0
M05(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N
M="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G
M3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R
M8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP
M86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@
M<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=
M.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I
M('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F
M("(D;W5T(ET["GT*:68@*"1M971A7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN
M="@B7#!#3$5!3B(I.PI]"F5L<V4@>PII9B`H(61E9FEN960@)&-A8VAE7VYA
M;64I('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H
M+"`P+"`P.PHD8W1X+3YA9&1F:6QE*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD
M8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]
M('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A
M8VAE7VYA;64@+CT@(EPP(B!X("@D8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H
M("1C86-H95]N86UE*3L*)&9H+3YP<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#
M2$4B*3L*?0IM>2`D;V9F<V5T(#T@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D
M97(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)
M0RD["B1F:"T^8VQO<V4@;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE
M<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I[
M"FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M
M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E
M<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI:
M:7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/
M.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE
M('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<[
M"B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I
M;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E
M=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D
M>FEP.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B
M17AT<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-
M4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@
M;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E
M<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I
M;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H
M;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R
M86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A
M=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@
M)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I
M>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E
M(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@
M>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T
M7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E
M<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;
M17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET[
M"F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*
M?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I
M92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@
M72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A
M;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.
M5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'
M5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT
M*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I
M*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A
M=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O
M<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER
M92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T
M.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R
M<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@
M1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)
M3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S
M.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T
M<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E
M(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L
M('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E
M<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I
M<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R
M97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P
M871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%
M35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW
M("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I
M9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN
M960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N
M86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).
M04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER
M(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R
M;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII
M9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"([
M"GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D
M;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K
M("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H
M*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG
M=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H
M(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D
M8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H
M95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D
M8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%2
M7T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H
M95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@
M/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!
M4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?
M<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*
M;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@
M)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM
M>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I
M;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T
M96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D
M9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L
M;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK
M*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@
M*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%2
M7U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L
M(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H
M(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N
M86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D
M<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N
M.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P
M?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]4
M14U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1
M)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE
M(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O
M9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345]
M.PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD
M<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@
M>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R
M;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@
M*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@
M*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)
M4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6
M>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?
M)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%2
M7T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH`````````````````\$X`````````````
M`````(`400`!```````````````(1T```0```````````````$=```$`````
M`````````/`400`!```````````````0$T$``0``````````````H$9```$`
M`````````````%@300`!``````````````!'0T,Z("A'3E4I(#$Q+C,N,"`R
M,#(R,#0R,2`H1F5D;W)A($-Y9W=I;B`Q,2XS+C`M,2D`````````````````
M1T-#.B`H1TY5*2`Q,2XS+C$@,C`R,S`S,3````````!'0T,Z("A'3E4I(#$Q
M+C,N,2`R,#(S,#,Q,````````$=#0SH@*$=.52D@,3$N,RXQ(#(P,C,P,S$P
M````````1T-#.B`H1TY5*2`Q,2XS+C`@,C`R,C`T,C$@*$9E9&]R82!#>6=W
M:6X@,3$N,RXP+3$I`````````````````$=#0SH@*$=.52D@,3$N,RXP(#(P
M,C(P-#(Q("A&961O<F$@0WEG=VEN(#$Q+C,N,"TQ*0````````````````!'
M0T,Z("A'3E4I(#$Q+C,N,"`R,#(R,#0R,2`H1F5D;W)A($-Y9W=I;B`Q,2XS
M+C`M,2D`````````````````1T-#.B`H1TY5*2`Q,2XS+C`@,C`R,C`T,C$@
M*$9E9&]R82!#>6=W:6X@,3$N,RXP+3$I`````````````````$=#0SH@*$=.
M52D@,3$N,RXP(#(P,C(P-#(Q("A&961O<F$@0WEG=VEN(#$Q+C,N,"TQ*0``
M``````````````!'0T,Z("A'3E4I(#$Q+C,N,"`R,#(R,#0R,2`H1F5D;W)A
M($-Y9W=I;B`Q,2XS+C`M,2D`````````````````1T-#.B`H1TY5*2`Q,2XS
M+C`@,C`R,C`T,C$@*$9E9&]R82!#>6=W:6X@,3$N,RXP+3$I````````````
M`````$=#0SH@*$=.52D@,3$N,RXP(#(P,C(P-#(Q("A&961O<F$@0WEG=VEN
M(#$Q+C,N,"TQ*0````````````````!'0T,Z("A'3E4I(#$Q+C,N,"`R,#(R
M,#0R,2`H1F5D;W)A($-Y9W=I;B`Q,2XS+C`M,2D`````````````````1T-#
M.B`H1TY5*2`Q,2XS+C`@,C`R,C`T,C$@*$9E9&]R82!#>6=W:6X@,3$N,RXP
M+3$I`````````````````$=#0SH@*$=.52D@,3$N,RXQ(#(P,C,P,S$P````
M````1T-#.B`H1TY5*2`Q,2XS+C$@,C`R,S`S,3````````````````````$`
M``"`%`$`(+$``$````#P%`$`4+$``$`````0$P$`8+$``$````!8$P$`@+$`
M`$``````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13V)3O2'Z@J*+>K_./BN&;QP$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``$\4``!(
MX```4!0``",5``!8X```,!4``%H7``!LX```8!<``,(7``"$X```T!<```$8
M``",X```$!@``!,8``"4X```(!@``",:``"8X```,!H``%P:``"PX```8!H`
M`'(;``"XX```@!L``'(@``#(X```@"```.(@``#@X```\"````TA``#HX```
M$"$```8B``#PX```$"(``,$H````X0``T"@``.,J```8X0``\"H``,(K```H
MX0``T"L``$-"``!$X0``4$(``*9"``!<X0``L$(``#A$``!DX0``0$0``&-&
M``!XX0``\$D```)*``"(X0``$$H``!%*``"0X0``($H``"%*``"4X0``,$H`
M`#%*``"8X0``0$H``$%*``"<X0``4$H```!.``"@X0```$X``!E.``"LX0``
M($X``-Q.``"PX0``X$X``.9.``"XX0``\$X``/%.``"\X0``0$\``'!/``#`
MX0``D$\``+-1```TX0``P%$``,51``#,X0``````````````````````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!$08`$0$'(`DP"&`'<`%0`1,'`!,!!@(+,`I@"7`(4`+````!
M$`D`$&(,,`M@"G`)4`C`!M`$X`+P```!!0(`!5(!,`$$`0`$0@```0````$9
M"@`9`1L"$3`08`]P#E`-P`O0">`"\`$&`@`&,@+``1,&`!,!!P(+,`I@"<`"
MT`$0"0`08@PP"V`*<`E0",`&T`3@`O````$%`@`%4@$P`00!``1"```!#`8`
M#'((,`=@!L`$T`+@`1,*`!,!,P`,,`M@"G`)4`C`!M`$X`+P`0D%``EB!3`$
M8`-P`L````$)!``)4@4P!,`"T`$*!0`*H@8P!6`$P`+0```!$PH`$P%7``PP
M"V`*<`E0",`&T`3@`O`!!`$`!$(```$,!P`,0@@P!V`&<`50!,`"T````0H&
M``HR!C`%8`1P`U`"P`$$`0`$0@```0````$````!`````0````$&`P`&0@(P
M`6````$````!!`$`!&(```$````!`````0@#``A"!,`"T````0``````````
M````````````````````````````````````````````````````````````
M``!8\$]D`````-(``0`!````$0```!$````H``$`;``!`+```0#0*```\"``
M`)!/```@8```,!H``/`J```0&```8!H``(`@```@&```@!L``!`B```0(0``
M0$0``%!"``"P0@``T!<``-H``0#P``$`"P$!`!`!`0`:`0$`)P$!`#,!`0!$
M`0$`4`$!`%X!`0!K`0$`>`$!`(4!`0"7`0$`H0$!`*H!`0"U`0$````!``(`
M`P`$``4`!@`'``@`"0`*``L`#``-``X`#P`0`'!A<BYE>&4`6%-?26YT97)N
M86QS7U!!4E]"3T]4`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`&UA:6X`
M;7E?<&%R7W!L`'!A<E]B87-E;F%M90!P87)?8VQE86YU<`!P87)?8W5R<F5N
M=%]E>&5C`'!A<E]D:7)N86UE`'!A<E]E;G9?8VQE86X`<&%R7V9I;F1P<F]G
M`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`<&%R7W-E='5P7VQI8G!A=&@`
M<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A=&4`>'-?:6YI=```````````
M````````````````````````````````````````````````````````````
M````````````````````:!`!````````````5!H!`-@2`0``$@$`````````
M``!D&@$`<!0!`!`2`0```````````-`:`0"`%`$`R!(!````````````Y!H!
M`#@5`0````````````````````````````````!(%0$``````%@5`0``````
M9!4!``````!T%0$``````(`5`0``````D!4!``````"@%0$``````+05`0``
M````R!4!``````#4%0$``````.05`0``````\!4!``````#\%0$```````@6
M`0``````'!8!```````P%@$``````$`6`0``````3!8!``````!4%@$`````
M`&`6`0``````:!8!``````!T%@$``````(`6`0``````C!8!``````"4%@$`
M`````)P6`0``````J!8!``````"T%@$``````,`6`0``````S!8!``````#4
M%@$``````-P6`0``````Z!8!``````#\%@$``````!`7`0``````&!<!````
M```D%P$``````#`7`0``````/!<!``````!$%P$``````%`7`0``````6!<!
M``````!D%P$``````'`7`0``````?!<!``````"(%P$``````)07`0``````
MH!<!``````"L%P$``````+@7`0````````````````#$%P$`````````````
M````W!<!``````#L%P$````````8`0``````&!@!```````P&`$``````$`8
M`0``````5!@!``````!D&`$``````'08`0``````B!@!``````"@&`$`````
M`+`8`0``````Q!@!``````#8&`$``````.@8`0``````"!D!```````<&0$`
M`````"P9`0``````0!D!``````!0&0$``````%P9`0``````;!D!````````
M`````````'@9`0````````````````!(%0$``````%@5`0``````9!4!````
M``!T%0$``````(`5`0``````D!4!``````"@%0$``````+05`0``````R!4!
M``````#4%0$``````.05`0``````\!4!``````#\%0$```````@6`0``````
M'!8!```````P%@$``````$`6`0``````3!8!``````!4%@$``````&`6`0``
M````:!8!``````!T%@$``````(`6`0``````C!8!``````"4%@$``````)P6
M`0``````J!8!``````"T%@$``````,`6`0``````S!8!``````#4%@$`````
M`-P6`0``````Z!8!``````#\%@$``````!`7`0``````&!<!```````D%P$`
M`````#`7`0``````/!<!``````!$%P$``````%`7`0``````6!<!``````!D
M%P$``````'`7`0``````?!<!``````"(%P$``````)07`0``````H!<!````
M``"L%P$``````+@7`0````````````````#$%P$`````````````````W!<!
M``````#L%P$````````8`0``````&!@!```````P&`$``````$`8`0``````
M5!@!``````!D&`$``````'08`0``````B!@!``````"@&`$``````+`8`0``
M````Q!@!``````#8&`$``````.@8`0``````"!D!```````<&0$``````"P9
M`0``````0!D!``````!0&0$``````%P9`0``````;!D!````````````````
M`'@9`0`````````````````/`%]?8WAA7V%T97AI=```%@!?7V5R<FYO````
M(P!?7V=E=')E96YT`````"X`7U]M86EN`````#``7U]M96UC<'E?8VAK``!,
M`%]?<W!R:6YT9E]C:&L`3P!?7W-T86-K7V-H:U]F86EL``!1`%]?<W1A8VM?
M8VAK7V=U87)D`&P`7V1L;%]C<G0P`',`7VEM<'5R95]P='(```"#`&%C8V5S
M<P`````"`6-A;&QO8P````!#`6-L;W-E9&ER``![`6-Y9W=I;E]D971A8VA?
M9&QL`'T!8WEG=VEN7VEN=&5R;F%L````F0%D;&Q?9&QL8W)T,````+8!96YV
M:7)O;@```-,!97AI=```/0)F<')I;G1F````20)F<F5E``"O`F=E='!I9```
M``"Z`F=E='!W=6ED``#(`F=E='5I9`````",`VQS965K`(X#;'-T870`DP-M
M86QL;V,`````H@-M96UC:'(`````I`-M96UC<'D`````I@-M96UM;W9E````
MJ@-M:V1I<@#L`V]P96X``/`#;W!E;F1I<@````,$<&]S:7A?;65M86QI9VX`
M````*`1P=&AR96%D7V%T9F]R:P````"V!')E860``+<$<F5A9&1I<@```+D$
M<F5A9&QI;FL``+P$<F5A;&QO8P```.H$<FUD:7(`?P5S<')I;G1F````AP5S
M=&%T``"2!7-T<F-H<@````"4!7-T<F-M<`````"9!7-T<F1U<`````"C!7-T
M<FQE;@````"H!7-T<FYC;7````"P!7-T<G)C:'(```"T!7-T<G-T<@````"Z
M!7-T<G1O:P`````L!G5N;&EN:P`````M`%]?96UU=&QS7V=E=%]A9&1R97-S
M```;`%!,7V1O7W5N9'5M<```;P%097)L7V%T9F]R:U]L;V-K``!P`5!E<FQ?
M871F;W)K7W5N;&]C:P````#L`5!E<FQ?8W)O86M?;F]C;VYT97AT``"``E!E
M<FQ?9V5T7W-V````KP)097)L7V=V7V9E=&-H<'8```!V`U!E<FQ?;6=?<V5T
M````[@-097)L7VYE=UA3`````/`#4&5R;%]N97=84U]F;&%G<P``Q@5097)L
M7W-A9F5S>7-M86QL;V,`````IP9097)L7W-V7W-E='!V`+`&4&5R;%]S=E]S
M971P=FX`````X@9097)L7W-Y<U]I;FET,P````#C!E!E<FQ?<WES7W1E<FT`
M:`=?7V5M=71L<U]V+E!,7V-U<G)E;G1?8V]N=&5X=`!K!V)O;W1?1'EN84QO
M861E<@```'P'<&5R;%]A;&QO8P````!^!W!E<FQ?8V]N<W1R=6-T`````'\'
M<&5R;%]D97-T<G5C=`"`!W!E<FQ?9G)E90"!!W!E<FQ?<&%R<V4`````@@=P
M97)L7W)U;@``B0)'971-;V1U;&5(86YD;&5!````$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!`&-Y9W=I;C$N9&QL`!00`0!C>6=G8V-?<RUS96@M,2YD;&P``"@0`0`H
M$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0
M`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0!C>6=P97)L
M-5\S-BYD;&P`/!`!`$M%4DY%3#,R+F1L;```````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T"$!`&@%`````````````#@G`0"H"`````````````#@+P$`:`4`````
M````````2#4!`*@(`````````````/`]`0`^```````````````P/@$`6`(`
M````````````B$`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````-BA````8```#````""@````L```&````""A,*%`H5"A8*%PH8"A````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
