package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`/6HNV0``````````/``+@(+`@(H`$0```#,`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```$&X!``,``(``
M`"`````````0```````````0````````$``````````````0```````!`,\!
M````$`$`%`L````@`0!0*P```-```.`!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MX!(!`'@"```````````````````````````````````N=&5X=````)A"````
M$````$0````$``````````````````!@``!@+F1A=&$```"``````&`````"
M````2```````````````````0```P"YR9&%T80``/$,```!P````1````$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#@`0```-`````"````D```````````````
M````0```0"YX9&%T80``W`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``#/`0`````!```"````E```````````````````0```0"YI9&%T80``
M%`L````0`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"B``````````````````!```#`+G)E;&]C```P`````%`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0TU0```
MZ)`Z``!%,<`QTC')Z*0Z``!%,<`QTC')Z*@Z``!%,<`QTC')Z*PZ``!%,<`Q
MTC')2(/$*.FL.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.F[-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%3J```$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'H8CD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQU(GP``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.BS-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GHMS<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ!8X``!(B?%)B<;H"S@``$&-
M3`8"2&/)Z.XV``!)B?%-B>!(B>I)B<9(B<'HFC<``$R)\4B)^NB?-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z!(X``!,B?'H2C8``$R)Z>@B-P``2(7`
M#X5K____9@\?A```````3(GIZ/@U``!,B>'H,#<``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z)PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=WYT``$B+`TB)A"2H````,<!(C50D($F)S.@%
M-@``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'H2C4`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^@1-0``D$%4
MN#`0``!55U93Z'`T``!(*<1(BS56G0``0;C_#P``2(L&2(F$)"@0```QP$B-
M7"0@28G,2(G52(T-9UP``$B)VN@)-@``A<!^18U0_TAC^H!\/"``=7"%TG0T
M28GXNB\```!(B=GH9#4``$B%P'0?2(L-Z)P``.CS,P``28GY28G83(GB2(L(
MZ&HS``#K&DB+#<F<``#HU#,``$F)Z$R)XDB+".A6,P``2(N$)"@0``!(*P9U
M%DB!Q#`0``!;7E]=05S##Q\`2&/XZX_H/C0``)!F9BX/'X0``````&:005=!
M5D%505155U932(/L.$B+'7F<``!(BP-(B40D*#'`,<"`.CU(B=8/E,!(B<](
M`<9(B?'HMC4``$B-5"0D2(GY2(G%Z/;[__])B<1(A<`/A/H```!(B<'HDC4`
M`$@YQ0^&R0```$QC?"0D3(LM/9P``$G!YP,/M@>$P`^$C@$``#P]#X2&`0``
M2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+90!(C4PH`N@L-```30'\
M28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<ZQYF#Q^$```````/M@%(
M@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V#`:(3`(!2(/``83)=?!(
MBT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#D#'`9@\?1```#[84!D&(
M%`1(@\`!A-)U\.O&9@\?1```3(LM69L``$V+=0!)@SX`#X2R````3(GP13'D
M#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!X@.+!2_:``"%P'53QP4A
MV@```0```$B)T>@Y,P``2(G!2(7`#X1=____36/\3(GR2<'G`TV)^.A+,P``
M28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^__\/'P!,B?'HD#,``$F)
M10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````,<#IC_[__[H0````13'D
MZ6?____H!S(``)!F#Q]$``!32(/L,$B+#42:``#H3S$``$R-#7!<``!,C06Q
M,```2(L(2(T5?%D``$B)P^C7,```3(T-B5D``,=$)"@`````3(T%N1(``$R)
M3"0@2(L+2(T5;%D``.BE,```D$B#Q#!;PV9F+@\?A```````#Q\`2(/L*$B+
M#=69``#HX#```$R-#0%<``!,C05",```2(L(2(T5#5D``$B#Q"CI9S```&9F
M+@\?A```````#Q]``#'`PV9F+@\?A```````9I!!5[C8$```059!54%455=6
M4^B:,```2"G$3(LM@)D``$F+10!(B80DR!```#'`2(UL)$!(B<Y)B=1(C0W9
M6```2(GJZ`CY__^Z+P```$B)\4B)P^AX,@``2(7`=#M(B?)(C0V[6```Z*3\
M__](B?!(BY0DR!```$DK50`/A7P!``!(@<38$```6UY?74%<05U!7D%?PP\?
M`$R)X4B-/8U8``!,C26/6```Z$(R``!(B?I(B<'HAS(``$F)QTB%P'2<08`_
M``^$UP````\?1```2(7;=!-,B?I(B=GH`#(``(7`#X2:````3(GYZ!`R``!)
MC50'_TF)QDDYUW(/ZQV0Q@(`2(/J`4DYUW0%@#HO=.],B?GHYS$``$F)QDB)
M\>C<,0``28U$!@%(/?\/```/ARC___](B70D,#'23(VT),````!,C0WF5P``
M3(ED)"A,B?%!N``0``!,B7PD(.C-+P``2(GJ3(GQZ%(Q``"%P'40BT0D4"4`
M\```/0"```!T3DB)^C')Z+0Q``!)B<=(A<`/A,7^__]!@#\`#X4N____2(7;
M=`6`.RYT%$&^`0```$R-/5Y7``#I7O___V:0@'L!`'2\Z^0/'X0``````+H!
M````3(GQZ',O``"%P'6A3(GR2(T--U<``.@@^___3(GQZ/@P``#I<O[__^@^
M+P``D&9F+@\?A```````9I!!5$B#[""Z+P```$F)S.C],```2(U0`4B%P$P/
M1>),B>%(@\0@05SIM#````\?0`!!5;@X$```05164^A@+@``2"G$2(L=1I<`
M`$B+`TB)A"0H$```,<!)B<U(A<D/A)T```"`.0`/A)0```#H?S```$R-0`%(
MB<9)@?@`$```#X>S````3(UD)"!!N0`0``!,B>I,B>'H=2X``$F-1#3_3#G@
M=Q?K&F8/'X0``````,8``$B#Z`%,.>!T!8`X+W3ONB\```!,B>'H0C```$B%
MP'0M3#G@=UA,B>'H`#```$B+E"0H$```2"L3=4](@<0X$```6UY!7$%=PP\?
M1```2(N$)"@0``!(*P-U+TB-#?-5``!(@<0X$```6UY!7$%=Z;HO``!F+@\?
MA```````Q@``ZZ,/'P`QP.NDZ.\M``"09F8N#Q^$```````/'P!!5T%6055!
M5%575E-(@^PX2(LM*98``$B+-4*6``!(C1VU50``2(M%`$B)1"0H,<!,C60D
M)$R)XDB)V>BP]?__2(7`=#M(8U0D)$B+!DB-!-!(BU`(2(D02(72=-@/'P!(
MBU`02(/`"$B)$$B%TG7P3(GB2(G9Z'7U__](A<!UQ4B+-=F5``!(C1U550``
M3(GB2(G9Z%?U__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z!7U__](A<!UODB+-7F5``!(
MC1W]5```3(GB2(G9Z/?T__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z+7T__](A<!UODB+
M-1F5``!(C1VF5```3(GB2(G9Z)?T__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(
MB1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z%7T__](
MA<!UODB+';F4``!,C2U-5```3(GB3(GIZ#?T__](A<!T0DAC5"0D2(L#2(T$
MT$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GI
MZ/7S__](A<!UODB+-5F4``!(C1W]4P``3(GB2(G9Z-?S__](A<!T0DAC5"0D
M2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P
M3(GB2(G9Z)7S__](A<!UODB+'?F3``!,C35+4P``3(GB3(GQZ'?S__](A<!T
M0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)
M$$B%TG7P3(GB3(GQZ#7S__](A<!UODB+-9F3``!(C1U)4P``3(GB2(G9Z!?S
M__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`0
M2(/`"$B)$$B%TG7P3(GB2(G9Z-7R__](A<!UODB+-3F3``!,C3WS4@``3(GB
M3(GYZ+?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$````
M``!(BU`02(/`"$B)$$B%TG7P3(GB3(GYZ'7R__](A<!UODB+/=F2``!(C36=
M4@``3(GB2(GQZ%?R__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(A=)TV&8N
M#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ!7R__](A<!UODB+/7F2
M``!(C37440``3(GB2(GQZ/?Q__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(
MA=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ+7Q__](A<!U
MODR)XDB-#>M1``#HH?'__TB)PDB%P'0(3(GYZ%'U__],B>)(C0W=40``Z(+Q
M__](B<)(A<!T#$B-#=M1``#H+O7__TR)XDB-#==1``#H7_'__TB)PDB%P'0R
M3(GQZ`_U__](BT0D*$@K10!U0$B-%=%1``!,B>E(@\0X6UY?74%<05U!7D%?
MZ>7T__],B>)(C0V>40``Z!;Q__](B<)(A<!TOTB)V>C&]/__Z[7H[R@``)!F
M9BX/'X0```````\?`%-(@^PP2(L=-)$``$B-#011``!(BP-(B40D*#'`2(U4
M)"3HR?#__TB%P'0L#[80@/HP#Y3`A-(/E,()T(/P`0^VP$B+5"0H2"L3=1!(
M@\0P6\-F#Q]$```QP.OFZ'\H``"09F8N#Q^$```````/'P!!5%-(@^PX2(L%
MPI```$R)1"1@2(U<)%A)B<Q,B4PD:$B)5"182(E<)"!(BQ!(B50D*#'2Z'<G
M``!)B=A,B>)(BT@8Z$@J``"Y_P```.A.*```D&9F+@\?A```````9I!5N#@`
M`0!75E/H<B<``$@IQ$B+-5B0```QTD&X`@```$B+!DB)A"0H``$`,<")STB-
M7"0@Z&<H``!(C6C_2(GJ2,'Z/TC!ZC!(C405``^WP$@IT$@IQ0^(F`````\?
M0`!%,<!(B>J)^>@S*```2(/X_P^$D@```$&X"``!`$B)VHGYZ*DH``"#^/]T
M<4B82(/X!W9,2(U$`_A(.=AR0DB-%0L_``#K$F8/'X0``````$B#Z`%(.=AR
M)TB+"D@Y"'7O2"G82`'H2(N4)"@``0!(*Q9U04B!Q#@``0!;7E]=PTB![0``
M`0!(@?T``/__#X5L____2,?`_____^O*2(T-OT\``.B7_O__2(T-ID\``.B+
M_O__Z/8F``"0#Q]$``!(@^PHZ&<G``")P>A0)P``2(7`=`-(BP!(@\0HPP\?
M`$%6055!5%932(/L0$B+-1V/``!,C2UU3P``2(L&2(E$)#@QP$F)S$B-5"0T
M3(GIZ*SN__](B<-(A<!T!8`X`'4O2(M$)#A(*P8/A:$```!,B>),B>E(@\1`
M6UY!7$%=05[I.O+__V8N#Q^$``````!,B>)(B<'H12@``$B%P'0@2(M$)#A(
M*P9U9DB#Q$!;7D%<05U!7L-F#Q^$``````!,B>'HZ"<``$B)V4F)QNC=)P``
M28U,!@+HPR8``$B)7"0@38G@3(T-\TT``$B-%>Y-``!(B<%)B<;H8B<``$B+
M1"0X2"L&=0A,B?+I8O___^C;)0``D&8N#Q^$``````!!5T%6055!5%575E-(
M@>R8`0``2(L]%HX``$R--8A-``!(BP=(B80DB`$``$B-!71-``!(B<Y,C6PD
M4$R)\4C'A"3@`````````$R)ZDC'A"0X`0```````$C'A"0``0```````$B)
MA"38````2(T%,4X``$B)A"08`0``2(T%*4X``$B)A"0@`0``2(T%(DX``$B)
MA"0H`0``2(T%&$X``$B)A"0P`0``2(T%#4X``$B)A"3X````Z"/M__])B<1(
MA<!T"8`X``^%@@0``.A])0``B<'H9B4``$B%P`^$[0,``$B+*$B%[0^$X0,`
M`$B)Z>B9)@``2(U,``'H?R4``$0/MD4`2(G#183`=#-)B=],C26N30``#Q^$
M``````!%#[;`3(GY3(GB2(/%`>@-)@``1`^V10!)@\<"183`==](C:PD$`$`
M`$B-#?A,``#K%V8/'T0``$B+30A(@\4(2(7)#X0'`0``3(GJZ&_L__])B<1(
MA<!TWX`X`'3:2(G!Z+KN__^%P'3.3(GAZ.XE``!(B<=(A<`/A-(```!FD$B)
M^>CH)0``2(G92(G%Z-TE``!$C:0%!`0``$UCY$R)X>BZ)```2(E<)"A(C17Y
M3```28GX28G'2(T%]4P``$R-#>)+``!(B40D($R)^>A-)0``NL`!``!,B?GH
MP"0``(/X_P^$?P(``$R)ZDR)^>A<)```B<.%P`^$P@```$B+'D4QY.CG(@``
M38GY2(T5[4P``$B+2!A)B=CHT2,``$B+/>J+``!(BX0DB`$``$@K!P^%9`0`
M`$R)X$B!Q)@!``!;7E]=05Q!74%>05_#9I`Q[4R-)2=,``#K'`\?1```2&/%
M3(NDQ-````!-A>1T0$&`/"0`=#E,B>&#Q0'HGNW__X7`=-I,B>'HTB0``$B)
MQTB%P`^%YO[__TACQ4R+I,30````387D=<9F#Q]$```Q_^G)_O__9@\?A```
M````BT0D8"4`\```/0!````/A2K___^+;"1HZ%,C```YQ0^%&?___XM$)&`E
M_P$``#W``0``#X4%____3(GAZ%\C``!,B>I(C0UD2P``28G$Z*WJ__](B<)(
MA<`/A%$"``!(BP[H2?'__TB)Q4B%P`^$/0(``$B)Z>@U)```28G`@_@'?CU(
MF$B-%0=,``!,B40D2$B-3`7XZ/4C``"%P'4A3(M$)$A!@_@(#X3R`@``08U`
M]TB8@'P%`"\/A.$"``"02(TU8$H``$R)ZDB)\>@NZO__2(7`#X3E`0``#[8`
MA,`/A-H!```\,`^$T@$``$B)\4B-%9Q*``!(C36R20``Z+SM__],C2W=20``
M2(TMODL``.@Y(@``2(ET)"A-B?A,B>&)1"0@3(T-NTD``$B-%8Y+``#H)R,`
M`.LU#Q]$``#H:R$``(,X$74XZ`$B``"#PP%(B70D,$V)Z8E<)"A-B?A(B>I,
MB>&)1"0@Z/`B``"ZP`$``$R)X>AC(@``@_C_=+Y,B?GHMB$``$R)\4R)XN@K
M[?__3(GAZ(/Z___IO?W__V8/'T0``.@#(0``@S@1#X1S_?__Z/4@``!(BQY%
M,>2+*.AH(```38GY28G82(T5*TH``$B+2!B);"0@Z$XA``#I>/W__V8/'X0`
M`````#';2(T-NTD``.L5#Q]$``!(8\-(BXS$\````$B%R70Y3(GJ@\,!Z.7H
M__](A<!TX(`X`'3;2(G!Z',B``!(B<5(A<`/A=;[__](8\-(BXS$\````$B%
MR77'N0T```!(C2UD20``Z$8A``!!N%,```!(B</IQOO__P\?A```````2(G!
MZ*CY__](BSW1B```2(N$)(@!``!(*P</A4L!``!,B>%(@<28`0``6UY?74%<
M05U!7D%?Z?4A```/'T0``$B++NF[_?__#Q^$``````"Z```!`$B)Z>@C(0``
M08G%A<`/A!;^__^)P>@!^/__2(G%2(7`#X@#_O__13'`2(U0]D2)Z>B&(```
M2(V4)$H!``!!N`8```!$B>GH`"$``$B#^`8/A<8```"!O"1*`0```$-!0P^%
MP_W__V:!O"1.`0``2$4/A;/]__]%,<!(C57.1(GIZ#8@``!(C9PD4`$``$&X
M*````$2)Z4B)VNBM(```2(/X*'5W2(T%04<``$B)7"0@38GX3(GA2(E$)"A,
MC0UA1P``2(T5)4D``,:$)'@!````Z,4@``#IXOW__TB+=@A(A?8/A!/]__](
MB?'H^R```$B#^`,/A@']__](C4P&_$B-%=%(``#HP"```(7`2`]$[NGE_/__
MZ!`?``!(C0VZ2```Z)3V__^0#Q\`05175E-(@^PX2(L-0(<``$B+/4F'``!(
MBP=(B40D*#'`3(UD)"3H-1X``$B-#>E&``!(B<-(BP`/MI"P````B)"Q````
M3(GBZ,/F__](A<!T"8`X,0^$50$``.C`\/__2(L+0;D#````0;@!````2(T5
M>T@``.B='0``2(G&2(7`="=(BT`02(L+3(T%8D@``$B+$.A7'0``2(M&$$B+
M$/9"#D`/A38!``!(BPM!N0,```!!N`$```!(C14X2```Z%,=``!(B<9(A<!T
M*DR)XDB-#0M&``#H-.;__TF)P$B+1A!-A<`/A-0```!(BQ!(BPOH^1P``$B+
M"T&X`0```$B-%?-'``#H%!T``$B+"TR-!?E'``!(B<+HTAP``$B+`TB-#;%%
M``!,B>+&@+$`````Z-GE__](B05:Q0``2(G!2(7`=#JZP`$``.B@'@``C5`!
M@_H!=@F#^!$/A8P```!(BT0D*$@K!P^%G@```$B#Q#A;7E]!7,,/'X``````
M2(L-&<4``.BD]___2(D%!<4``$B)P4B%P'6K2(T-=D<``.B!'```9@\?A```
M````@'@!``^$IO[__^F<_O__D$B+%=G$``!(BPA(BQ+H'NC__^D@____9@\?
MA```````2(L+Z"@<``#IO?[__T&)P4B+!:G$``!,BP6:Q```2(T-.T<``$B+
M$.@;'```Z.X<``"09F8N#Q^$``````!FD$%505132(/L,$B+'3"%``!(BP-(
MB40D*#'`28G-Z+[M__^Z+P```$B)P4F)Q.B.'@``2(U0`4B%P$P/1>),B>'H
M2QX``$B-5"0D2(T-QD0``$F)Q.B7Y/__2(7`="(/MA"$T@^4P(#Z,`^4P@C0
M=1!-A>UT"T&`?0``=1P/'T``2(M$)"A(*P-U0DB#Q#!;05Q!7<,/'T``0;@$
M````2(T5/T4``$R)X>@#'@``A<!USTB+1"0H2"L#=1%,B>E(@\0P6T%<05WI
M%.7__^@/'```D)"0D)"0D)"0D)"0D)"005=!5D%505155U932('LN`(``$B+
M/4:$``!(BP=(B80DJ`(``#'`2(G*3(U9.$B-3"0@2(V<)*````!)B<IFD$&+
M`TV+"TF#PA!)@\,(#\B)P$V)R$F)0O!,B<A)P>@@2,'H*$G!Z3@E`/\``$P)
MR$V)P4G!X`A)P>$808'@``#_`$6)R4P)R$P)P$F)0OA).=IUK$R-B0`"```/
M'T0``$B+06A(,T%`2(/!"$@S00A(,T'X3(T$`$C!Z!],"<"#X/](B4%X3#G)
M==5,BQI(BW(02(M:($B+>AA,B=A,B=E(BVH(3(M4)"A(P>D;2,'@!4@)R`-$
M)"!.C907F7F"6HV,&)EY@EI(B?!(,?A((>A(,?A$C0P!2(GI2(GH2,'H`DC!
MX1Y,B<M("<%(B?!(P>L;08G(3#'`3"'82#'P3`'038G*2<'B!4D)VD:-)!!,
MB=A-B=I)P>H"2,'@'DV)Y4P)T$R+5"0P2<'M&T@QP8G#3"')3HV4%IEY@EI,
M,<%,`=%-B>))P>(%30GJ3(ML)#A!`<I,B<E)P>D"2,'A'D6)TD^-A"B9>8):
M3`G)38G52#'(08G)3"'@2<'M&T@QV$P!P$V)T$G!X`5-">A,BVPD0$$!P$R)
MX$G![`)(P>`>18G`2HV<*YEY@EI,">!-B<5(,<%)P>T;08G$3"'13#')2`'9
M3(G#2,'C!4P)ZTR+;"1(`<M,B=%)P>H"2,'A'HG;3XV,*9EY@EI,"=%)B=U(
M,<A)P>T;08G*3"'`3#'@3`'(28G92<'A!4T)Z4R+;"1000'!3(G`2<'H`DC!
MX!Y%B<E/C:0LF7F"6DP)P$V)S4@QP4G![1M!B<!((=E,,=%,`>%-B<Q)P>0%
M30GL3(ML)%A!`<Q(B=E(P>L"2,'A'D6)Y$^-E"J9>8):2`G938GE2#'(2<'M
M&XG+3"'(3#'`3`'038GB2<'B!4T)ZDR+;"1@00'"3(G(2<'I`DC!X!Y%B=)/
MC80HF7F"6DP)R$V)U4@QP4G![1M!B<%,(>%(,=E,`<%-B=!)P>`%30GH3(ML
M)&A!`<A,B>%)P>P"2,'A'D6)P$J-G"N9>8):3`GA38G%2#'(2<'M&T&)S$PA
MT$PQR$@!V$R)PTC!XP5,">M,BVPD<`'#3(G02<'J`DC!X!Z)VT^-C"F9>8):
M3`G028G=2#'!2<'M&T&)PDPAP4PQX4P!R4F)V4G!X05-">E,BVPD>$$!R4R)
MP4G!Z`)(P>$>18G)3XVD+)EY@EI,"<%-B<U(,<A)P>T;08G(2"'83#'03`'@
M38G,2<'D!4T)[$R+K"2`````00'$2(G82,'@'DC!ZP)%B>1/C90JF7F"6D@)
MV$V)Y4@QP4G![1N)PTPAR4PQP4P!T4V)XDG!X@5-">I,BZPDB````$$!RDR)
MR4G!Z0)(P>$>18G23XV$*)EY@EI,"<E-B=5(,<A)P>T;08G)3"'@2#'83`'`
M38G02<'@!4T)Z$R+K"20````00'`3(G@2<'L`DC!X!Y%B<!*C9PKF7F"6DP)
MX$V)Q4@QP4G![1M!B<1,(=%,,<E(`=E,B<-(P>,%3`GK3(NL))@````!RTR)
MT4C!X1Y)P>H"B=M/C8PIF7F"6DP)T4F)W4@QR$G![1M!B<I,(<!,,>!,`<A)
MB=E)P>$%30GI3(NL)*````!!`<%,B<!)P>@"2,'@'D6)R4^-I"R9>8):3`G`
M38G-2#'!2<'M&T&)P$@AV4PQT4P!X4V)S$G!Y`5-">Q&C30A2(G92,'K`DC!
MX1Y("=E(BYPDJ````$@QR$&)S$PAR$V-E!J9>8):3(GS3#'`2,'K&TP!T$V)
M\DG!X@5)"=I(BYPDL````$$!PDR)R$C!X!Y)P>D"18G238V$&)EY@EI,"<A,
MB=-(,<%(P>L;08G!3"'Q3#'A20'(3(G12,'A!4@)V4R)\TG![@)(P>,>B<E,
M"?-)`<A(BXPDN````$@QV$6)Q4PAT$F-C`R9>8):38GL3#'(2<'L&T@!R$R)
MZ4C!X05,">&)R4@!R$R)T4G!Z@)(P>$>08G$3`G128G:B=M),<I%,=!,BY0D
MP````$^-C!&AZ]EN38GB30'!38G@2<'J&TG!X`5-"=!%B<!-`<%-B>A)P>`>
M2<'M`D6)RD@#G"3(````30GH28G-B<E(`XPDT````$TQQ40QZ$V)U4B-G`.A
MZ]EN3(G02<'M&TC!X`5,">B)P$@!PTR)X$G![`)(P>`>08G=3`G@38G$18G`
M3`.$)-@```!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<E,
MB=%)P>H"2,'A'D6)S$P)T4F)PHG`23'*03':3(GC3XV4$*'KV6Y-B>!(P>L;
M2<'@!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G330GH28G-2`.$).````")R4TQ
MQ4@#C"3H````13'I28G=3HV,"*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'!3(G@
M2<'L`DC!X!Y%B<U,">!-B<1%B<!,`X0D\````$DQQ$4QXDV)[$Z-E!&AZ]EN
M3(GI2<'L&TC!X05,">&)R4D!RDB)V4C!ZP)(P>$>18G42`G92(G#B<!(`X0D
M^````$@QRT$QV4R)XT^-C`BAZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<%-B>A)
MP>T"2<'@'D2)RTT)Z$F)S8G)33'%2`.,)``!``!%,>I)B=U.C900H>O9;DB)
MV$G![1M(P>`%3`GHB<!)`<),B>!)P>P"2,'@'D6)U4P)X$V)Q$6)P$P#A"0(
M`0``23'$13'A38GL3HV,":'KV6Y,B>E)P>P;2,'A!4P)X8G)20')2(G92,'K
M`DC!X1Y%B<Q("=E(B<.)P$@#A"00`0``2#'+03':3(GC3XV4$*'KV6Y-B>!(
MP>L;2<'@!4D)V$6)P$T!PDV)Z$G![0))P>`>1(G330GH28G-B<E-,<5%,>E)
MB=U.C8P(H>O9;DB)V$C!X`5)P>T;2`.,)!@!``!,">B)P$D!P4R)X$G![`)(
MP>`>18G-3`G@38G$18G`3`.$)"`!``!),<1%,>)-B>Q.C901H>O9;DR)Z4G!
M[!M(P>$%3`GAB<E)`<I(B=E(P>L"2,'A'D6)U$@)V4B)PXG`2`.$)"@!``!(
M,<M!,=E,B>-/C8P(H>O9;DV)X$C!ZQM)P>`%20G818G`30'!38GH2<'M`DG!
MX!Y$B<M-">A)B<V)R4TQQ44QZDF)W4Z-E!"AZ]EN2(G82<'M&TC!X`5,">A(
M`XPD,`$``(G`20'"3(G@2<'L`DC!X!Y%B=5,">!-B<1%B<!,`X0D.`$``$DQ
MQ$4QX4V)[$Z-C`FAZ]EN3(GI2<'L&TC!X05,">&)R4D!R4B)V4C!ZP)(P>$>
M18G,2`G92(G#B<!(`X0D0`$``$@QRT$QVDR)XT^-E!"AZ]EN38G@2,'K&TG!
MX`5)"=A%B<!-`<)-B>A)P>T"2<'@'D2)TTT)Z$F)S8G)33'%13'I28G=3HV,
M"*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'!3(G@2,'@'DG![`)%B<U(`XPD2`$`
M`$P)X$V)Q$6)P$P#A"10`0``23'$13'B3HVD$:'KV6Y,B>E-B>I)P>H;2,'A
M!4P)T8G)20',2(G92,'K`DC!X1Y%B>)("=E(B<.)P$@#A"18`0``2#'+03'9
M3(G33XV,"*'KV6Y-B=!(P>L;2<'@!4D)V$R)ZTG![0)(P>,>10'!28G(B<E,
M">M%B<E),=A)B=Y!B=U%,>!-B<Q.C80`H>O9;DR)R$G![!M(P>`%3`G@18TD
M`$V)T$G!Z@*XW+P;CTG!X!Y-"=!-"<9%B<),(=--(<Y)"=Y(BYPD8`$``$@!
MPT@!V4R)XTP!\4V)YDC!XP5)P>X;3`GS`<M,B<E)P>D"2,'A'HG;3`G)20G(
M28G.08G)32'@32'630G&3(N$)&@!``!)`<!-`>A)B=U-`?!)B=Y)P>4%2<'N
M&TT)]4F)SD4!Q4V)X$G![`))P>`>18GM30G@18G$30G&22'>3"'A3`GQ3(NT
M)'`!``!)`<9-`=9-B>I,`?%-B>Y)P>(%2<'N&TT)\D$!RDB)V4C!ZP)(P>$>
M18G22`G920G(28G.B<M-(>A-(>9-"<9,BX0D>`$``$D!P$T!R$V)T4T!\$V)
MUDG!X05)P>X;30GQ28G.10'!38GH2<'M`DG!X!Y%B<E-">A%B<5-"<9-(=9,
M(>E,"?%,B[0D@`$``$D!QDT!YDV)S$P!\4V)SDG!Y`5)P>X;30GT00',3(G1
M2<'J`DC!X1Y%B>1,"=%)"<A)B<Y!B<I-(<A-(>Y-"<9,BX0DB`$``$D!P$D!
MV$R)XTT!\$V)YDC!XP5)P>X;3`GS28G.1`'#38G(2<'I`DG!X!Z)VTT)R$6)
MP4T)QDTAYDPAR4P)\4R+M"20`0``20'&30'N28G=3`'Q28G>2<'E!4G![AM-
M"?5!`<U,B>%(P>$>2<'L`D6)[4P)X4D)R$F)SD&)S$DAV$TASDT)QDR+A"28
M`0``20'`30'038GJ30'P38GN2<'B!4G![AM-"?))B<Y%`<))B=A(P>L"2<'@
M'D6)TDD)V$2)PTT)QDTA[D@AV4P)\4R+M"2@`0``20'&30'.38G13`'Q38G6
M2<'A!4G![AM-"?%!`<E,B>E)P>T"2,'A'D6)R4P)Z4D)R$F)SD&)S4TAT$DA
MWDT)QDR+A"2H`0``20'`30'@38G,30'P38G.2<'D!4G![AM-"?1)B<Y%`<1-
MB=!)P>H"2<'@'D6)Y$T)T$6)PDT)QDTASDPAT4P)\4R+M"2P`0``20'&20'>
M3(GC3`'Q38GF2,'C!4G![AM,"?,!RTR)R4G!Z0)(P>$>B=M,"<E)"<A)B<Y!
MB<E-(>!-(=9-"<9,BX0DN`$``$D!P$T!Z$F)W4T!\$F)WDG!Y05)P>X;30GU
M28G.10'%38G@2<'L`DG!X!Y%B>U-">!%B<1-"<9)(=Y,(>%,"?%,B[0DP`$`
M`$D!QDT!UDV)ZDP!\4V)[DG!X@5)P>X;30GR00'*2(G92,'K`DC!X1Y%B=)(
M"=E)"<A)B<Z)RTTAZ$TAYDT)QDR+A"3(`0``20'`30'(38G130'P38G62<'A
M!4G![AM-"?%)B<Y%`<%-B>A)P>T"2<'@'D6)R4T)Z$6)Q4T)QDTAUDPAZ4P)
M\4R+M"30`0``20'&30'F38G,3`'Q38G.2<'D!4G![AM-"?1!`<Q,B=%)P>H"
M2,'A'D6)Y$P)T4D)R$F)SD&)RDTAR$TA[DT)QDR+A"38`0``20'`20'83(GC
M30'P38GF2,'C!4G![AM,"?-)B<Y$`<--B<A)P>D"2<'@'HG;30G(30G&18G!
M32'F3"')3`GQ3(NT).`!``!)`<9-`>Y)B=U,`?%)B=Y)P>4%2<'N&TT)]4$!
MS4R)X4G![`)(P>$>18GM3`GA20G(28G,08G/22'832',30G$3(N$).@!``!)
M`<!-`=!-B>I-`>!-B>Q)P>(%2<'L&TT)XD4!PDF)V$C!ZP))P>`>18G220G8
M2(G+18G$3`G#3"'K3"'A2`G92(N<)/`!``!(`<-(`X0D^`$``$D!V4R)TTP!
M^$R+O"08`@``20')3(G12,'K&TC!X05("=F)R4D!R4R)Z4G![0)(P>$>1(G+
M3`GI20G(28G-08G.32'032'E30G%38U$!0!(B=A)B=U)P>T;2,'@!4P)Z(G`
M20'`3(G02<'J`DC!X!Y%B<5)"<*XUL%BRDPQT40QR4R+C"0``@``20'!30'A
M38GL3`')38GI2<'L&TG!X05-">%)B=Q%B<E,`<E)P>0>20''2,'K`D&)R4P)
MXTV)U$6)TDDQW$4QQ$R+A"0(`@``20'`30'P38G.30'@38G,2<'N&TG!Y`5-
M"?1-B>Y)P>T"2<'F'D6)Y$T)]4R+M"00`@``30'@18G$20'&30'R28G>33'N
M03'.3(GA30'R38GF2,'A!4G![AM,"?&)R4D!RDR)R4G!Z0)(P>$>18G620G)
MB=E,B>M,,<M,`?E,B[PD(`(``$0QPTR-!!E,B?%,B?-(P>$%2,'K&TD!QT@)
MV8G)20'(3(GA2<'L`DC!X1Y$B<-,">%%B>Q-B<U%B<E),<U-`?Q,B[PD.`(`
M`$4QU4^-%"Q)B=Q)B=U)P>T;2<'D!4T)[$V)]4G![@))P>4>18GD30GN3(NL
M)"@"``!-`>)%B=1)`<5-`>E)B<V)R4TQ]44QQ4V)X$T!Z4V)Y4G!X`5)P>T;
M30GH18G`30'!28G82,'K`DG!X!Y%B<U)"=A(BYPD,`(``$@!PT@!V4R)\TPQ
MPT0QTTD!QTR-%!E,B>E,B>M(P>L;2,'A!4@)V8G)20'*3(GA2<'L`DC!X1Y$
MB=-)"<Q$B?%-B<9%B<!-,>9,`?E,B[PD2`(``$4QSDZ-##%(B=E)B=Y)`<=)
MP>X;2,'A!4P)\8G)20')3(GI2<'M`DC!X1Y%B<Y,">E,BZPD0`(``$D!Q4T!
MZ$V)Y4DQS44QU4^-%"A-B?!-B?5)P>T;2<'@!4T)Z$6)P$T!PDF)V$C!ZP))
MP>`>18G53`G#18G@28G,B<E-`?A),=Q,B[PD6`(``$4QS$^-#"!-B>A-B>Q)
M`<=)P>P;2<'@!4T)X$6)P$T!P4V)\$G![@))P>`>18G,30GP3(NT)%`"``!)
M`<9,`?%)B=Y-,<9%,=9.C10Q3(GA38GF2<'N&TC!X05,"?&)R4D!RDR)Z4G!
M[0)(P>$>18G620G-B=E,B<-%B<!,,>M,`?E,B[PD:`(``$0QRTR-#!E,B?%,
MB?-(P>L;2,'A!4@)V8G)20')3(GA2<'L`DC!X1Y$B<M,">%,BZ0D8`(``$D!
MQ$T!X$V)[$D!QTDQS$4QU$^-%"!)B=A)B=Q)P>P;2<'@!4T)X$6)P$T!PDV)
M\$G![@))P>`>18G430G&18GH28G-B<E-,?5-`?A,B[PD>`(``$4QS4^-#"A-
MB>!-B>5)P>T;2<'@!4T)Z$6)P$T!P4F)V$C!ZP))P>`>18G-20G82(N<)'`"
M``!(`<-(`=E,B?-,,<-$,=-,C1093(GI3(GK2,'K&TC!X05("=F)R4D!RDR)
MX4G![`)(P>$>1(G320G,1(GQ38G&20''33'F3`'Y18G`3(N\)(@"``!%,<Y)
MB=E)`<Y(B=E)P>D;20''2,'A!4P)R8G)20'.3(GI2<'M`DC!X1Y%B?%,">E,
MBZPD@`(``$D!Q4T!Z$V)Y4DQS44QU4^-%"A-B<A-B<U)P>T;2<'@!4T)Z$6)
MP$T!PDF)V$C!ZP))P>`>18G53`G#18G@28G,B<E),=Q-`?A%,?1-B>Y-`>!-
MB>Q)P>X;2<'D!4T)]$V)SD6)Y$T!X$G!YAY)P>D"18G$30GQ3(NT))`"``!)
M`<9(`X0DF`(``$P!\4F)WDP!V(G;33'.2`'813'638GB3`'Q38GF2<'B!4G!
M[AM-"?)$`=%-B>I)P>T"2<'B'HG)30G538G*1`-*($TQZD$!_4R)2B!%,=!)
MB<I,B6H83`'`28G(2<'J&P'I2<'@!4B)2@A-"=!$`<!(B0),B>!)P>P"2,'@
M'DP)X`'P2(LU&&X``$B)0A!(BX0DJ`(``$@K!G442('$N`(``%M>7UU!7$%=
M05Y!7\/HC@4``)!F9BX/'X0``````&:02(/L*+F`````Z"(&``"ZB:O-[[G^
MW+J82(E0"$B!ZIG)^BM(QP`!(T5G2(E($$C'0!AV5#(02(E0($C'0"@`````
M2,=`,`````#'0'@`````2(/$*,-F+@\?A```````055!5%575E-(@^PH26/`
M3(M!*$B)STB)TTB+5S!)B<1(C7<X08T,P$PYP4B)3RA(8T]X2(/2`$C!Z!U(
M`=!(B4<PA<ET2+U`````2(G:*<U$.>5!#T_L2`'Q3&/M38GHZ(L%``"+1W@!
MZ(E'>(/X0'0.2(/$*%M>7UU!7$%=PY!(B?E!*>Q,`>OHDNC__T&#_#\/CM@`
M``!!@^Q`18GE0<'M!D&-;0%(P>4&2`'=9I#S#V\#2(GY2(/#0`\1!O,/;TO0
M#Q%.$/,/;U/@#Q%6(/,/;UOP#Q%>,.A"Z/__2#GK=<U!P>4&12GL26/$2(/X
M"')=2(M5`$B-3@A(@^'X2(E7.$B+5`7X2(E4!OA(*<Y(`?!(*?5(@^#X2(/X
M"'(82(/@^#'23(M$%0!,B0012(/""$@YPG+N1(EG>$B#Q"A;7E]=05Q!7<,/
M'X``````J`1U)$B%P'3?#[95`(A7.*@"=-0/MU0%_F:)5`;^Z\A(B=WI<O__
M_XM5`(E7.(M4!?R)5`;\Z[`/'X0``````$%455=64TB#["!(BT(HBW(HBWHP
M2,'H`P_.@^`_#\](B<N-2`%(F$F)U,9$`CB`2(UJ.$ACP4@!Z(/Y.`^.3@$`
M`$&X0````$$IR'00,=*)T8/"`<8$"`!$.<)R\DR)X>@KY___2<=$)#@`````
M2<=$)$``````2,=%$`````!(QT48`````$C'12``````2,=%*`````!(QT4P
M`````$&)?"1P3(GA08ET)'3HW^;__TF+!"1,B>%(P>@8B`-)BP0D2,'H$(A#
M`4F+!"2(8P))BP0DB$,#28M$)`A(P>@8B$,$28M$)`A(P>@0B$,%28M$)`B(
M8P9)BT0D"(A#!TF+1"002,'H&(A#"$F+1"002,'H$(A#"4F+1"00B&,*28M$
M)!"(0PM)BT0D&$C!Z!B(0PQ)BT0D&$C!Z!"(0PU)BT0D&(AC#DF+1"08B$,/
M28M$)"!(P>@8B$,028M$)"!(P>@0B$,128M$)""(8Q))BT0D((A#$TB#Q"!;
M7E]=05SI6@(``&8N#Q^$``````"Z.````$4QR2G*@_H(<DF)T4C'``````!(
MQT0(^`````!(C4@(2(/A^$@IR`'"@^+X@_H(#X+5_O__@^+X,<!!B<"#P`A.
MB0P!.=!R\NF]_O__9@\?1```]L($=2.%T@^$JO[__\8``/;"`@^$GO[__S')
M9HE,$/[IDO[__P\?`,<``````,=$$/P`````Z7S^__^0D)"0D)"0D)"0D)"0
M_R4BS@``D)#_)1+.``"0D/\E`LX``)"0_R7RS0``D)#_)>+-``"0D/\ETLT`
M`)"0_R7"S0``D)#_):K-``"0D/\EFLT``)"0_R6*S0``D)#_)7K-``"0D/\E
M:LT``)"0_R5:S0``D)#_)4K-``"0D/\E.LT``)"0_R4JS0``D)#_)1K-``"0
MD/\E"LT``)"0_R7ZS```D)#_)>K,``"0D/\E.LL``)"0D)"0D)"0D)#_)2+,
M``"0D)"0D)"0D)"0_R6JS```D)`/'X0``````%%02#T`$```2(U,)!AR&4B!
MZ0`0``!(@PD`2"T`$```2#T`$```=^=(*<%(@PD`6%G#D)"0D)"0D)"0D)"0
MD)#_);K*``"0D)"0D)"0D)"0_R6RR@``D)"0D)"0D)"0D/\ELLH``)"0D)"0
MD)"0D)#_):K*``"0D)"0D)"0D)"0_R6BR@``D)"0D)"0D)"0D/\EFLH``)"0
MD)"0D)"0D)#_):K*``"0D)"0D)"0D)"0_R6JR@``D)"0D)"0D)"0D/\EPLH`
M`)"0D)"0D)"0D)#_);K*``"0D)"0D)"0D)"0_R6RR@``D)"0D)"0D)"0D/\E
MJLH``)"0D)"0D)"0D)#_):+*``"0D)"0D)"0D)"0_R6:R@``D)"0D)"0D)"0
MD/\EDLH``)"0D)"0D)"0D)#_)8K*``"0D)"0D)"0D)"0_R6"R@``D)"0D)"0
MD)"0D/\E>LH``)"0D)"0D)"0D)#_)7+*``"0D)"0D)"0D)"0_R5JR@``D)"0
MD)"0D)"0D/\E8LH``)"0D)"0D)"0D)#_)5K*``"0D)"0D)"0D)"0_R52R@``
MD)"0D)"0D)"0D/\E6LH``)"0D)"0D)"0D)#_)5+*``"0D)"0D)"0D)"0_R5*
MR@``D)"0D)"0D)"0D/\E0LH``)"0D)"0D)"0D)#_)3K*``"0D)"0D)"0D)"0
M_R4RR@``D)"0D)"0D)"0D/\E*LH``)"0D)"0D)"0D)#_)2+*``"0D)"0D)"0
MD)"0_R4:R@``D)"0D)"0D)"0D/\E$LH``)"0D)"0D)"0D)#_)0K*``"0D)"0
MD)"0D)"0_R4"R@``D)"0D)"0D)"0D/\E^LD``)"0D)"0D)"0D)#_)?+)``"0
MD)"0D)"0D)"0_R7JR0``D)"0D)"0D)"0D/\EXLD``)"0D)"0D)"0D)#_)=K)
M``"0D)"0D)"0D)"02(/L*#'2Z%4```#_%6_(``"0D)"0D)"0D)"0D)"0D)##
MD)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0
MD)##D)"0D)"0D)"0D)"0D)"05E-(@^PH2(G.2(G3N0@```#HN@0``$B%VP^$
M$0(``$BZ`````%D!``!(B7-`2(TUC/___TB-#34'``!(B9/P````2(T5/P<`
M`$B)4U!(BQ74QP``2(FS@````$B--6[___](B4M(2(T-<____TB)4QA(C16H
MI```2(FSB````$B--6K___](B8N0````2+E(`0``O`L``$B)LY@```!(C35[
M_?__2(E3.$B-%1#]__](B4L(2(T-!?[__TB)<R!(C37J`P``2(E3*$B-%0\$
M``!(QP,`````QT,0!P```$B)2S!(B7-X2(F3&`$``$B%P`^$O0```$B+@.``
M``!(A<`/A*T```"+%2.D``"%T@^%GP```$B#/2,4````#X0-`@``2(,]'10`
M```/A&\"``!(@ST7%`````^$40(``$B#/1$4````#X0S`@``2(,]"Q0````/
MA!4"``!(@ST%%`````^$]P$``$B#/?\3````#X39`0``2(,]^1,````/A+L!
M``!F#V\%LQ,```\1`&8/;PVX$P``#Q%($&8/;Q6\$P``#Q%0(&8/;QW`$P``
M#Q%8,#')Z'4#```QR4B)@^@```!(C04U$P``2(E#6$B-!:H3``!(B4-@2(T%
M'Z,``$B)0VA(C07TI```2(E#<$B-!0EF``!(B8,@`0``2(T%-V8``$B)@R@!
M``!(C07MLO__2(F#,`$``.B!`@``N`$```!(@\0H6U[##Q]$``!(B7!`2(TU
MA?W__TBZ`````%D!``!(C0T\!0``2(FP@````$B--7;]__](B9#P````2(T5
M"`4``$B)L(@```!(C35J_?__2(E02$B+%:?%``!(B4A02(N(0`$``$B)L)``
M``!(C356_?__2(FPF````$B--7BB``!(B0I(C15N^___2(T-!_O__TB)<#A(
MOD@!``"\"P``2(EP"$B-->[[__](B5`@2(T5TP$``$B)2"A(C0WX`0``2,<`
M`````,=`$`<```!(B7`P2(E0>$B)B!@!``!(A<`/A*;^__](B<-(BX#@````
M2(7`#X7F_?__Z8[^__]F+@\?A```````2(L02(D5!A(``.GD_?__D$B+4#A(
MB14M$@``Z37^__](BU`P2(D5%1(``.D7_O__2(M0*$B)%?T1``#I^?W__TB+
M4"!(B17E$0``Z=O]__](BU`82(D5S1$``.F]_?__2(M0$$B)%;41``#IG_W_
M_TB+4`A(B16=$0``Z8']__],BP6)H0``BQ6+H0``2(L-C*$``.G'````#Q^`
M`````$B#[#B#^@)T=W=5A=(/A'T````QP$V%P$B-%<'___](B0U:H0``#Y3`
M3(D%0*$``,<%/J$```$```")!2"A``#HVP```$B#^/](B048H0``#Y7`#[;`
M2(/$.,,/'T0``+@!````@_H#=>RZ`P```$B#Q#CI2`````\?A```````N@(`
M``!(@\0XZ3(```!FD#'2Z"D```"%P'2[2(L-QJ```(E$)"SH10```(M$)"Q(
MQP6NH```_____^N:D)"0D+@!````PY"0D)"0D)"0D)##D)"0D)"0D)"0D)"0
MD)"0_R6"PP``D)"0D)"0D)"0D/\E@L,``)"0D)"0D)"0D)#_)7K#``"0D)"0
MD)"0D)"0_R7ZPP``D)"0D)"0D)"0D$%5051(@^PH3(TM8:```$F)S$B)T4R)
MZNCS^O__3(GJ3(GA2(/$*$%<05WI`````/\E,L,``)"0D)"0D)"0D)#_)1+%
M``"0D`\?A```````055!5%932(/L6(F,)(````!(B50D.$R)1"0PZ.#W__](
MC50D.$B+'61@``!,C40D,$B-C"2`````2(L#2(E$)$@QP.C(]O__2(L5(6``
M`$B+#0I@``!,C04#]___Z![W__](BP7G7P``3(LE$&```(`X`'4VZ'[V__](
MB057GP``2(G!2(7`#X2#`0``Z%[V__],BR7G7P``3(GAZ._V__](BP#&@`\*
M````3(GAZ-WV__](BP"`B&($```#BX0D@````(U(!(UP`TACR4C!X0/H4O;_
M_TR+5"0X1(N$)(````!)B<%(B07CG@``28L"28D!2(T%MR$``$F)00A(BP7;
M#@``28E!$$B-!:0A``!)B4$808/X`0^.Z````$F#P@A!@\`#N@0````/'T0`
M`$F+`DACRH/"`4F#P@A)B03)1#G"=>I(8])(BPV(G@``08GP2<<$T0````!(
MC17.Q/__2,=$)"``````Z&CU__^%P'112(L-79X``$R-;"1$Z&/U__],B>I(
MC0W,'@``08G$Z*F^__](A<!T-TB+#36>``#H./7__^A;]?__2(M$)$A(*P-U
M8T2)X$B#Q%A;7D%<05W#2(L-#)X``.C_]/__ZZ%(BPWNG0``2(7)=`?H;-G_
M_^NV2(T-'!X``$R)ZNA+OO__2(D%S)T``$B)P4B%P'28Z]FZ!````.DV____
MN0$```#H1O;__^@1]O__D.GKO?__D)"0D)"0D)"0D)#__________V!20``!
M``````````````#__________P``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````I0
M05(N<&T*```````````(<T```0``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE9`!0051(`%5315(`4UE3
M5$5-`"]T;7``5$U01$E2`%1%35!$25(`5$5-4`!435``55-%4DY!344`)3`R
M>``E<R5S)7,E<P!P87(M```````````E<SH@8W)E871I;VX@;V8@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````)7,Z
M('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E
M;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@!P87)L+F5X90`N
M<&%R`'-H;W)T(')E860``$-!0TA%`"5S)7-C86-H92TE<R5S`"5S)7-T96UP
M+25U)7,`)7,E<W1E;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ
M5D524TE/3@`Q+C`U.0````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C
M=&]R>0``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B
M9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS
M:2YC`````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ
M*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*
M<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E
M=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#
M(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P
M86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@
M/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F
M:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68[
M"FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S
M:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@
M;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS
M965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE
M("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B
M=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*
M)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?
M=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$
M('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`[
M"G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*
M;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M
M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*
M1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@
M=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD
M("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N
M97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@
M)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^
M9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD
M>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@
M)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@
M7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*
M8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^
M+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W
M;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F
M:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?
M<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT[
M"FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@
M/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7
M:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]
M/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T
M7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B
M72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@
M/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?
M;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?
M7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G
M;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA
M;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R
M<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R
M7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#
M86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B
M)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M
M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B
M*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H
M)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T
M<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T
M<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D
M8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B
M3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I
M<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N
M(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F
M+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER
M<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@
M)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM
M>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D
M9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH
M6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L
M("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A
M9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D
M8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\
M<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C
M)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F
M=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE
M>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M
M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK
M.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*
M;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U
M;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U
M;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES
M='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M
M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ
M1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@
M;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN
M;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P
M+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O
M+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A
M73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C
M86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?
M:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM
M>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2![
M"G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$
M4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER
M92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I
M;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R
M97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?
M34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T
M:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T
M87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T
M87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L
M("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%
M55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O
M8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M
M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@
M8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R
M"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD
M("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM0
M05)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA
M;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6
M*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R
M(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#([
M"GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF
M("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF
M("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I
M('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B
M=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`]
M(#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@
M)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]
M"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF
M="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE
M=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z
M.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF
M>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*
M=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*
M<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I
M<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@
M)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W
M)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z
M("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z
M.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O
M<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R
M(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT
M<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE
M=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI
M)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G
M/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D
M<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q
M6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+
M6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY
M("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4
M*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I
M+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z
M("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A
M9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E
M860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L
M<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@
M*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<
M0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H
M+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D
M97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET
M7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A
M+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@
M;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N
M9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S
M?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F
M;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.
M0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@
M("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF
M;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE
M+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F
M8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME
M>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@
M*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>
M+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI
M('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M
M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D
M,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*
M;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE
M*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN
M=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL
M92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I
M;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD
M=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH
M7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]6
M15)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2
M.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L
M("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG
M+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H
M:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C
M:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*
M;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2![
M"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B
M)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H
M(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE
M*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@
M,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A
M8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S
M=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H
M95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D
M8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%
M(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R
M*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I
M.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@
M;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL
M87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H
M*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H
M*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP
M.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@
M("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ
M2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q
M<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N
M:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II
M<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X
M=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]
M*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY
M("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M
M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE
M3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI
M8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*
M?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I
M;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA
M;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N
M8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R
M7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE
M(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M
M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&
M:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D
M9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E
M<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O
M9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@
M6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A
M<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S
M<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@
M/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I
M(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L
M("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA
M;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H
M.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I
M97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E
M(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E
M<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,[
M"G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%
M<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER
M92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I
M9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U
M:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(
M96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0[
M"G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V
M86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ
M571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@
M?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*
M979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R
M('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M
M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]4
M14U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF
M;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$
M25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O
M=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@
M)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*
M979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM]
M.PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA
M;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U
M:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@
M/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]
M"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K
M*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D
M8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@
M/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D
M<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A
M73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@
M/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S
M:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H
M+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#
M2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I
M>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`[
M"G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O
M7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D
M9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[
M7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.
M5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D
M,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D
M;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92([
M"G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L
M;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N
M86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*
M;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM
M;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H
M)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]
M"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.
M04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE
M('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA
M;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T
M<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O
M9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L
M("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US
M("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?
M97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D
M<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T
M<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*
M)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I
M"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2([
M"B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@
M;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*
M<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM0
M05)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI
M;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M
M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z
M.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R
M.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]
M?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI
M;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P
M87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T]
M(#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141]
M(#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#
M2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O
M<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%2
M7R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[
M(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L
M971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF
M("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]
M"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U
M:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N
M9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I
M=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E
M>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)
M3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*
M9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N
M86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@
M)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D
M0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*
M)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D
M.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?
M15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH```````````"@3P``
M````````````````D!1!``$``````````````*A'0``!``````````````"@
M1T```0```````````````!5!``$``````````````!@300`!````````````
M``!`1T```0``````````````8!-!``$``````````````$=#0SH@*$=.52D@
M,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q,2XT+C``````````
M``````````!'0T,Z("A'3E4I(#$Q+C0N,````````````````````$=#0SH@
M*$=.52D@,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q,2XT+C``
M``````````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````````````
M`$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q
M,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````
M`````````$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H
M1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```
M`````````````````$=#0SH@*$=.52D@,3$N-"XP````````````````````
M1T-#.B`H1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q
M+C0N,````````````````````$=#0SH@*$=.52D@,3$N-"XP````````````
M```````````````````!````D!0!`)"P``!``````!4!`,"P``!`````&!,!
M`-"P``!`````8!,!`/"P``!`````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13-L%X^$1D0-U+SV,]5*?F3P$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``#,4``!(
MX```0!0``&H6``!<X```<!8``-(6``!TX```X!8``!$7``!\X```(!<``",7
M``"$X```,!<``#,9``"(X```0!D``&P9``"@X```<!D``((:``"HX```D!H`
M`((?``"XX```D!\``/(?``#0X````"```%,@``#8X```8"```'LA``#DX```
M@"$``)TA``#TX```H"$``)8B``#\X```H"(``&TI```,X0``<"D``(,K```D
MX0``D"L``&(L```TX0``<"P``.-"``!0X0``\$(``$9#``!HX0``4$,``-A$
M``!PX0``X$0```-'``"$X0``H$H``+)*``"4X0``P$H``,%*``"<X0``T$H`
M`-%*``"@X0``X$H``.%*``"DX0``\$H``/%*``"HX0```$L``+!.``"LX0``
ML$X``,E.``"XX0``T$X``(Q/``"\X0``D$\``)9/``#$X0``H$\``*%/``#(
MX0``\$\``"!0``#,X0``0%```&!2``!`X0``8%(``&52``#8X0``````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!$P<`$P$&`@LP"F`)<`A0`L````$0"0`08@PP"V`*<`E0",`&
MT`3@`O````$%`@`%4@$P`00!``1"```!`````1D*`!D!&P(1,!!@#W`.4`W`
M"]`)X`+P`08"``8R`L`!$P8`$P$'`@LP"F`)P`+0`1`)`!!B##`+8`IP"5`(
MP`;0!.`"\````04"``52`3`!!P,`!V(#,`+````!$08`$0$'(`DP"&`'<`%0
M`00!``1"```!#`8`#'((,`=@!L`$T`+@`1,*`!,!,P`,,`M@"G`)4`C`!M`$
MX`+P`0D%``EB!3`$8`-P`L````$)!``)4@4P!,`"T`$*!0`*H@8P!6`$P`+0
M```!$PH`$P%7``PP"V`*<`E0",`&T`3@`O`!!`$`!$(```$,!P`,0@@P!V`&
M<`50!,`"T````0H&``HR!C`%8`1P`U`"P`$$`0`$0@```0````$````!````
M`0````$&`P`&0@(P`6````$````!!`$`!&(```$````!`````0@#``A"!,`"
MT````0``````````````````````````````````````````````````````
M``#UJ+MD`````-P``0`!````$@```!(````H``$`<``!`+@``0!P*0``@"$`
M`$!0```@8```0!D``)`K```@%P```"```'`9``"0'P``,!<``)`:``"@(@``
MH"$``.!$``#P0@``4$,``.`6``#D``$`^@`!`!4!`0`:`0$`)`$!`#$!`0`]
M`0$`3@$!`%8!`0!B`0$`<`$!`'T!`0"*`0$`EP$!`*D!`0"S`0$`O`$!`,<!
M`0````$``@`#``0`!0`&``<`"``)``H`"P`,``T`#@`/`!``$0!P87(N97AE
M`%A37TEN=&5R;F%L<U]005)?0D]/5`!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!M86EN`&UY7W!A<E]P;`!P87)?8F%S96YA;64`<&%R7V-L96%N=7``
M<&%R7V-U<G)E;G1?97AE8P!P87)?9&EE`'!A<E]D:7)N86UE`'!A<E]E;G9?
M8VQE86X`<&%R7V9I;F1P<F]G`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`
M<&%R7W-E='5P7VQI8G!A=&@`<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A
M=&4`>'-?:6YI=```````````````````````````````````````````````
M````````````````````:!`!````````````=!H!`.`2`0`($@$`````````
M``"$&@$`@!0!`!@2`0```````````/`:`0"0%`$`T!(!````````````!!L!
M`$@5`0````````````````````````````````!8%0$``````&@5`0``````
M=!4!``````"$%0$``````)`5`0``````H!4!``````"P%0$``````,05`0``
M````V!4!``````#D%0$``````/05`0```````!8!```````,%@$``````!@6
M`0``````+!8!``````!`%@$``````%`6`0``````7!8!``````!D%@$`````
M`'`6`0``````>!8!``````"$%@$``````)`6`0``````G!8!``````"D%@$`
M`````*P6`0``````N!8!``````#$%@$``````-`6`0``````W!8!``````#D
M%@$``````.P6`0``````^!8!```````,%P$``````"`7`0``````*!<!````
M```T%P$``````$`7`0``````3!<!``````!4%P$``````&`7`0``````:!<!
M``````!T%P$``````(`7`0``````C!<!``````"8%P$``````*07`0``````
ML!<!``````"\%P$``````,@7`0``````U!<!`````````````````.`7`0``
M``````````````#X%P$```````@8`0``````'!@!```````T&`$``````$P8
M`0``````7!@!``````!P&`$``````(`8`0``````D!@!``````"D&`$`````
M`+P8`0``````S!@!``````#@&`$``````/08`0``````!!D!```````D&0$`
M`````#@9`0``````2!D!``````!<&0$``````&P9`0``````>!D!``````"(
M&0$`````````````````E!D!`````````````````%@5`0``````:!4!````
M``!T%0$``````(05`0``````D!4!``````"@%0$``````+`5`0``````Q!4!
M``````#8%0$``````.05`0``````]!4!````````%@$```````P6`0``````
M&!8!```````L%@$``````$`6`0``````4!8!``````!<%@$``````&06`0``
M````<!8!``````!X%@$``````(06`0``````D!8!``````"<%@$``````*06
M`0``````K!8!``````"X%@$``````,06`0``````T!8!``````#<%@$`````
M`.06`0``````[!8!``````#X%@$```````P7`0``````(!<!```````H%P$`
M`````#07`0``````0!<!``````!,%P$``````%07`0``````8!<!``````!H
M%P$``````'07`0``````@!<!``````",%P$``````)@7`0``````I!<!````
M``"P%P$``````+P7`0``````R!<!``````#4%P$`````````````````X!<!
M`````````````````/@7`0``````"!@!```````<&`$``````#08`0``````
M3!@!``````!<&`$``````'`8`0``````@!@!``````"0&`$``````*08`0``
M````O!@!``````#,&`$``````.`8`0``````]!@!```````$&0$``````"09
M`0``````.!D!``````!(&0$``````%P9`0``````;!D!``````!X&0$`````
M`(@9`0````````````````"4&0$`````````````````#P!?7V-X85]A=&5X
M:70``!8`7U]E<G)N;P```",`7U]G971R965N=``````N`%]?;6%I;@`````P
M`%]?;65M8W!Y7V-H:P``3`!?7W-P<FEN=&9?8VAK`$\`7U]S=&%C:U]C:&M?
M9F%I;```40!?7W-T86-K7V-H:U]G=6%R9`!L`%]D;&Q?8W)T,`!S`%]I;7!U
M<F5?<'1R````@P!A8V-E<W,``````@%C86QL;V,`````0P%C;&]S961I<@``
M>P%C>6=W:6Y?9&5T86-H7V1L;`!]`6-Y9W=I;E]I;G1E<FYA;````)D!9&QL
M7V1L;&-R=#````"V`65N=FER;VX```#3`65X:70``#T"9G!R:6YT9@```$D"
M9G)E90``KP)G971P:60`````N@)G971P=W5I9```R`)G971U:60`````C`-L
M<V5E:P".`VQS=&%T`),#;6%L;&]C`````*(#;65M8VAR`````*0#;65M8W!Y
M`````*8#;65M;6]V90```*H#;6MD:7(`[`-O<&5N``#P`V]P96YD:7(````#
M!'!O<VEX7VUE;6%L:6=N`````"@$<'1H<F5A9%]A=&9O<FL`````M@1R96%D
M``"W!')E861D:7(```"Y!')E861L:6YK``"\!')E86QL;V,```#J!')M9&ER
M`'\%<W!R:6YT9@```(<%<W1A=```D@5S=')C:'(`````E`5S=')C;7``````
MF05S=')D=7``````HP5S=')L96X`````J`5S=')N8VUP````L`5S=')R8VAR
M````M`5S=')S='(`````N@5S=')T;VL`````+`9U;FQI;FL`````0@9V9G!R
M:6YT9@``+0!?7V5M=71L<U]G971?861D<F5S<P``&P!03%]D;U]U;F1U;7``
M`&\!4&5R;%]A=&9O<FM?;&]C:P``<`%097)L7V%T9F]R:U]U;FQO8VL`````
M[`%097)L7V-R;V%K7VYO8V]N=&5X=```@`)097)L7V=E=%]S=@```*\"4&5R
M;%]G=E]F971C:'!V````=@-097)L7VUG7W-E=````.X#4&5R;%]N97=84P``
M``#P`U!E<FQ?;F5W6%-?9FQA9W,``,8%4&5R;%]S869E<WES;6%L;&]C````
M`*<&4&5R;%]S=E]S971P=@"P!E!E<FQ?<W9?<V5T<'9N`````.(&4&5R;%]S
M>7-?:6YI=#,`````XP9097)L7W-Y<U]T97)M`&@'7U]E;75T;'-?=BY03%]C
M=7)R96YT7V-O;G1E>'0`:P=B;V]T7T1Y;F%,;V%D97(```!\!W!E<FQ?86QL
M;V,`````?@=P97)L7V-O;G-T<G5C=`````!_!W!E<FQ?9&5S=')U8W0`@`=P
M97)L7V9R964`@0=P97)L7W!A<G-E`````(('<&5R;%]R=6X``(D"1V5T36]D
M=6QE2&%N9&QE00```!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`8WEG=VEN,2YD
M;&P`%!`!`&-Y9V=C8U]S+7-E:"TQ+F1L;```*!`!`"@0`0`H$`$`*!`!`"@0
M`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!
M`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`&-Y9W!E<FPU7S,V+F1L;``\$`$`
M2T523D5,,S(N9&QL````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T"$!`&@%`````````````#@G`0"H"`````````````#@+P$`:`4`````
M````````2#4!`*@(`````````````/`]`0`^```````````````P/@$`6`(`
M````````````B$`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````'BB````8```#````""@````L```&````)"@H*"PH,"@T*#@H/"@````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
