/** @file id1map_load.h  id Tech 1 map format reader.
 *
 * @ingroup wadmapconverter
 *
 * @authors Copyright © 2007-2013 Daniel Swanson <danij@dengine.net>
 *
 * @par License
 * GPL: http://www.gnu.org/licenses/gpl.html
 *
 * <small>This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version. This program is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details. You should have received a copy of the GNU
 * General Public License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA</small>
 */

#ifndef WADMAPCONVERTER_ID1MAP_LOAD_H
#define WADMAPCONVERTER_ID1MAP_LOAD_H

#include "id1map.h"

/**
 * Determine the size (in bytes) of an element of the specified map data
 * lump @a type for the current map format.
 *
 * @param mapFormat     Map format identifier.
 * @param type          Map lump data type.
 * @return Size of an element of the specified type.
 */
size_t ElementSizeForMapLumpType(Id1Map::Format mapFormat, MapLumpType type);

/**
 * Read a line definition from the archived map.
 *
 * @param line  Line definition to be populated.
 * @param reader  Reader instance.
 */
void MLine_Read(mline_t &line, Id1Map &map, Reader &reader);

/// Doom64 format variant of @ref MLine_Read()
void MLine64_Read(mline_t &line, Id1Map &map, Reader &reader);

/// Hexen format variant of @ref MLine_Read()
void MLineHx_Read(mline_t &line, Id1Map &map, Reader &reader);

/**
 * Read a side definition from the archived map.
 *
 * @param side  Side definition to be populated.
 * @param reader  Reader instance.
 */
void MSide_Read(mside_t &side, Id1Map &map, Reader &reader);

/// Doom64 format variant of @ref MSide_Read()
void MSide64_Read(mside_t &side, Id1Map &map, Reader &reader);

/**
 * Read a sector definition from the archived map.
 *
 * @param sector  Sector definition to be populated.
 * @param reader  Reader instance.
 */
void MSector_Read(msector_t &sector, Id1Map &map, Reader &reader);

/// Doom64 format variant of @ref MSector_Read()
void MSector64_Read(msector_t &sector, Id1Map &map, Reader &reader);

/**
 * Read a thing definition from the archived map.
 *
 * @param thing  Thing definition to be populated.
 * @param reader  Reader instance.
 */
void MThing_Read(mthing_t &thing, Id1Map &map, Reader &reader);

/// Doom64 format variant of @ref MThing_Read()
void MThing64_Read(mthing_t &thing, Id1Map &map, Reader &reader);

/// Hexen format variant of @ref MThing_Read()
void MThingHx_Read(mthing_t &thing, Id1Map &map, Reader &reader);

/**
 * Read a surface tint definition from the archived map.
 *
 * @param tint  Surface tint definition to be populated.
 * @param reader  Reader instance.
 */
void SurfaceTint_Read(surfacetint_t &tint, Id1Map &map, Reader &reader);

#endif // WADMAPCONVERTER_ID1MAP_LOAD_H
