package tests::OldDlfAdapterTest;

use strict;

use base qw/ tests::DlfConverterProcessFixture /;

use vars qw/$converter_script/;

use Lire::OldDlfAdapter;
use Lire::DlfConverterProcess;
use File::Basename qw/dirname/;
use Cwd qw/realpath/;
use Lire::Utils qw/tempdir create_file/;

BEGIN {
    $converter_script =
      realpath( dirname(__FILE__)  . '/helpers' ) . '/oldconverter2dlf';
    chmod 0755, $converter_script;
};

sub schema_fixtures {
    return
      {
       'message' => <<EOF,
<lire:dlf-schema superservice="message" timestamp="ts"
 xmlns:lire="http://www.logreport.org/LDSML/">
<lire:field name="ts"  type="timestamp"/>
<lire:field name="host"    type="hostname"/>
<lire:field name="ipaddr"  type="ip"/>
<lire:field name="message" type="string"/>
<lire:field name="status"  type="string"/>
</lire:dlf-schema>

EOF
      };
}

sub import_job_fixtures {
    return
      {
       'test' => <<EOF,
ts=1030000000;ipaddr=127.0.0.1;host=localhost;message=Test
ts=1030000100;ipaddr=192.168.1.1;message=Ping
ts=;message=Corrupted
ignored=field;ts=1030000200;host=host.test.com;message=This is a msg.
EOF
      };
}

my %results =
  (
   'test' =>
   {
    'line_count'  => 4,
    'error_count' => 1,
    'dlf_count'   => 3,
    'ignored_count' => 0,
    'saved_count' => 0,
    'dlf' =>
    {
     'message' =>
     [
      {
       'dlf_id'  => 1,
       'dlf_source'  => 'myjob-20041202',
       'ts'      => 1030000000,
       'ipaddr'  => "127.0.0.1",
       'host'    => "localhost",
       'message' => "Test",
       'status'  => undef,
      },
      {
       'dlf_id'  => 2,
       'dlf_source'  => 'myjob-20041202',
       'ts'      => 1030000100,
       'host'    => undef,
       'ipaddr'  => "192.168.1.1",
       'message' => "Ping",
       'status'  => undef,
      },
      {
       'dlf_id'  => 3,
       'dlf_source'  => 'myjob-20041202',
       'ts'      => 1030000200,
       'host'    => "host.test.com",
       'ipaddr'  => undef,
       'message' => "This_is_a_msg.",
       'status'  => undef,
      },
     ],
    },
   },
  );



sub converter_fixtures {
    return
      [
       new Lire::OldDlfAdapter( "message", $converter_script ),
      ];
}

sub test_old_conversion {
    my $self = $_[0];

    local $ENV{'OldDlfAdapterTest'} = join( ":", @{Lire::Config->get( 'lr_schemas_path' )});
    my $p = $self->create_dlf_converter_process( "test", "oldconverter" );
    $p->{'_job_id'} = 'myjob-20041202';
    $p->run_import_job();
    $self->assert_dlf_converter_match_results( $results{'test'}, $p );
}

sub test_new {
    my $self = $_[0];

    $self->assert_dies( qr/missing \'schema\' parameter/,
                        sub { new Lire::OldDlfAdapter() } );

    $self->assert_dies( qr/invalid schema name:/,
                      sub { new Lire::OldDlfAdapter( 'unknown', $converter_script ) });

    $self->assert_dies( qr/missing \'script\' parameter/,
                        sub { new Lire::OldDlfAdapter( 'message' ) } );

    my $tmpdir = tempdir( "old_dlf_adapter_XXXXXX", 'CLEANUP' => 1 );
    create_file( $tmpdir . "oldconverter" );
    $self->assert_dies( qr/script is not executable: \'/,
                      sub { new Lire::OldDlfAdapter( 'message', "$tmpdir/old_dlf_adapter" ) } );

    my $adapter = new Lire::OldDlfAdapter( "message", $converter_script );
    $self->assert_isa( 'Lire::OldDlfAdapter', $adapter );
    $self->assert_str_equals( 'oldconverter', $adapter->name() );
    $self->assert_str_equals( $converter_script, $adapter->script() );
    $self->assert_str_equals( "message", $adapter->schemas() );
}

1;
