// Copyright 2017 PDFium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Original code copyright 2014 Foxit Software Inc. http://www.foxitsoftware.com

#ifndef CORE_FPDFAPI_RENDER_CPDF_DIBSOURCE_H_
#define CORE_FPDFAPI_RENDER_CPDF_DIBSOURCE_H_

#include <map>
#include <memory>
#include <vector>

#include "core/fpdfapi/page/cpdf_clippath.h"
#include "core/fpdfapi/page/cpdf_countedobject.h"
#include "core/fpdfapi/page/cpdf_graphicstates.h"
#include "core/fpdfapi/parser/cpdf_stream_acc.h"
#include "core/fpdfapi/render/cpdf_imageloader.h"
#include "core/fpdfapi/render/cpdf_rendercontext.h"
#include "core/fpdfapi/render/cpdf_renderoptions.h"
#include "core/fxcrt/retain_ptr.h"
#include "core/fxcrt/unowned_ptr.h"
#include "core/fxge/cfx_defaultrenderdevice.h"
#include "core/fxge/dib/cfx_dibsource.h"

class CCodec_Jbig2Context;
class CCodec_ScanlineDecoder;
class CPDF_Color;
class CPDF_Dictionary;
class CPDF_Document;
class CPDF_Stream;

struct DIB_COMP_DATA {
  float m_DecodeMin;
  float m_DecodeStep;
  int m_ColorKeyMin;
  int m_ColorKeyMax;
};

#define FPDF_HUGE_IMAGE_SIZE 60000000

class CPDF_DIBSource : public CFX_DIBSource {
 public:
  template <typename T, typename... Args>
  friend RetainPtr<T> pdfium::MakeRetain(Args&&... args);

  enum class LoadState : uint8_t { kFail, kSuccess, kContinue };

  ~CPDF_DIBSource() override;

  bool Load(CPDF_Document* pDoc, const CPDF_Stream* pStream);

  // CFX_DIBSource
  bool SkipToScanline(int line, PauseIndicatorIface* pPause) const override;
  uint8_t* GetBuffer() const override;
  const uint8_t* GetScanline(int line) const override;
  void DownSampleScanline(int line,
                          uint8_t* dest_scan,
                          int dest_bpp,
                          int dest_width,
                          bool bFlipX,
                          int clip_left,
                          int clip_width) const override;

  const CPDF_ColorSpace* GetColorSpace() const { return m_pColorSpace; }
  uint32_t GetMatteColor() const { return m_MatteColor; }

  LoadState StartLoadDIBSource(CPDF_Document* pDoc,
                               const CPDF_Stream* pStream,
                               bool bHasMask,
                               const CPDF_Dictionary* pFormResources,
                               CPDF_Dictionary* pPageResources,
                               bool bStdCS = false,
                               uint32_t GroupFamily = 0,
                               bool bLoadMask = false);
  LoadState ContinueLoadDIBSource(PauseIndicatorIface* pPause);
  RetainPtr<CPDF_DIBSource> DetachMask();

  bool IsJBigImage() const;

 private:
  CPDF_DIBSource();

  LoadState StartLoadMask();
  LoadState StartLoadMaskDIB();
  bool ContinueToLoadMask();
  LoadState ContinueLoadMaskDIB(PauseIndicatorIface* pPause);
  bool LoadColorInfo(const CPDF_Dictionary* pFormResources,
                     const CPDF_Dictionary* pPageResources);
  DIB_COMP_DATA* GetDecodeAndMaskArray(bool* bDefaultDecode, bool* bColorKey);
  RetainPtr<CFX_DIBitmap> LoadJpxBitmap();
  void LoadPalette();
  LoadState CreateDecoder();
  bool CreateDCTDecoder(const uint8_t* src_data,
                        uint32_t src_size,
                        const CPDF_Dictionary* pParams);
  void TranslateScanline24bpp(uint8_t* dest_scan,
                              const uint8_t* src_scan) const;
  bool TranslateScanline24bppDefaultDecode(uint8_t* dest_scan,
                                           const uint8_t* src_scan) const;
  void ValidateDictParam();
  void DownSampleScanline1Bit(int orig_Bpp,
                              int dest_Bpp,
                              uint32_t src_width,
                              const uint8_t* pSrcLine,
                              uint8_t* dest_scan,
                              int dest_width,
                              bool bFlipX,
                              int clip_left,
                              int clip_width) const;
  void DownSampleScanline8Bit(int orig_Bpp,
                              int dest_Bpp,
                              uint32_t src_width,
                              const uint8_t* pSrcLine,
                              uint8_t* dest_scan,
                              int dest_width,
                              bool bFlipX,
                              int clip_left,
                              int clip_width) const;
  void DownSampleScanline32Bit(int orig_Bpp,
                               int dest_Bpp,
                               uint32_t src_width,
                               const uint8_t* pSrcLine,
                               uint8_t* dest_scan,
                               int dest_width,
                               bool bFlipX,
                               int clip_left,
                               int clip_width) const;
  bool TransMask() const;

  UnownedPtr<CPDF_Document> m_pDocument;
  UnownedPtr<const CPDF_Stream> m_pStream;
  UnownedPtr<const CPDF_Dictionary> m_pDict;
  RetainPtr<CPDF_StreamAcc> m_pStreamAcc;
  CPDF_ColorSpace* m_pColorSpace = nullptr;
  uint32_t m_Family = 0;
  uint32_t m_bpc = 0;
  uint32_t m_bpc_orig = 0;
  uint32_t m_nComponents = 0;
  uint32_t m_GroupFamily = 0;
  uint32_t m_MatteColor = 0;
  bool m_bLoadMask = false;
  bool m_bDefaultDecode = true;
  bool m_bImageMask = false;
  bool m_bDoBpcCheck = true;
  bool m_bColorKey = false;
  bool m_bHasMask = false;
  bool m_bStdCS = false;
  DIB_COMP_DATA* m_pCompData = nullptr;
  uint8_t* m_pLineBuf = nullptr;
  uint8_t* m_pMaskedLine = nullptr;
  RetainPtr<CFX_DIBitmap> m_pCachedBitmap;
  RetainPtr<CPDF_DIBSource> m_pMask;
  RetainPtr<CPDF_StreamAcc> m_pGlobalStream;
  std::unique_ptr<CCodec_ScanlineDecoder> m_pDecoder;
  std::unique_ptr<CCodec_Jbig2Context> m_pJbig2Context;
  UnownedPtr<CPDF_Stream> m_pMaskStream;
  LoadState m_Status = LoadState::kFail;
};

#endif  // CORE_FPDFAPI_RENDER_CPDF_DIBSOURCE_H_
