# test_main - tests for ubuntu_sso.main
#
# Author: Natalia Bidart <natalia.bidart@canonical.com>
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the main SSO client code."""

import logging
import os
import urllib2

import gobject

# Unable to import 'lazr.restfulclient.*'
# pylint: disable=F0401
from lazr.restfulclient.errors import HTTPError
# pylint: enable=F0401
from mocker import Mocker, MockerTestCase, ARGS, KWARGS, ANY
from twisted.internet.defer import Deferred
from twisted.trial.unittest import TestCase

import ubuntu_sso.keyring
import ubuntu_sso.main

from contrib.testing.testcase import MementoHandler
from ubuntu_sso import gui
from ubuntu_sso.keyring import get_token_name, U1_APP_NAME
from ubuntu_sso.main import (
    AuthenticationError, blocking, EmailTokenError,
    except_to_errdict, InvalidEmailError, InvalidPasswordError,
    keyring_get_credentials, keyring_store_credentials, logger,
    NewPasswordError, PING_URL, SERVICE_URL,
    RegistrationError, ResetPasswordTokenError,
    SSOCredentials, SSOLogin, SSOLoginProcessor)


# Access to a protected member 'yyy' of a client class
# pylint: disable=W0212


APP_NAME = 'The Coolest App Ever'
CAPTCHA_PATH = os.path.abspath(os.path.join(os.curdir, 'ubuntu_sso', 'tests',
                                            'files', 'captcha.png'))
CAPTCHA_ID = 'test'
CAPTCHA_SOLUTION = 'william Byrd'
CANT_RESET_PASSWORD_CONTENT = "CanNotResetPassowrdError: " \
                              "Can't reset password for this account"
RESET_TOKEN_INVALID_CONTENT = "AuthToken matching query does not exist."
EMAIL = 'test@example.com'
EMAIL_ALREADY_REGISTERED = 'a@example.com'
EMAIL_TOKEN = 'B2Pgtf'
HELP = 'help text'
PASSWORD = 'be4tiFul'
RESET_PASSWORD_TOKEN = '8G5Wtq'
TOKEN = {u'consumer_key': u'xQ7xDAz',
         u'consumer_secret': u'KzCJWCTNbbntwfyCKKjomJDzlgqxLy',
         u'token_name': u'test',
         u'token': u'GkInOfSMGwTXAUoVQwLUoPxElEEUdhsLVNTPhxHJDUIeHCPNEo',
         u'token_secret': u'qFYImEtlczPbsCnYyuwLoPDlPEnvNcIktZphPQklAWrvyfFMV'}
TOKEN_NAME = get_token_name(APP_NAME)
STATUS_UNKNOWN = {'status': 'yadda-yadda'}
STATUS_ERROR = {'status': 'error', 'errors': {'something': ['Bla', 'Ble']}}
STATUS_OK = {'status': 'ok'}
STATUS_EMAIL_UNKNOWN = {'status': 'yadda-yadda'}
STATUS_EMAIL_ERROR = {'errors': {'email_token': ['Error1', 'Error2']}}
STATUS_EMAIL_OK = {'email': EMAIL}
TC_URL = 'tcurl'
WINDOW_ID = 5

NO_OP = lambda *args, **kwargs: None
LOGIN_OR_REGISTER_ARGS = (APP_NAME, TC_URL, HELP, WINDOW_ID)
LOGIN_OR_REGISTER_GUI_ARGS = LOGIN_OR_REGISTER_ARGS + (False,)
LOGIN_ONLY_ARGS = (APP_NAME, HELP, WINDOW_ID)
LOGIN_ONLY_GUI_ARGS = (APP_NAME, None, HELP, WINDOW_ID, True)


class FakedResponse(object):
    """Fake a urlopen response."""

    def __init__(self, *args, **kwargs):
        for k, val in kwargs.iteritems():
            setattr(self, k, val)


class FakedCaptchas(object):
    """Fake the captcha generator."""

    def new(self):
        """Return a fix captcha)."""
        return {'image_url': 'file://%s' % CAPTCHA_PATH,
                'captcha_id': CAPTCHA_ID}


class FakedRegistrations(object):
    """Fake the registrations service."""

    def register(self, email, password, captcha_id, captcha_solution):
        """Fake registration. Return a fix result."""
        if email == EMAIL_ALREADY_REGISTERED:
            return {'status': 'error',
                    'errors': {'email': 'Email already registered'}}
        elif captcha_id is None and captcha_solution is None:
            return STATUS_UNKNOWN
        elif captcha_id != CAPTCHA_ID or captcha_solution != CAPTCHA_SOLUTION:
            return STATUS_ERROR
        else:
            return STATUS_OK

    def request_password_reset_token(self, email):
        """Fake password reset token. Return a fix result."""
        if email is None:
            return STATUS_UNKNOWN
        elif email != EMAIL:
            raise HTTPError(response=None, content=CANT_RESET_PASSWORD_CONTENT)
        else:
            return STATUS_OK

    def set_new_password(self, email, token, new_password):
        """Fake the setting of new password. Return a fix result."""
        if email is None and token is None and new_password is None:
            return STATUS_UNKNOWN
        elif email != EMAIL or token != RESET_PASSWORD_TOKEN:
            raise HTTPError(response=None, content=RESET_TOKEN_INVALID_CONTENT)
        else:
            return STATUS_OK


class FakedAuthentications(object):
    """Fake the authentications service."""

    def authenticate(self, token_name):
        """Fake authenticate. Return a fix result."""
        if not token_name.startswith(TOKEN_NAME):
            raise HTTPError(response=None, content=None)
        else:
            return TOKEN


class FakedAccounts(object):
    """Fake the accounts service."""

    def __init__(self):
        self.preferred_email = EMAIL

    def validate_email(self, email_token):
        """Fake the email validation. Return a fix result."""
        if email_token is None:
            return STATUS_EMAIL_UNKNOWN
        elif email_token == EMAIL_ALREADY_REGISTERED:
            return {'status': 'error',
                    'errors': {'email': 'Email already registered'}}
        elif email_token != EMAIL_TOKEN:
            return STATUS_EMAIL_ERROR
        else:
            return STATUS_EMAIL_OK

    # pylint: disable=E0202, C0103

    def me(self):
        """Fake the 'me' information."""
        return {u'username': u'Wh46bKY',
                u'preferred_email': self.preferred_email,
                u'displayname': u'',
                u'unverified_emails': [u'aaaaaa@example.com'],
                u'verified_emails': [],
                u'openid_identifier': u'Wh46bKY'}


class FakedSSOServer(object):
    """Fake an SSO server."""

    def __init__(self, authorizer, service_root):
        self.captchas = FakedCaptchas()
        self.registrations = FakedRegistrations()
        self.authentications = FakedAuthentications()
        self.accounts = FakedAccounts()


class SSOLoginProcessorTestCase(TestCase, MockerTestCase):
    """Test suite for the SSO login processor."""

    def setUp(self):
        """Init."""
        self.processor = SSOLoginProcessor(sso_service_class=FakedSSOServer)
        self.register_kwargs = dict(email=EMAIL, password=PASSWORD,
                                    captcha_id=CAPTCHA_ID,
                                    captcha_solution=CAPTCHA_SOLUTION)
        self.login_kwargs = dict(email=EMAIL, password=PASSWORD,
                                 token_name=TOKEN_NAME)

    def tearDown(self):
        """Clean up."""
        self.processor = None

    def test_generate_captcha(self):
        """Captcha can be generated."""
        filename = self.mktemp()
        self.addCleanup(lambda: os.remove(filename))
        captcha_id = self.processor.generate_captcha(filename)
        self.assertEqual(CAPTCHA_ID, captcha_id, 'captcha id must be correct.')
        self.assertTrue(os.path.isfile(filename), '%s must exist.' % filename)

        with open(CAPTCHA_PATH) as f:
            expected = f.read()
        with open(filename) as f:
            actual = f.read()
        self.assertEqual(expected, actual, 'captcha image must be correct.')

    def test_register_user_checks_valid_email(self):
        """Email is validated."""
        self.register_kwargs['email'] = 'notavalidemail'
        self.assertRaises(InvalidEmailError,
                          self.processor.register_user, **self.register_kwargs)

    def test_register_user_checks_valid_password(self):
        """Password is validated."""
        self.register_kwargs['password'] = ''
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        # 7 chars, one less than expected
        self.register_kwargs['password'] = 'tesT3it'
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        self.register_kwargs['password'] = 'test3it!'  # no upper case
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

        self.register_kwargs['password'] = 'testIt!!'  # no number
        self.assertRaises(InvalidPasswordError,
                          self.processor.register_user, **self.register_kwargs)

    # register

    def test_register_user_if_status_ok(self):
        """A user is succesfuy registered into the SSO server."""
        result = self.processor.register_user(**self.register_kwargs)
        self.assertEqual(EMAIL, result, 'registration was successful.')

    def test_register_user_if_status_error(self):
        """Proper error is raised if register fails."""
        self.register_kwargs['captcha_id'] = CAPTCHA_ID * 2  # incorrect
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, STATUS_ERROR['errors'])
            self.assertEqual(val, "\n".join(STATUS_ERROR['errors'][k]))

    def test_register_user_if_status_error_with_string_message(self):
        """Proper error is raised if register fails."""
        self.register_kwargs['email'] = EMAIL_ALREADY_REGISTERED
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, {'email': 'Email already registered'})
            self.assertEqual(val, 'Email already registered')

    def test_register_user_if_status_unknown(self):
        """Proper error is raised if register returns an unknown status."""
        self.register_kwargs['captcha_id'] = None
        self.register_kwargs['captcha_solution'] = None
        failure = self.assertRaises(RegistrationError,
                                    self.processor.register_user,
                                    **self.register_kwargs)
        self.assertIn('Received unknown status: %s' % STATUS_UNKNOWN, failure)

    # login

    def test_login_if_http_error(self):
        """Proper error is raised if authentication fails."""
        self.login_kwargs['token_name'] = APP_NAME * 2  # invalid token name
        self.assertRaises(AuthenticationError,
                          self.processor.login, **self.login_kwargs)

    def test_login_if_no_error(self):
        """A user can be succesfully logged in into the SSO service."""
        result = self.processor.login(**self.login_kwargs)
        self.assertEqual(TOKEN, result, 'authentication was successful.')

    # is_validated

    def test_is_validated(self):
        """If preferred email is not None, user is validated."""
        result = self.processor.is_validated(token=TOKEN)
        self.assertTrue(result, 'user must be validated.')

    def test_is_not_validated(self):
        """If preferred email is None, user is not validated."""
        service = FakedSSOServer(None, None)
        service.accounts.preferred_email = None
        result = self.processor.is_validated(sso_service=service,
                                             token=TOKEN)
        self.assertFalse(result, 'user must not be validated.')

    def test_is_not_validated_empty_result(self):
        """If preferred email is None, user is not validated."""
        service = FakedSSOServer(None, None)
        service.accounts.me = lambda: {}
        result = self.processor.is_validated(sso_service=service,
                                             token=TOKEN)
        self.assertFalse(result, 'user must not be validated.')

    # validate_email

    def test_validate_email_if_status_ok(self):
        """A email is succesfuy validated in the SSO server."""
        self.login_kwargs['email_token'] = EMAIL_TOKEN  # valid email token
        result = self.processor.validate_email(**self.login_kwargs)
        self.assertEqual(TOKEN, result, 'email validation was successful.')

    def test_validate_email_if_status_error(self):
        """Proper error is raised if email validation fails."""
        self.login_kwargs['email_token'] = EMAIL_TOKEN * 2  # invalid token
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, STATUS_EMAIL_ERROR['errors'])
            self.assertEqual(val, "\n".join(STATUS_EMAIL_ERROR['errors'][k]))

    def test_validate_email_if_status_error_with_string_message(self):
        """Proper error is raised if register fails."""
        self.login_kwargs['email_token'] = EMAIL_ALREADY_REGISTERED
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        for k, val in failure.args[0].items():
            self.assertIn(k, {'email': 'Email already registered'})
            self.assertEqual(val, 'Email already registered')

    def test_validate_email_if_status_unknown(self):
        """Proper error is raised if email validation returns unknown."""
        self.login_kwargs['email_token'] = None
        failure = self.assertRaises(EmailTokenError,
                                    self.processor.validate_email,
                                    **self.login_kwargs)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, failure)

    # reset_password

    def test_request_password_reset_token_if_status_ok(self):
        """A reset password token is succesfuly sent."""
        result = self.processor.request_password_reset_token(email=EMAIL)
        self.assertEqual(EMAIL, result,
                         'password reset token must be successful.')

    def test_request_password_reset_token_if_http_error(self):
        """Proper error is raised if password token request fails."""
        exc = self.assertRaises(ResetPasswordTokenError,
                                self.processor.request_password_reset_token,
                                email=EMAIL * 2)
        self.assertIn(CANT_RESET_PASSWORD_CONTENT, exc)

    def test_request_password_reset_token_if_status_unknown(self):
        """Proper error is raised if password token request returns unknown."""
        exc = self.assertRaises(ResetPasswordTokenError,
                                self.processor.request_password_reset_token,
                                email=None)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, exc)

    def test_set_new_password_if_status_ok(self):
        """A new password is succesfuy set."""
        result = self.processor.set_new_password(email=EMAIL,
                                                 token=RESET_PASSWORD_TOKEN,
                                                 new_password=PASSWORD)
        self.assertEqual(EMAIL, result,
                         'new password must be set successfully.')

    def test_set_new_password_if_http_error(self):
        """Proper error is raised if setting a new password fails."""
        exc = self.assertRaises(NewPasswordError,
                                self.processor.set_new_password,
                                email=EMAIL * 2,
                                token=RESET_PASSWORD_TOKEN * 2,
                                new_password=PASSWORD)
        self.assertIn(RESET_TOKEN_INVALID_CONTENT, exc)

    def test_set_new_password_if_status_unknown(self):
        """Proper error is raised if setting a new password returns unknown."""
        exc = self.assertRaises(NewPasswordError,
                                self.processor.set_new_password,
                                email=None, token=None, new_password=None)
        self.assertIn('Received invalid reply: %s' % STATUS_UNKNOWN, exc)


class BlockingSampleException(Exception):
    """The exception that will be thrown by the fake blocking."""


class SsoDbusTestCase(TestCase):
    """Test the SSOLogin DBus interface."""

    def setUp(self):
        """Create the mocking bus."""
        self.real_blocking = ubuntu_sso.main.blocking
        self.mocker = Mocker()
        self.mockbusname = self.mocker.mock()
        mockbus = self.mocker.mock()
        self.mockbusname.get_bus()
        self.mocker.result(mockbus)
        mockbus._register_object_path(ARGS)
        self.mockprocessorclass = None

        def ksc(k, val, callback, *cb_args):
            """Assert over token and app_name."""
            self.assertEqual(k, APP_NAME)
            self.assertEqual(val, TOKEN)
            self.keyring_was_set = True
            self.keyring_values = k, val
            callback(*cb_args)

        self.patch(ubuntu_sso.main, "keyring_store_credentials", ksc)
        self.keyring_was_set = False
        self.keyring_values = None

    def tearDown(self):
        """Verify the mocking bus and shut it down."""
        self.mocker.verify()
        self.mocker.restore()

    def fake_ok_blocking(self, f, app, cb, eb):
        """A fake blocking function that succeeds."""
        cb(app, f())

    def fake_err_blocking(self, f, app, cb, eb):
        """A fake blocking function that fails."""
        try:
            f()
        except Exception, e:  # pylint: disable=W0703
            eb(app, except_to_errdict(e))
        else:
            eb(app, except_to_errdict(BlockingSampleException()))

    def test_creation(self):
        """Test that the object creation is successful."""
        self.mocker.replay()
        SSOLogin(self.mockbusname)

    def create_mock_processor(self):
        """Create a mock processor from a dummy processor class."""
        self.mockprocessorclass = self.mocker.mock()
        mockprocessor = self.mocker.mock()
        self.mockprocessorclass(ARGS, KWARGS)
        self.mocker.result(mockprocessor)
        return mockprocessor

    def test_generate_captcha(self):
        """Test that the captcha method works ok."""
        d = Deferred()
        filename = "sample filename"
        expected_result = "expected result"
        self.create_mock_processor().generate_captcha(filename)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "CaptchaGenerated", verify)
        self.patch(client, "CaptchaGenerationError", d.errback)
        client.generate_captcha(APP_NAME, filename)
        return d

    def test_generate_captcha_error(self):
        """Test that the captcha method fails as expected."""
        d = Deferred()
        filename = "sample filename"
        expected_result = "expected result"
        self.create_mock_processor().generate_captcha(filename)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "CaptchaGenerated", d.errback)
        self.patch(client, "CaptchaGenerationError", verify)
        client.generate_captcha(APP_NAME, filename)
        return d

    def test_register_user(self):
        """Test that the register_user method works ok."""
        d = Deferred()
        expected_result = "expected result"
        self.create_mock_processor().register_user(EMAIL, PASSWORD, CAPTCHA_ID,
                                                   CAPTCHA_SOLUTION)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "UserRegistered", verify)
        self.patch(client, "UserRegistrationError", d.errback)
        client.register_user(APP_NAME, EMAIL, PASSWORD, CAPTCHA_ID,
                             CAPTCHA_SOLUTION)
        return d

    def test_register_user_error(self):
        """Test that the register_user method fails as expected."""
        d = Deferred()
        expected_result = "expected result"
        self.create_mock_processor().register_user(EMAIL, PASSWORD, CAPTCHA_ID,
                                                   CAPTCHA_SOLUTION)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "UserRegistered", d.errback)
        self.patch(client, "UserRegistrationError", verify)
        client.register_user(APP_NAME, EMAIL, PASSWORD, CAPTCHA_ID,
                             CAPTCHA_SOLUTION)
        return d

    def test_login(self):
        """Test that the login method works ok."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.login(EMAIL, PASSWORD, TOKEN_NAME)
        self.mocker.result(TOKEN)
        processor.is_validated(TOKEN)
        self.mocker.result(True)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            self.assertTrue(self.keyring_was_set, "The keyring should be set")
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", verify)
        self.patch(client, "LoginError", d.errback)
        self.patch(client, "UserNotValidated", d.errback)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_login_user_not_validated(self):
        """Test that the login sends EmailNotValidated signal."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.login(EMAIL, PASSWORD, TOKEN_NAME)
        self.mocker.result(TOKEN)
        processor.is_validated(TOKEN)
        self.mocker.result(False)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, email):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(email, EMAIL)
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", d.errback)
        self.patch(client, "LoginError", d.errback)
        self.patch(client, "UserNotValidated", verify)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_login_error(self):
        """Test that the login method fails as expected."""
        d = Deferred()
        self.create_mock_processor()
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)

        def fake_gtn(*args):
            """A fake get_token_name that fails."""
            raise BlockingSampleException()

        self.patch(ubuntu_sso.main, "get_token_name", fake_gtn)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "LoggedIn", d.errback)
        self.patch(client, "LoginError", verify)
        self.patch(client, "UserNotValidated", d.errback)
        client.login(APP_NAME, EMAIL, PASSWORD)
        return d

    def test_validate_email(self):
        """Test that the validate_email method works ok."""
        d = Deferred()
        self.create_mock_processor().validate_email(EMAIL, PASSWORD,
                                                    EMAIL_TOKEN, TOKEN_NAME)
        self.mocker.result(TOKEN)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            self.assertTrue(self.keyring_was_set, "The keyring should be set")
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "EmailValidated", verify)
        self.patch(client, "EmailValidationError", d.errback)
        client.validate_email(APP_NAME, EMAIL, PASSWORD, EMAIL_TOKEN)
        return d

    def test_validate_email_error(self):
        """Test that the validate_email method fails as expected."""
        d = Deferred()
        self.create_mock_processor()
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)

        def fake_gtn(*args):
            """A fake get_token_name that fails."""
            raise BlockingSampleException()

        self.patch(ubuntu_sso.main, "get_token_name", fake_gtn)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertFalse(self.keyring_was_set, "Keyring should not be set")
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "EmailValidated", d.errback)
        self.patch(client, "EmailValidationError", verify)
        client.validate_email(APP_NAME, EMAIL, PASSWORD, EMAIL_TOKEN)
        return d

    def test_request_password_reset_token(self):
        """Test that the request_password_reset_token method works ok."""
        d = Deferred()
        processor = self.create_mock_processor()
        processor.request_password_reset_token(EMAIL)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.result(EMAIL)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordResetTokenSent", verify)
        self.patch(client, "PasswordResetError", d.errback)
        client.request_password_reset_token(APP_NAME, EMAIL)
        return d

    def test_request_password_reset_token_error(self):
        """Test the request_password_reset_token method fails as expected."""
        d = Deferred()

        self.create_mock_processor().request_password_reset_token(EMAIL)
        self.mocker.result(EMAIL)
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordResetTokenSent", d.errback)
        self.patch(client, "PasswordResetError", verify)
        client.request_password_reset_token(APP_NAME, EMAIL)
        return d

    def test_set_new_password(self):
        """Test that the set_new_password method works ok."""
        d = Deferred()
        self.create_mock_processor().set_new_password(EMAIL, EMAIL_TOKEN,
                                                      PASSWORD)
        self.mocker.result(EMAIL)
        self.patch(ubuntu_sso.main, "blocking", self.fake_ok_blocking)
        self.mocker.replay()

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, EMAIL)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordChanged", verify)
        self.patch(client, "PasswordChangeError", d.errback)
        client.set_new_password(APP_NAME, EMAIL, EMAIL_TOKEN, PASSWORD)
        return d

    def test_set_new_password_error(self):
        """Test that the set_new_password method fails as expected."""
        d = Deferred()
        expected_result = "expected result"

        self.create_mock_processor().set_new_password(EMAIL, EMAIL_TOKEN,
                                                      PASSWORD)
        self.mocker.result(expected_result)
        self.patch(ubuntu_sso.main, "blocking", self.fake_err_blocking)
        self.mocker.replay()

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(app_name, APP_NAME)
            d.callback("Ok")

        client = SSOLogin(self.mockbusname,
                          sso_login_processor_class=self.mockprocessorclass)
        self.patch(client, "PasswordChanged", d.errback)
        self.patch(client, "PasswordChangeError", verify)
        client.set_new_password(APP_NAME, EMAIL, EMAIL_TOKEN, PASSWORD)
        return d


class BlockingFunctionTestCase(TestCase):
    """Tests for the "blocking" function."""

    timeout = 5

    def test_blocking(self):
        """Test the normal behaviour."""
        d = Deferred()
        expected_result = "expected result"

        def f():
            """No failure."""
            return expected_result

        def verify(app_name, result):
            """The actual test."""
            self.assertEqual(result, expected_result)
            self.assertEqual(app_name, APP_NAME)
            d.callback(result)

        blocking(f, APP_NAME, verify, d.errback)
        return d

    def test_blocking_error(self):
        """Test the behaviour when an Exception is raised."""
        d = Deferred()
        expected_error_message = "expected error message"

        def f():
            """Failure."""
            raise BlockingSampleException(expected_error_message)

        def verify(app_name, errdict):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            self.assertEqual(errdict["errtype"], "BlockingSampleException")
            self.assertEqual(errdict["message"], expected_error_message)
            d.callback("Ok")

        blocking(f, APP_NAME, d.errback, verify)
        return d


class TestExceptToErrdictException(Exception):
    """A dummy exception for the following testcase."""


class ExceptToErrdictTestCase(TestCase):
    """Tests for the except_to_errdict function."""

    def test_first_arg_is_dict(self):
        """If the first arg is a dict, use it as the base dict."""
        sample_dict = {
            "errorcode1": "error message 1",
            "errorcode2": "error message 2",
            "errorcode3": "error message 3",
        }
        e = TestExceptToErrdictException(sample_dict)
        result = except_to_errdict(e)

        self.assertEqual(result["errtype"], e.__class__.__name__)
        for k in sample_dict.keys():
            self.assertIn(k, result)
            self.assertEqual(result[k], sample_dict[k])

    def test_first_arg_is_str(self):
        """If the first arg is a str, use it as the message."""
        sample_string = "a sample string"
        e = TestExceptToErrdictException(sample_string)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], sample_string)

    def test_first_arg_is_unicode(self):
        """If the first arg is a unicode, use it as the message."""
        sample_string = u"a sample string"
        e = TestExceptToErrdictException(sample_string)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], sample_string)

    def test_no_args_at_all(self):
        """If there are no args, use the class docstring."""
        e = TestExceptToErrdictException()
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)
        self.assertEqual(result["message"], e.__class__.__doc__)

    def test_some_other_thing_as_first_arg(self):
        """If first arg is not basestring nor dict, then repr all args."""
        sample_args = (None, u"unicode2\ufffd", "errorcode3")
        e = TestExceptToErrdictException(*sample_args)
        result = except_to_errdict(e)
        self.assertEqual(result["errtype"], e.__class__.__name__)


class KeyringCredentialsTestCase(TestCase, MockerTestCase):
    """Check the functions that access the keyring."""

    # Invalid name (should match ([a-z_][a-z0-9_]*|[A-Z_][A-Z0-9_]*)$)
    # pylint: disable=C0103

    def test_keyring_store_cred(self):
        """Verify the method that stores credentials."""
        idle_add = lambda f, *args, **kwargs: f(*args, **kwargs)
        self.patch(gobject, "idle_add", idle_add)
        token_value = TOKEN
        mockKeyringClass = self.mocker.replace("ubuntu_sso.keyring.Keyring")
        mockKeyringClass(APP_NAME)
        mockKeyring = self.mocker.mock()
        callback = self.mocker.mock()
        self.mocker.result(mockKeyring)
        mockKeyring.set_ubuntusso_attr(token_value)
        callback(1, 2, 3)
        self.mocker.replay()

        keyring_store_credentials(APP_NAME, token_value, callback, 1, 2, 3)

    def test_keyring_get_cred(self):
        """The method returns the right token."""
        mockKeyringClass = self.mocker.replace("ubuntu_sso.keyring.Keyring")
        mockKeyringClass(APP_NAME)
        mockKeyring = self.mocker.mock()
        self.mocker.result(mockKeyring)
        mockKeyring.get_ubuntusso_attr()
        self.mocker.result(TOKEN)
        self.mocker.replay()

        token = keyring_get_credentials(APP_NAME)
        self.assertEqual(token, TOKEN)

    def test_keyring_get_cred_not_found(self):
        """The method returns None when the token is not found."""
        mockKeyringClass = self.mocker.replace("ubuntu_sso.keyring.Keyring")
        mockKeyringClass(APP_NAME)
        mockKeyring = self.mocker.mock()
        self.mocker.result(mockKeyring)
        mockKeyring.get_ubuntusso_attr()
        self.mocker.result(None)
        self.mocker.replay()

        token = keyring_get_credentials(APP_NAME)
        self.assertEqual(token, None)


class RegisterSampleException(Exception):
    """A mock exception thrown just when testing."""


class CredentialsTestCase(TestCase, MockerTestCase):
    """Tests for the credentials related DBus methods."""

    # Invalid name (should match ([a-z_][a-z0-9_]*|[A-Z_][A-Z0-9_]*)$)
    # pylint: disable=C0103

    timeout = 5

    def test_find_credentials(self):
        """find_credentials immediately returns the token when found."""
        expected_token = "expected token"
        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(expected_token)
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        token = client.find_credentials(APP_NAME)
        self.assertEqual(token, expected_token)

    def test_credentials_not_found(self):
        """find_credentials immediately returns '' when no token found."""
        expected_creds = {}
        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(None)
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        token = client.find_credentials(APP_NAME)
        self.assertEqual(token, expected_creds)

    def test_login_or_register(self):
        """login_or_register_... throws the signal when token is found."""
        expected_creds = TOKEN
        d = Deferred()

        def verify(app_name, credentials):
            """The actual test."""
            self.assertEqual(credentials, expected_creds)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(expected_creds)
        self.mocker.replay()
        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_or_register_ui", self.fail)
        self.patch(client, "CredentialsFound", verify)
        self.patch(client, "CredentialsError", self.fail)
        client.login_or_register_to_get_credentials(*LOGIN_OR_REGISTER_ARGS)
        return d

    def test_login_or_register_not_found(self):
        """Check that login_or_register_... opens the ui when no cred found."""
        d = Deferred()

        def verify(result, *a):
            """The actual test."""
            self.assertEqual(result, APP_NAME)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(None)
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_or_register_ui", verify)
        self.patch(client, "CredentialsFound", self.fail)
        self.patch(client, "CredentialsError", self.fail)
        client.login_or_register_to_get_credentials(*LOGIN_OR_REGISTER_ARGS)
        return d

    def test_login_or_register_problem(self):
        """login_or_register_... returns the right signal on error."""
        expected_error = "Sample Error - not for resale"
        d = Deferred()

        def verify(app_name, error_message, detailed_error):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.throw(RegisterSampleException(expected_error))
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_or_register_ui", self.fail)
        self.patch(client, "CredentialsFound", self.fail)
        self.patch(client, "CredentialsError", verify)
        client.login_or_register_to_get_credentials(*LOGIN_OR_REGISTER_ARGS)
        return d

    def test_login_only(self):
        """login_only_... throws the signal when token is found."""
        expected_creds = TOKEN
        d = Deferred()

        def verify(app_name, credentials):
            """The actual test."""
            self.assertEqual(credentials, expected_creds)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(expected_creds)
        self.mocker.replay()
        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_only_ui", self.fail)
        self.patch(client, "CredentialsFound", verify)
        self.patch(client, "CredentialsError", self.fail)
        client.login_to_get_credentials(*LOGIN_ONLY_ARGS)
        return d

    def test_login_only_not_found(self):
        """Check that login_only_... opens the ui when no cred found."""
        d = Deferred()

        def verify(result, *a):
            """The actual test."""
            self.assertEqual(result, APP_NAME)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.result(None)
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_only_ui", verify)
        self.patch(client, "CredentialsFound", self.fail)
        self.patch(client, "CredentialsError", self.fail)
        client.login_to_get_credentials(*LOGIN_ONLY_ARGS)
        return d

    def test_login_only_problem(self):
        """login_only_... returns the right signal on error."""
        expected_error = "Sample Error - not for resale"
        d = Deferred()

        def verify(app_name, error_message, detailed_error):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        kgt = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgt(ARGS)
        self.mocker.throw(RegisterSampleException(expected_error))
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "_show_login_only_ui", self.fail)
        self.patch(client, "CredentialsFound", self.fail)
        self.patch(client, "CredentialsError", verify)
        client.login_to_get_credentials(*LOGIN_ONLY_ARGS)
        return d

    def test_clear_token(self):
        """Check that clear_token tries removing the correct token."""
        mockKeyringClass = self.mocker.replace("ubuntu_sso.keyring.Keyring")
        mockKeyringClass(APP_NAME)
        mockKeyring = self.mocker.mock()
        self.mocker.result(mockKeyring)
        mockKeyring.delete_ubuntusso_attr()
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        client.clear_token(APP_NAME)

    def test_clear_token_failed(self):
        """Check that clear_token fails correctly."""
        mockKeyringClass = self.mocker.replace("ubuntu_sso.keyring.Keyring")
        mockKeyringClass(APP_NAME)
        self.mocker.throw(self.mocker.mock())
        fake_logger = self.mocker.replace("ubuntu_sso.main.logger")
        fake_logger.exception(ANY, APP_NAME)
        self.mocker.replay()

        client = SSOCredentials(self.mocker.mock())
        client.clear_token(APP_NAME)

    def test_login_error_cb(self):
        """The login error callback should throw the signal."""
        d = Deferred()

        def verify(app_name, error_message, detailed_error):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        self.mocker.replay()
        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "CredentialsError", verify)

        client._login_error_cb(None, APP_NAME, 'some error')
        return d

    def test_show_login_or_register_ui_error(self):
        """An error happens when trying to register."""
        d = Deferred()
        mockgui = self.mocker.replace("ubuntu_sso.gui.UbuntuSSOClientGUI")
        mockgui(ARGS)
        self.mocker.throw(Exception())
        self.mocker.replay()

        def verify(app_name, msg, full_error):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "CredentialsError", verify)

        client._show_login_or_register_ui(APP_NAME, "http:tc_url", "help", 0)
        return d

    def test_login_success_cb_works(self):
        """Check that the right signal is sent."""
        expected_creds = TOKEN
        d = Deferred()
        kgc = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgc(ARGS)
        self.mocker.result(expected_creds)
        self.mocker.replay()

        def verify(app_name, credentials):
            """The actual test."""
            self.assertEqual(credentials, expected_creds)
            d.callback("ok")

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "CredentialsFound", verify)

        client._login_success_cb(None, APP_NAME, EMAIL)
        return d

    def test_login_success_cb_error(self):
        """An error happens when accessing the keyring."""
        d = Deferred()
        kgc = self.mocker.replace("ubuntu_sso.main.keyring_get_credentials")
        kgc(ARGS)
        self.mocker.throw(Exception())
        self.mocker.replay()

        def verify(app_name, msg, full_error):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "CredentialsError", verify)

        client._login_success_cb(None, APP_NAME, EMAIL)
        return d

    def test_auth_denied_cb(self):
        """When the user decides not to allow the registration or login."""
        d = Deferred()
        self.mocker.replay()

        def verify(app_name):
            """The actual test."""
            self.assertEqual(app_name, APP_NAME)
            d.callback("ok")

        client = SSOCredentials(self.mocker.mock())
        self.patch(client, "AuthorizationDenied", verify)

        client._login_auth_denied_cb(None, APP_NAME)
        return d


class CredentialsGUITestCase(TestCase):
    """login_or_register_to_get_credentials opens the proper GUI."""

    def setUp(self):
        """Init."""
        self.args = None
        self.kwargs = None

        idle_add = lambda f, *args, **kwargs: f(*args, **kwargs)
        self.patch(gobject, "idle_add", idle_add)
        self.patch(ubuntu_sso.main, 'keyring_get_credentials', lambda n: None)

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        logger.addHandler(self.memento)

        self.client = SSOCredentials(None)
        self.login_or_register = \
            self.client.login_or_register_to_get_credentials
        self.login_only = self.client.login_to_get_credentials

    def test_login_or_register_opens_proper_ui(self):
        """The proper GUI is created if the token is None."""

        class FakedUbuntuSSOClientGUI(object):
            """Fake a SSO GUI."""

            # Method should have "self" as first argument
            # pylint: disable=E0213

            def __init__(sself, *args, **kwargs):
                self.args = args
                self.kwargs = kwargs
                sself.connect = lambda *a: None

        self.patch(gui, "UbuntuSSOClientGUI", FakedUbuntuSSOClientGUI)
        self.login_or_register(*LOGIN_OR_REGISTER_ARGS)

        self.assertEqual(self.args, LOGIN_OR_REGISTER_GUI_ARGS)
        self.assertEqual(self.kwargs, {})

    def test_ui_exceptions_are_logged(self):
        """If GUI fails the exception is logged."""

        def fail(*args, **kwargs):
            """Raise an exception."""
            self.args = AssertionError((args, kwargs))
            # pylint: disable=E0702
            raise self.args

        self.patch(gui, "UbuntuSSOClientGUI", fail)
        self.login_or_register(*LOGIN_OR_REGISTER_ARGS)

        self.assertTrue(1, len(self.memento.records))
        self.assertTrue(self.memento.check(logging.ERROR,
                                           str(LOGIN_OR_REGISTER_GUI_ARGS)))
        exc_info = self.memento.records[0].exc_info
        self.assertIn(self.args, exc_info)

    def test_ui_signals_are_connected(self):
        """Signals from GUI are properly connected."""

        self.args = []

        class FakedUbuntuSSOClientGUI(object):
            """Fake a SSO GUI."""

            # Method should have "self" as first argument
            # pylint: disable=E0213

            def __init__(sself, *args, **kwargs):
                sself.connect = lambda *a: self.args.append(a)

        expected = [
            (gui.SIG_LOGIN_SUCCEEDED, self.client._login_success_cb),
            (gui.SIG_LOGIN_FAILED, self.client._login_error_cb),
            (gui.SIG_REGISTRATION_SUCCEEDED, self.client._login_success_cb),
            (gui.SIG_REGISTRATION_FAILED, self.client._login_error_cb),
            (gui.SIG_USER_CANCELATION, self.client._login_auth_denied_cb),
        ]
        self.patch(gui, "UbuntuSSOClientGUI", FakedUbuntuSSOClientGUI)
        self.login_or_register(*LOGIN_OR_REGISTER_ARGS)
        self.assertEqual(self.args, expected)

    def test_login_only_opens_proper_ui(self):
        """The proper GUI is created if the token is None."""

        class FakedUbuntuSSOClientGUI(object):
            """Fake a SSO GUI."""

            # Method should have "self" as first argument
            # pylint: disable=E0213

            def __init__(sself, *args, **kwargs):
                self.args = args
                self.kwargs = kwargs
                sself.connect = lambda *a: None

        self.patch(gui, "UbuntuSSOClientGUI", FakedUbuntuSSOClientGUI)
        self.login_only(*LOGIN_ONLY_ARGS)

        self.assertEqual(self.args, LOGIN_ONLY_GUI_ARGS)
        self.assertEqual(self.kwargs, {})

    def test_login_only_ui_exceptions_are_logged(self):
        """If GUI fails the exception is logged."""

        def fail(*args, **kwargs):
            """Raise an exception."""
            self.args = AssertionError((args, kwargs))
            # pylint: disable=E0702
            raise self.args

        self.patch(gui, "UbuntuSSOClientGUI", fail)
        self.login_only(*LOGIN_ONLY_ARGS)

        self.assertTrue(1, len(self.memento.records))
        self.assertTrue(self.memento.check(logging.ERROR,
                                           str(LOGIN_ONLY_GUI_ARGS)))
        exc_info = self.memento.records[0].exc_info
        self.assertIn(self.args, exc_info)

    def test_login_only_ui_signals_are_connected(self):
        """Signals from GUI are properly connected."""

        self.args = []

        class FakedUbuntuSSOClientGUI(object):
            """Fake a SSO GUI."""

            # Method should have "self" as first argument
            # pylint: disable=E0213

            def __init__(sself, *args, **kwargs):
                sself.connect = lambda *a: self.args.append(a)

        expected = [
            (gui.SIG_LOGIN_SUCCEEDED, self.client._login_success_cb),
            (gui.SIG_LOGIN_FAILED, self.client._login_error_cb),
            (gui.SIG_REGISTRATION_SUCCEEDED, self.client._login_success_cb),
            (gui.SIG_REGISTRATION_FAILED, self.client._login_error_cb),
            (gui.SIG_USER_CANCELATION, self.client._login_auth_denied_cb),
        ]
        self.patch(gui, "UbuntuSSOClientGUI", FakedUbuntuSSOClientGUI)
        self.login_only(*LOGIN_ONLY_ARGS)
        self.assertEqual(self.args, expected)


class PingServerTestCase(TestCase):
    """On successful login/registration, the U1 server is pinged."""

    def setUp(self):
        """Init."""
        self.patch(ubuntu_sso.main, 'keyring_get_credentials', lambda a: TOKEN)
        self.calls = []
        self.args = None
        self.kwargs = None

        def fake_it(*args, **kwargs):
            """Fake a call."""
            self.args = args
            self.kwargs = kwargs
            return FakedResponse(code=200)

        self.patch(urllib2, 'urlopen', fake_it)

        self.client = SSOCredentials(None)

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        logger.addHandler(self.memento)

    def _patch(self, name):
        """Patch a method so when it's called its name is stored."""
        self.patch(self.client, name,
                   lambda *a, **kw: self.calls.append((name, a, kw)))

    def test_on_registration_successful_u1_server_is_pinged(self):
        """When a registration is successful, the PING_URL is pinged."""
        self.client._login_success_cb(None, U1_APP_NAME, EMAIL)

        self.assertEqual(len(self.args), 1)
        self.assertEqual(self.args[0].get_full_url(), PING_URL + EMAIL)
        self.assertEqual(self.kwargs, {})

    def test_on_registration_successful_no_server_is_pinged(self):
        """When a registration is successful, the PING_URL is not pinged."""
        self.client._login_success_cb(None, APP_NAME, EMAIL)

        self.assertEqual(self.args, None)
        self.assertEqual(self.kwargs, None)

    def test_ping_is_signed_with_credentials(self):
        """Ping to PING_URL is signed with user credentials."""
        result = self.client._ping_url(U1_APP_NAME, EMAIL, TOKEN)
        headers = self.args[0].headers
        self.assertIn('Authorization', headers)
        oauth_stuff = headers['Authorization']

        expected = 'oauth_consumer_key="xQ7xDAz", ' \
            'oauth_signature_method="HMAC-SHA1", oauth_version="1.0", ' \
            'oauth_token="GkInOfSMGwTXAUoVQwLUoPxElEEUdhsLVNTPhxHJDUIeHCPNEo' \
            '", oauth_signature="'
        self.assertIn(expected, oauth_stuff)
        self.assertEqual(result, 200)

    def test_ping_is_done_before_sending_signal(self):
        """Ping the server before sending any credential signal."""
        self._patch('_ping_url')
        self._patch('CredentialsFound')
        self._patch('CredentialsError')
        self._patch('AuthorizationDenied')

        self.client._login_success_cb(None, APP_NAME, EMAIL)

        self.assertEqual(len(self.calls), 2)
        self.assertEqual(self.calls[0], ('_ping_url',
                                         (APP_NAME, EMAIL, TOKEN), {}))
        self.assertEqual(self.calls[1], ('CredentialsFound',
                                         (APP_NAME, TOKEN), {}))

    def test_send_error_if_ping_failed(self):
        """Ping the server before sending any credential signal."""

        def fail(*args, **kwargs):
            """Raise an exception."""
            self.args = AssertionError((args, kwargs))
            # pylint: disable=E0702
            raise self.args

        self.patch(self.client, '_ping_url', fail)
        self._patch('CredentialsFound')
        self._patch('CredentialsError')
        self._patch('AuthorizationDenied')

        self.client._login_success_cb(None, APP_NAME, EMAIL)

        self.assertEqual(len(self.calls), 1)
        self.assertEqual(self.calls[0][0], 'CredentialsError')
        self.assertEqual(self.calls[0][1][0], APP_NAME)

    def test_credentials_are_not_stored_if_ping_failed(self):
        """Credentials are not stored if the ping fails."""

        def fail(*args, **kwargs):
            """Raise an exception."""
            self.args = AssertionError((args, kwargs))
            # pylint: disable=E0702
            raise self.args

        self.patch(self.client, '_ping_url', fail)
        self._patch('clear_token')

        self.client._login_success_cb(None, APP_NAME, EMAIL)

        self.assertEqual(len(self.calls), 1)
        self.assertEqual(self.calls[0][0], 'clear_token')
        self.assertEqual(self.calls[0][1][0], APP_NAME)


class EnvironOverridesTestCase(TestCase):
    """Some URLs can be set from the environment for testing/QA purposes."""

    def test_override_ping_url(self):
        """The ping url can be set from the environ via USSOC_PING_URL."""
        fake_url = 'this is not really a URL'
        old_url = os.environ.get('USSOC_PING_URL')
        os.environ['USSOC_PING_URL'] = fake_url
        try:
            creds = SSOCredentials(None)
            self.assertEqual(creds.ping_url, fake_url)
        finally:
            if old_url:
                os.environ['USSOC_PING_URL'] = old_url
            else:
                del os.environ['USSOC_PING_URL']

    def test_no_override_ping_url(self):
        """If the environ is unset, the default ping url is used."""
        creds = SSOCredentials(None)
        self.assertEqual(creds.ping_url, PING_URL)

    def test_override_service_url(self):
        """The service url can be set from the env var USSOC_SERVICE_URL."""
        fake_url = 'this is not really a URL'
        old_url = os.environ.get('USSOC_SERVICE_URL')
        os.environ['USSOC_SERVICE_URL'] = fake_url
        try:
            proc = SSOLoginProcessor(sso_service_class=FakedSSOServer)
            self.assertEqual(proc.service_url, fake_url)
        finally:
            if old_url:
                os.environ['USSOC_SERVICE_URL'] = old_url
            else:
                del os.environ['USSOC_SERVICE_URL']

    def test_no_override_service_url(self):
        """If the environ is unset, the default service url is used."""
        proc = SSOLoginProcessor(sso_service_class=FakedSSOServer)
        self.assertEqual(proc.service_url, SERVICE_URL)
