/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.installers;

import gov.nasa.worldwind.formats.rpf.RPFCrawler;
import gov.nasa.worldwind.formats.rpf.RPFDataSeries;
import gov.nasa.worldwind.formats.rpf.RPFFrameFilename;
import gov.nasa.worldwind.formats.rpf.RPFFrameProperty;
import gov.nasa.worldwind.servers.tools.WaveletEncodeRPFs;
import gov.nasa.worldwind.servers.tools.WaveletEncoderListener;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URL;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.BevelBorder;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 *
 * @author  brownrigg
 * @version $Id$
 */
public class RPFInstaller extends javax.swing.JFrame {
    
    /** Creates new form RPFInstaller */
    public RPFInstaller() {
        initComponents();
    }
    
    //
    // Verifies that a WMS is installed on this system where we expect it to be.
    // The convention for TerraRaptor is that it resides in the same directory as
    // this installer.
    //
    private void verifyWmsLocation() {
        try {
            URL myUrl = RPFInstaller.class.getClassLoader().getResource("WEB-INF");
            if (myUrl == null) 
                throw new Exception("Could not locate WMS's WEB-INF directory");
            
            this.wmsDir = new File(myUrl.toURI()); 
            this.wmsConfigFile = new File(wmsDir.getAbsolutePath() + File.separator + "config.xml");
            if (!this.wmsConfigFile.exists()) 
                throw new Exception("Could not find WMS's config.xml as expected");

            if (!wmsConfigFile.canWrite()) 
                throw new Exception("WMS installation found; user does not have privileges to modify its configuration");
                
            loadWMSConfigDoc();
            showWMSLayers();

        } catch (Exception ex) {
            JOptionPane.showMessageDialog(this, ex.getMessage(), "WMS Probe Error", JOptionPane.ERROR_MESSAGE);
            System.exit(1);
        }
    }
        
    //
    // Loads the WMS's config.xml file into an XML Document.
    //
    private void loadWMSConfigDoc() throws Exception {
        DocumentBuilderFactory docfac = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = docfac.newDocumentBuilder();
        this.wmsConfigDoc = builder.parse(wmsConfigFile);
    }
    
    private void showWMSLayers() {
        try {
            numWmsGroups = 0;
            numWmsLayers = 0;
            groupNames.removeAllElements();
            
            DefaultMutableTreeNode treeRoot = new DefaultMutableTreeNode();

            XPathFactory xpfac = XPathFactory.newInstance();
            XPath xpath = xpfac.newXPath();

            Node rootNode = (Node) xpath.evaluate(WMSROOT, wmsConfigDoc, XPathConstants.NODE);
            NodeList nlist = (NodeList) xpath.evaluate(MAPSOURCE, rootNode, XPathConstants.NODESET);
            populateWmsTree(xpath, nlist, treeRoot);
            
            wmsLayersTree = new JTree(treeRoot);
            wmsLayersTree.setRootVisible(false);
            wmsLayersTree.setShowsRootHandles(true);
            wmsLayersTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
            renderer.setOpenIcon(null);
            renderer.setLeafIcon(null);
            wmsLayersTree.setCellRenderer(renderer);

            preExpandWmsTree(new TreePath(treeRoot));
            wmsLayersPanel.getViewport().setView(wmsLayersTree);
            wmsLayersPanel.validate();
            
            // fix up the JLabels...
            numWmsGroupsLbl.setText(Integer.toString(numWmsGroups));
            numWmsLayersLbl.setText(Integer.toString(numWmsLayers));
            
            // update the Group-combobox to reflect available groupnames...
            updateGroupsCombobox();
        } catch (Exception ex) {
        //??????????
        }
    }
    
    private void populateWmsTree(XPath xpath, NodeList nlist, DefaultMutableTreeNode treeBranch) throws Exception {
        for (int i = 0; i < nlist.getLength(); i++) {
            Node docNode = nlist.item(i);
            WmsLayerNode wmsLayer = new WmsLayerNode();
            wmsLayer.node = docNode;
            DefaultMutableTreeNode treeNode = new DefaultMutableTreeNode(wmsLayer);
            String label = xpath.evaluate(MAPSOURCE_NAME, docNode);
            if ("".equals(label)) {
                label = xpath.evaluate(MAPSOURCE_TITLE, docNode);
                ++numWmsGroups;
                groupNames.add(label);
                wmsLayer.isLeaf = false;
            } else {
                ++numWmsLayers;
                wmsLayer.isLeaf = true;
            }
            wmsLayer.text = label;
            treeBranch.add(treeNode);
            // walk any children...
            NodeList childList = (NodeList) xpath.evaluate(MAPSOURCE, docNode, XPathConstants.NODESET);
            populateWmsTree(xpath, childList, treeNode);
        }
    }

    private void preExpandWmsTree(TreePath p) {
        wmsLayersTree.expandPath(p);
        DefaultMutableTreeNode currNode = (DefaultMutableTreeNode) p.getLastPathComponent();
        int numChildren = currNode.getChildCount();
        for (int i = 0; i < numChildren; i++) {
            TreePath newPath = p.pathByAddingChild(currNode.getChildAt(i));
            preExpandWmsTree(newPath);
        }
    }
    
    private void updateGroupsCombobox() {
        DefaultComboBoxModel model = new DefaultComboBoxModel();
        model.addElement(COMBO_PROMPT);
        Iterator<String> iter = groupNames.iterator();
        while (iter.hasNext()) {
            model.addElement(iter.next());
        }
        groupsCombo.setModel(model);
    }

    private void testRpfRoot(File rootDir) {
        this.rpfSeries = new HashSet<Object>(10);
        final RPFCrawler crawler = new RPFCrawler();
        final File rpfDirTmp = rootDir;

        // quick&dirty make a progress dialog...
        final JDialog progress = new JDialog(this, "RPF Scan", true);
        JPanel dialogPanel = new JPanel();
        dialogPanel.setLayout(new BorderLayout());

        JPanel p = new JPanel();
        p.setLayout(new BoxLayout(p, BoxLayout.Y_AXIS));
        p.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createBevelBorder(BevelBorder.LOWERED),
                BorderFactory.createEmptyBorder(5,5,5,5)));

        JLabel l = new JLabel("Searching for RPF files");
        l.setAlignmentX(Component.CENTER_ALIGNMENT);
        p.add(l);
        p.add(Box.createRigidArea(new Dimension(0,10)));
        JProgressBar progressBar = new JProgressBar(JProgressBar.HORIZONTAL);
        progressBar.setIndeterminate(true);
        progressBar.setStringPainted(false);
        p.add(progressBar);
        JPanel p2 = new JPanel();
        p2.setLayout(new GridLayout(1,2));
        l = new JLabel("Files found: ", JLabel.RIGHT);
        p2.add(l);
        final JLabel progressText = new JLabel("", JLabel.LEFT);
        progressText.setAlignmentX(0f);
        p2.add(progressText);
        p.add(p2);
        dialogPanel.add(p, BorderLayout.CENTER);

        p = new JPanel();  p.setLayout(new FlowLayout(FlowLayout.CENTER));
        final JButton cancel = new JButton("Cancel");
        cancel.addActionListener( new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                crawler.stop();
                progress.dispose();
            }
        });
        p.add(cancel);
        dialogPanel.add(p, BorderLayout.SOUTH);
        progress.setContentPane(dialogPanel);
        progress.pack();
        progress.setSize(300, 150);
        progress.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        crawler.start(rpfDirTmp,
            new RPFCrawler.RPFGrouper(RPFFrameProperty.DATA_SERIES)
            {
                int numFilesFound = 0;
                public void addToGroup(Object groupKey, File rpfFile, RPFFrameFilename rpfFrameFilename)
                {
                    rpfSeries.add(groupKey);
                    ++numFilesFound;
                    progressText.setText(Integer.toString(numFilesFound));
                }

                public void finished()
                {
                    System.out.println("size of HashSet: " + rpfSeries.size());
                    rpfLayers = new RpfLayer[rpfSeries.size()];
                    int i = 0;
                    RPFInstaller.this.dataSeriesPanel.removeAll();
                    RPFInstaller.this.numDataSeriesLbl.setText(Integer.toString(rpfSeries.size()));
                    RPFInstaller.this.numRpfFilesLbl.setText(Integer.toString(numFilesFound));

                    if (rpfSeries.size() > 0) {
                        RPFInstaller.this.rpfDir = rpfDirTmp;
                        RPFInstaller.this.rpfDirFld.setText(rpfDirTmp.toString());
                        RPFInstaller.this.rpfDirFld.setBackground(GO);

                        // build out UI for each layer...
                       Iterator<Object> iter = rpfSeries.iterator();
                        while (iter.hasNext()) {
                            String seriesCode = (String) iter.next();
                            RPFDataSeries rpf = RPFDataSeries.dataSeriesFor(seriesCode);
                            rpfLayers[i] = new RpfLayer();
                            rpfLayers[i].seriesCode = seriesCode;
                            rpfLayers[i].seriesLabel = rpf.scaleOrResolution + " " + rpf.dataSeries;
                            rpfLayers[i].checkbox = new JCheckBox();
                            rpfLayers[i].checkbox.setSelected(true);
                            rpfLayers[i].textfield = new JTextField(rpfLayers[i].seriesLabel, 30);
                            rpfLayers[i].isAlreadyInstalled = false;

                            JPanel p = new JPanel();
                            p.setLayout(new FlowLayout(FlowLayout.LEFT));
                            p.add(rpfLayers[i].checkbox);
                            p.add(rpfLayers[i].textfield);
                            RPFInstaller.this.dataSeriesPanel.add(p);
                            ++i;
                        }
                        RPFInstaller.this.dataSeriesPanel.add(new Box.Filler(new Dimension(0, 0), 
                                new Dimension(0, 0),
                                new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE)));
                        RPFInstaller.this.installBtn.setEnabled(true);
                    }
                    else {
                        RPFInstaller.this.rpfDir = null;
                        RPFInstaller.this.rpfDirFld.setText(rpfDirTmp.toString());
                        RPFInstaller.this.rpfDirFld.setBackground(NOGO);
                        RPFInstaller.this.installBtn.setEnabled(false);
                        JOptionPane.showMessageDialog(progress, "No RPF files found!");                        
                    } 

                    RPFInstaller.this.dataSeriesPanel.repaint();
                    RPFInstaller.this.dataSeriesPanel.getParent().validate();
                    progress.setCursor(null);
                    progress.dispose();
                }
            }, false);

        progress.setVisible(true); 
    }
            

    //
    // Adds any new <MapSource> configuration to WMS's config.xml Document.
    //
    private void fixUpWMSConfig() throws Exception {
        
        // get the group name from the combo box...
        String groupName = (String) groupsCombo.getSelectedItem();

        try {
            XPathFactory xpfac = XPathFactory.newInstance();
            XPath xpath = xpfac.newXPath();

            // find the Document node in which to install incoming layers (may require a new grouping layer)...
            Node groupNode = null;
            Node rootNode = (Node) xpath.evaluate(WMSROOT, wmsConfigDoc, XPathConstants.NODE);
            NodeList nlist = (NodeList) xpath.evaluate(MAPSOURCE, rootNode, XPathConstants.NODESET);
            for (int i = 0; i < nlist.getLength(); i++) {
                // looping over installed groups...
                Node n = nlist.item(i);
                String name = xpath.evaluate(MAPSOURCE_NAME, n);
                String title = xpath.evaluate(MAPSOURCE_TITLE, n);
                
                if ("".equals(name) && groupName.equals(title)) {
                    // verify that the incoming layers do not conflict with existing layers...
                    for (int k=0; k<rpfLayers.length; k++)
                        rpfLayers[k].isAlreadyInstalled = false; // assume so unless we prove otherwise
                        
                    NodeList layerNodes = (NodeList) xpath.evaluate(MAPSOURCE, n, XPathConstants.NODESET);
                    for (int j=0; j<layerNodes.getLength(); j++) {
                        Node n2 = (Node) layerNodes.item(j);
                        String n2Name = xpath.evaluate(MAPSOURCE_NAME, n2);
                        
                        for (int k=0; k<rpfLayers.length; k++) {
                            // looping over layers to be imported...
                            if (!rpfLayers[k].checkbox.isSelected()) continue;
                            if (rpfLayers[k].textfield.getText().equalsIgnoreCase(n2Name)) {
                                // two in this group with same name; had better point to same root-dir
                                String n2RootDir = xpath.evaluate(MAPSOURCE_ROOTDIR, n2);
                                if (!n2RootDir.equals(rpfDir.getAbsolutePath())) {
                                    String msg = "Imported layer \"" + rpfLayers[k].textfield.getText() +
                                            "\" conflicts with existing layer\n" +
                                            " Layers with same name but differing source directories are not permitted.";                               
                                    JOptionPane.showMessageDialog(this, msg, "Layer Uniqueness Error", JOptionPane.ERROR_MESSAGE);
                                    return;
                                }
                                rpfLayers[k].isAlreadyInstalled = true;
                            }
                        }
                    }
                    
                    // incoming layer is unique, this is our grouping node...
                    groupNode = n;
                    break;
                }
            }
            
            if (groupNode == null) {
                // layers are to be imported into new grouping layer...
                Element group = wmsConfigDoc.createElement(MAPSOURCE_TAG);
                group.setAttribute(MSTITLE_ATTR, groupName);
                rootNode.appendChild(group);
                groupNode = group;
            }

            // at this point, we can get down to installing the imported layers into our Document...
            for (int i = 0; i < rpfLayers.length; i++) {
                if (!rpfLayers[i].checkbox.isSelected() || rpfLayers[i].isAlreadyInstalled) continue;
                String layerName = rpfLayers[i].textfield.getText();
                
                Element mapSource = wmsConfigDoc.createElement(MAPSOURCE_TAG);
                mapSource.setAttribute(MSNAME_ATTR, layerName);
                mapSource.setAttribute(MSTITLE_ATTR, rpfLayers[i].textfield.getText().replace(",", "_"));

                Element e = wmsConfigDoc.createElement(ROOTDIR_TAG);
                e.appendChild(wmsConfigDoc.createTextNode(rpfDir.getAbsolutePath()));
                mapSource.appendChild(e);

                e = wmsConfigDoc.createElement(CLASS_TAG);
                e.appendChild(wmsConfigDoc.createTextNode(rpfGenClassName));
                mapSource.appendChild(e);

                e = wmsConfigDoc.createElement(PROP_TAG);
                e.setAttribute("name", PROPDATASERIES_ATTR);
                e.setAttribute("value", rpfLayers[i].seriesCode);
                mapSource.appendChild(e);
                
                groupNode.appendChild(mapSource);
            }

            // reflect the import layers in the UI...            
            showWMSLayers();
            isConfigModified = true;

        } catch (Exception ex) {
            String errMsg = "Failed to install layers into WMS's config.xml: " + ex.toString();
            JOptionPane.showMessageDialog(this, errMsg, "Configuration Error", JOptionPane.ERROR_MESSAGE);
            throw ex;
        }
    }

    //
    // Write the amended config.xml Document, saving a copy of the old one first.
    //
    private void writeNewConfig() {
        try {
            String cfgFile = wmsConfigFile.getAbsoluteFile().toString();
            File newConfig = new File(cfgFile);
            File saveConfig = new File(cfgFile + "_SAVE");
            wmsConfigFile.renameTo(saveConfig);
            FileOutputStream out = new FileOutputStream(newConfig);

            // Use a Transformer for output
            TransformerFactory tFactory =
                    TransformerFactory.newInstance();
            Transformer transformer = tFactory.newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");

            DOMSource source = new DOMSource(wmsConfigDoc);
            StreamResult result = new StreamResult(out);
            transformer.transform(source, result);

            String msg = "Installation Completed\nPlease restart the WMS to activate new Layers";
            JOptionPane.showMessageDialog(this, msg, "Success", JOptionPane.INFORMATION_MESSAGE);                

        } catch (Exception ex) {
            String errMsg = "Failed to save new WMS config.xml: " + ex.toString();
            JOptionPane.showMessageDialog(this, errMsg, "Configuration Error", JOptionPane.ERROR_MESSAGE);
        }
    }
    
    private void generateWaveletEncodings()  {
        final JDialog progress = new JDialog(this, "Encoding framefiles", true);
        JPanel dialogPanel = new JPanel();
        dialogPanel.setLayout(new BorderLayout());

        JPanel p = new JPanel();
        p.setLayout(new BoxLayout(p, BoxLayout.Y_AXIS));
        p.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createBevelBorder(BevelBorder.LOWERED),
                BorderFactory.createEmptyBorder(5, 5, 5, 5)));

        JPanel p2 = new JPanel();
        p2.setLayout(new GridLayout(3,1));
        JLabel l = new JLabel("Progress:", JLabel.CENTER);
        p2.add(l);
        final JLabel layerText = new JLabel("", JLabel.CENTER);
        layerText.setAlignmentX(0f);
        p2.add(layerText);
        final JLabel progressText = new JLabel("", JLabel.CENTER);
        progressText.setAlignmentX(0f);
        progressText.setForeground(Color.BLUE);
        p2.add(progressText);
        p.add(p2);
        dialogPanel.add(p, BorderLayout.CENTER);

        p = new JPanel();
        p.setLayout(new FlowLayout(FlowLayout.CENTER));
        final JButton cancel = new JButton("Cancel");
        p.add(cancel);
        dialogPanel.add(p, BorderLayout.SOUTH);
        progress.setContentPane(dialogPanel);
        progress.pack();
        progress.setSize(300, 150);
        progress.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

        final Thread encoderThread = new Thread() {

            public void run() {

                final Runnable UIThread = new Runnable() {
                    public void run() {
                        progressText.setText(msg);
                    }
                };

                final WaveletEncoderListener listener = new WaveletEncoderListener() {
                    public void status(String status) {
                        msg = status;
                        SwingUtilities.invokeLater(UIThread);
                    }
                };

                cancel.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent evt) {
                        stopEncoding();
                    }
                });
                
                try {
                    String outDir = rpfDir.getAbsolutePath();
                    for (int i = 0; i < rpfLayers.length; i++) {
                        if (cancelled.get()) {
                            break;
                        }
                        try {
                            if (!rpfLayers[i].checkbox.isSelected()) {
                                continue;
                            }
                            layerText.setText("encoding layer: " + rpfLayers[i].textfield.getText());
                            encoder = new WaveletEncodeRPFs(RPFDataSeries.dataSeriesFor(rpfLayers[i].seriesCode));
                            encoder.addListener(listener);
                            encoder.generateWaveletFiles(rpfDir.getAbsolutePath(), outDir);
                        } catch (Exception ex) {
                            Object[] options = {"Continue", "cancel"};
                            String msg = "Failed to generate wavelet encodings for series " + rpfLayers[i].seriesLabel +
                                    "\n   reason: " + ex.toString();
                            int answer = JOptionPane.showOptionDialog(RPFInstaller.this, msg, "Error",
                                    JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE, null, options, options[0]);
                            if (answer == 1) {
                                break;
                            }
                        }
                    }

                    progress.setVisible(false);
                    progress.dispose();

                    if (!cancelled.get()) {
                        SwingUtilities.invokeAndWait(new Runnable() {
                            public void run() {
                                try {
                                    fixUpWMSConfig();
                                } catch (Exception ex) {
                                    String msg = "Unable to complete installation:\n" + ex.getMessage();
                                    JOptionPane.showMessageDialog(RPFInstaller.this, msg, "Install Error", JOptionPane.ERROR_MESSAGE);
                                }
                            }
                        });
                    }
                } catch (Exception ex) {
                    String msg = "Unable to complete installation:\n" + ex.getMessage();
                    JOptionPane.showMessageDialog(RPFInstaller.this, msg, "Install Error", JOptionPane.ERROR_MESSAGE);
                }

            }

            public void stopEncoding() {
                // the sequence is important here...
                cancelled.set(true);
                if (encoder != null)
                    encoder.stopEncoding();
            }

            private WaveletEncodeRPFs encoder;
            private AtomicBoolean cancelled = new AtomicBoolean(false);
            private String msg;
        };

        encoderThread.start();
        progress.setVisible(true);
    }
                
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jPanel9 = new javax.swing.JPanel();
        jPanel1 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        numDataSeriesLbl = new javax.swing.JLabel();
        jLabel3 = new javax.swing.JLabel();
        numRpfFilesLbl = new javax.swing.JLabel();
        jLabel5 = new javax.swing.JLabel();
        rpfDirFld = new javax.swing.JTextField();
        browseBtn = new javax.swing.JButton();
        jLabel6 = new javax.swing.JLabel();
        groupsCombo = new javax.swing.JComboBox();
        jScrollPane1 = new javax.swing.JScrollPane();
        dataSeriesPanel = new javax.swing.JPanel();
        jPanel2 = new javax.swing.JPanel();
        installBtn = new javax.swing.JButton();
        jPanel3 = new javax.swing.JPanel();
        wmsLayersPanel = new javax.swing.JScrollPane();
        numWmsGroupsLbl = new javax.swing.JLabel();
        jLabel8 = new javax.swing.JLabel();
        numWmsLayersLbl = new javax.swing.JLabel();
        jLabel10 = new javax.swing.JLabel();
        jPanel10 = new javax.swing.JPanel();
        closeBtn = new javax.swing.JButton();

        org.jdesktop.layout.GroupLayout jPanel9Layout = new org.jdesktop.layout.GroupLayout(jPanel9);
        jPanel9.setLayout(jPanel9Layout);
        jPanel9Layout.setHorizontalGroup(
            jPanel9Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(0, 100, Short.MAX_VALUE)
        );
        jPanel9Layout.setVerticalGroup(
            jPanel9Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(0, 100, Short.MAX_VALUE)
        );

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("WMS RPF-Data Installer");
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                windowClosingHandler(evt);
            }
        });

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("Install Files"));

        jLabel1.setText("From folder:");

        numDataSeriesLbl.setForeground(new java.awt.Color(0, 0, 255));
        numDataSeriesLbl.setText("0");

        jLabel3.setForeground(new java.awt.Color(0, 0, 255));
        jLabel3.setText("data series,");

        numRpfFilesLbl.setForeground(new java.awt.Color(0, 0, 255));
        numRpfFilesLbl.setText("0");

        jLabel5.setForeground(new java.awt.Color(0, 0, 255));
        jLabel5.setText("files");

        rpfDirFld.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                rpfDirFldActionPerformed(evt);
            }
        });

        browseBtn.setText("browse");
        browseBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                browseBtnActionPerformed(evt);
            }
        });

        jLabel6.setText("Group name:");

        groupsCombo.setEditable(true);
        groupsCombo.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                groupsComboActionPerformed(evt);
            }
        });

        dataSeriesPanel.setLayout(new javax.swing.BoxLayout(dataSeriesPanel, javax.swing.BoxLayout.Y_AXIS));
        jScrollPane1.setViewportView(dataSeriesPanel);

        org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 414, Short.MAX_VALUE)
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel1Layout.createSequentialGroup()
                        .add(numDataSeriesLbl)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jLabel3)
                        .add(12, 12, 12)
                        .add(numRpfFilesLbl)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jLabel5))
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel1Layout.createSequentialGroup()
                        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                            .add(jLabel1)
                            .add(jLabel6))
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel1Layout.createSequentialGroup()
                                .add(rpfDirFld, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 259, Short.MAX_VALUE)
                                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                                .add(browseBtn))
                            .add(groupsCombo, 0, 318, Short.MAX_VALUE))))
                .addContainerGap())
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1Layout.createSequentialGroup()
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(browseBtn)
                    .add(jLabel1)
                    .add(rpfDirFld, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(jLabel6)
                    .add(groupsCombo, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(numDataSeriesLbl)
                    .add(jLabel3)
                    .add(numRpfFilesLbl)
                    .add(jLabel5))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 312, Short.MAX_VALUE)
                .addContainerGap())
        );

        installBtn.setText("Install >>");
        installBtn.setEnabled(false);
        installBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                installAction(evt);
            }
        });

        org.jdesktop.layout.GroupLayout jPanel2Layout = new org.jdesktop.layout.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .add(installBtn)
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel2Layout.createSequentialGroup()
                .add(87, 87, 87)
                .add(installBtn)
                .addContainerGap(330, Short.MAX_VALUE))
        );

        jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder("WMS"));

        numWmsGroupsLbl.setForeground(new java.awt.Color(0, 0, 255));
        numWmsGroupsLbl.setText("0");

        jLabel8.setForeground(new java.awt.Color(0, 0, 255));
        jLabel8.setText("groups, ");

        numWmsLayersLbl.setForeground(new java.awt.Color(0, 0, 255));
        numWmsLayersLbl.setText("0");

        jLabel10.setForeground(new java.awt.Color(0, 0, 255));
        jLabel10.setText("layers");

        org.jdesktop.layout.GroupLayout jPanel3Layout = new org.jdesktop.layout.GroupLayout(jPanel3);
        jPanel3.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
            jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(wmsLayersPanel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 310, Short.MAX_VALUE)
                    .add(jPanel3Layout.createSequentialGroup()
                        .add(numWmsGroupsLbl)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jLabel8)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(numWmsLayersLbl)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                        .add(jLabel10)))
                .addContainerGap())
        );
        jPanel3Layout.setVerticalGroup(
            jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel3Layout.createSequentialGroup()
                .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(numWmsGroupsLbl)
                    .add(jLabel8)
                    .add(numWmsLayersLbl)
                    .add(jLabel10))
                .add(6, 6, 6)
                .add(wmsLayersPanel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 378, Short.MAX_VALUE)
                .addContainerGap())
        );

        closeBtn.setText("Close");
        closeBtn.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        closeBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                closeBtnActionPerformed(evt);
            }
        });

        org.jdesktop.layout.GroupLayout jPanel10Layout = new org.jdesktop.layout.GroupLayout(jPanel10);
        jPanel10.setLayout(jPanel10Layout);
        jPanel10Layout.setHorizontalGroup(
            jPanel10Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel10Layout.createSequentialGroup()
                .addContainerGap(872, Short.MAX_VALUE)
                .add(closeBtn)
                .addContainerGap())
        );
        jPanel10Layout.setVerticalGroup(
            jPanel10Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel10Layout.createSequentialGroup()
                .addContainerGap()
                .add(closeBtn)
                .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel3, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addContainerGap())
            .add(org.jdesktop.layout.GroupLayout.TRAILING, jPanel10, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .addContainerGap()
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel3, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel10, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void browseBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_browseBtnActionPerformed
        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(new RPFInstaller.DirPassFilter());
        chooser.setDialogType(JFileChooser.CUSTOM_DIALOG);
        chooser.setApproveButtonText("Select");
        chooser.setDialogTitle("Identify directory/folder");
        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        chooser.setMultiSelectionEnabled(false);

        int status = chooser.showDialog(this, null);
        if (status == JFileChooser.CANCEL_OPTION)
            return;
        
        testRpfRoot(chooser.getSelectedFile());
}//GEN-LAST:event_browseBtnActionPerformed

    private void rpfDirFldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_rpfDirFldActionPerformed
        // TODO add your handling code here:
}//GEN-LAST:event_rpfDirFldActionPerformed

    private void groupsComboActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_groupsComboActionPerformed
        // TODO add your handling code here:
}//GEN-LAST:event_groupsComboActionPerformed

    private void closeBtnActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_closeBtnActionPerformed
        if (isConfigModified) {
            Object[] options = {"Yes", "No", "Cancel"};
            String msg = "Save changes to WMS?";
            int answer = JOptionPane.showOptionDialog(this, msg, "Confirm Changes",
                JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE, null, options, options[1]);
            if (answer == 2) return;
            if (answer == 0)
                writeNewConfig();
        }
        System.exit(0);
}//GEN-LAST:event_closeBtnActionPerformed

    private void installAction(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_installAction
        // make certain user has entered a group-name...
        String groupName = (String) groupsCombo.getSelectedItem();
        if (COMBO_PROMPT.equals(groupName)) {
            JOptionPane.showMessageDialog(this, "Please enter a grouping name", "Need Group Name", JOptionPane.ERROR_MESSAGE);
            return;
        }
        generateWaveletEncodings();
    }//GEN-LAST:event_installAction

    private void windowClosingHandler(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_windowClosingHandler
        closeBtnActionPerformed(null);
    }//GEN-LAST:event_windowClosingHandler
    
    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                try {
                    // Set System L&F
                    UIManager.setLookAndFeel(
                            UIManager.getSystemLookAndFeelClassName());
                } catch (Exception e) { /* best effort */ }
                RPFInstaller This = new RPFInstaller(); 
                This.verifyWmsLocation();
                This.setVisible(true);
            }
        });
    }

    // ----------------------------------------------------------------------
    // A Swing FileFilter that passes only directories...
    private class DirPassFilter extends FileFilter {
        public boolean accept(File f) {
            return f.isDirectory();
        }
        public String getDescription() { return ""; }
    }

    // ---------------------------------------------------------------------
    // A convenient bundling of series-code and user-assignable label.
    private class RpfLayer {
        String seriesCode;
        String seriesLabel;
        JCheckBox checkbox;
        JTextField textfield;
        boolean isAlreadyInstalled;
    }
    
    // ---------------------------------------------------------------------
    // An Object used to populate nodes in the JTree of existing groups/layers
    private class WmsLayerNode {
        Node  node;
        String text;
        boolean isLeaf;
        public String toString() {
            return text;
        }
        public void setName(String name) { }
        public boolean isLeaf() { return isLeaf; }
    }
    
    private File wmsDir;
    private File wmsConfigFile;
    private Document wmsConfigDoc;
    private HashSet<Object> rpfSeries;
    private RpfLayer[] rpfLayers;
    private File rpfDir = null;
    private Vector<String> groupNames = new Vector<String>(10);
    private int numWmsGroups = 0;
    private int numWmsLayers = 0;
    private boolean isConfigModified = false;
 
    // we manage these components explicitly...
    private javax.swing.JTree wmsLayersTree = new javax.swing.JTree();
    
    private static Color NOGO = new Color(1f, .8f, .8f);
    private static Color GO = new Color(.8f, 1f, .8f);
    private static String rpfGenClassName = "gov.nasa.worldwind.servers.wms.generators.RPFGenerator";
    
    private static final String WMSROOT = "/wms-config";
    private static final String MAPSOURCE = "./mapsource";
    private static final String MAPSOURCE_NAME = "@name";
    private static final String MAPSOURCE_TITLE = "@title";
    private static final String MAPSOURCE_ROOTDIR = "./root-dir";

    private static final String MAPSOURCE_TAG = "mapsource";
    private static final String MSNAME_ATTR = "name";
    private static final String MSTITLE_ATTR = "title";
    private static final String ROOTDIR_TAG = "root-dir";
    private static final String CLASS_TAG = "class";
    private static final String PROP_TAG = "property";
    private static final String PROPWAVPREL_ATTR = "wavelet_preload_size";
    private static final String PROPWAVTHRESH_ATTR = "wavelet_image_threshold";
    private static final String PROPWAVROOTDIR_ATTR = "wavelet_encoding_root_dir";
    private static final String PROPDATASERIES_ATTR = "data_series";

    private static final String COMBO_PROMPT = "--enter a group name or select from the list --";
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton browseBtn;
    private javax.swing.JButton closeBtn;
    private javax.swing.JPanel dataSeriesPanel;
    private javax.swing.JComboBox groupsCombo;
    private javax.swing.JButton installBtn;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel10;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel10;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JPanel jPanel9;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JLabel numDataSeriesLbl;
    private javax.swing.JLabel numRpfFilesLbl;
    private javax.swing.JLabel numWmsGroupsLbl;
    private javax.swing.JLabel numWmsLayersLbl;
    private javax.swing.JTextField rpfDirFld;
    private javax.swing.JScrollPane wmsLayersPanel;
    // End of variables declaration//GEN-END:variables
    
}
