/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_return_account_from_pem.c
    \brief  This interface returns the uid, gids and poolindex, i.e. leaseid,
            using LCMAPS and takes the user credential in PEM format as input
    \author Martijn Steenbakkers for the EU DataGrid.

    This interface returns the uid, gids and poolindex, i.e. leaseid, using LCMAPS.
    As input it requires the user credential in PEM format
    -# lcmaps_return_account_from_pem: interface function
*/

#ifndef LCMAPS_RETURN_ACCOUNT_FROM_PEM_C
#define LCMAPS_RETURN_ACCOUNT_FROM_PEM_C

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>


/* LCMAPS includes */
#include "lcmaps.h"
#include "lcmaps_log.h"
#include "lcmaps_return_account_from_pem.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCMAPS_MAX_POLICIES 10
#define LCMAPS_LOG_FILE "/var/log/lcmaps-suexec.log"
/* #define LCMAPS_LOG_STRING "return_account_from_pem" */
#define LCMAPS_BUFSIZ 8192
#define LCMAPS_LOG_BUFSIZ 2048
#define LCMAPS_DEFAULT_DB_FILE "lcmaps.db"
#define LCMAPS_REQUIRED_DB_PATH LCMAPS_ETC_HOME

/* Flags to indicate which type of logging is required */
#define DO_USRLOG           ((unsigned short)0x0001)
#define DO_SYSLOG           ((unsigned short)0x0002)

/*
 * If the following is set, LCMAPS will not allow the environment variable
 * LCMAPS_DB_FILE to contain an absolute path.
 * For now I don't see cases were this is necessary, so unset it
 */
#define DONT_ALLOW_ABSOLUTE_LCMAPS_DB_FILE 0
/*
 * Also this should ideally happen in the calling application (glexec)
 * so unset it.
 */
#define CHECK_LCMAPS_DB_PERMISSIONS 0

/******************************************************************************
                       Define module specific variables
******************************************************************************/
#if 0
static FILE *           lcmaps_log_stream = NULL;      /* stream for LCMAPS logging */
#endif
#if 0
static char *           lcmaps_log_string = NULL;      /* (constant) log string to be prepended to log information */
#endif
static char *           lcmaps_log_file = NULL;

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
/*
static int           start_logging();
static int           log_line(int , char * , ...);
static int           log_a_string(int , char * , char *);
static int           end_logging();
*/
extern int lcmaps_tokenize( const char * command, char ** args, int * n, char * sep);


/******************************************************************************
Function:   lcmaps_return_account_from_pem_va
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The arguments of this function are passed as a variable argument list.

Parameters:
    narg:  the number of arguments that follow
    ap:    vararg list, that consists of
        - input:  The PEM-encoded string containing the user proxy
        - output: a structure that contains account information:
                uid, gids (primary and secondary) and the poolindex
        - input, optional: the mapcounter
Returns:
    0: success
    1: failure
******************************************************************************/
static int lcmaps_return_account_from_pem_va(
    int        narg,
    ...
)
{
    va_list                             ap;
    char *                              pem_string = NULL;        /* input variable */
    int                                 mapcounter = -1;         /* input variable */
    lcmaps_account_info_t *             plcmaps_account = NULL;  /* output variable */
    char *                              lcmaps_policy_string = NULL;
    char *                              lcmaps_policies[LCMAPS_MAX_POLICIES];
    int                                 lcmaps_npols = LCMAPS_MAX_POLICIES;
    char *                              lcmaps_db_file_env = NULL;
    char *                              lcmaps_db_file = NULL;
    int                                 rc = -1;
    uid_t                               uid = -1;
    gid_t *                             pgid_list = NULL;
    int                                 npgid = 0;
    gid_t *                             sgid_list = NULL;
    int                                 nsgid = 0;
    char *                              poolindex = NULL;
    int                                 jj = 0;

#if 0
    time_t                              my_clock;
    struct tm  *                        timestamp = NULL;
    size_t                              string_size;
    char *                              lcmaps_log_string_default = NULL;
#endif
#if CHECK_LCMAPS_DB_PERMISSIONS
    struct stat                         prg_info;   /* program info holder       */
    struct stat                         dir_info;   /* directory info holder     */
#endif

    /*
     * Take log file from environment variables.
     * If not defined take the default ones
     */
    lcmaps_log_file = getenv("LCMAPS_LOG_FILE");
    lcmaps_log_file = (lcmaps_log_file ? lcmaps_log_file : LCMAPS_LOG_FILE );

    /*
     * The other environment variables should in principle not be touched,
     * because reasonably defaults are provided by the LCMAPS core.
     * An exception is made for LCMAPS_LOG_STRING, because we want to enforce
     * that it is logged which interface is used.
     * For traceability also add a timestamp.
     * setenv with overwrite=0 only overwrites if the variable is non-existant
     */
#if 0
    time(&my_clock);
    /* timestamp = localtime(&my_clock); */
    timestamp = gmtime(&my_clock);
    string_size = strlen(LCMAPS_LOG_STRING) + strlen(": ") + 19 + 1;
    lcmaps_log_string_default = (char *) malloc(string_size * sizeof(char));

    snprintf(lcmaps_log_string_default, string_size, "%s: %04d-%02d-%02d.%02d:%02d:%02d",
        LCMAPS_LOG_STRING,
        timestamp->tm_year + 1900, timestamp->tm_mon + 1, timestamp->tm_mday,
        timestamp->tm_hour, timestamp->tm_min, timestamp->tm_sec);

    setenv("LCMAPS_LOG_STRING", lcmaps_log_string_default, 0);
    lcmaps_log_string = getenv("LCMAPS_LOG_STRING");

    if (lcmaps_log_string_default != NULL)
    {
        free(lcmaps_log_string_default);
        lcmaps_log_string_default = NULL;
    }
#endif

    /*
     * Set the lcmaps_db_file. If we're very paranoid we set DONT_ALLOW_ABSOLUTE_LCMAPS_DB_FILE, but normally
     * this is not needed, because the calling application (glexec) should take care that the
     * environment variables are set correctly
     */
#if DONT_ALLOW_ABSOLUTE_LCMAPS_DB_FILE
    /*
     * "Another exception is made for the lcmaps.db file: This interface is used by
     *  VO services (i.e. not site-controlled services), so the VO should not be able
     *  to determine which lcmaps.db file (or in the future a general lcmaps config file)
     *  they take."
     */
    if ( (lcmaps_db_file_env = getenv("LCMAPS_DB_FILE")) != NULL)
    {
        /*
         * OK, one tries to set the location of the lcmaps.db file.
         * We don't allow paths to be included in the filename. LCMAPS will add the
         * site-controlled path to it.
         * If a path is found: error
         * If no path is found: add the default path to it and performs various checks
         */
        if (strchr(lcmaps_db_file_env, '/') != NULL)
        {
            lcmaps_log(LOG_ERR, "%s: (LCMAPS_DB_FILE = %s)\n", __func__, lcmaps_db_file_env);
            lcmaps_log(LOG_ERR,
                "%s: Attempt to include path to lcmaps.db file by environment variable\n", __func__);
            lcmaps_log(LOG_ERR, "%s: This is not allowed for this lcmaps interface\n", __func__);
            return 1;
        }
    }
    else
    {
        lcmaps_db_file_env = LCMAPS_DEFAULT_DB_FILE;
    }
    /* prepend required path */
    lcmaps_db_file = (char *) malloc(strlen(LCMAPS_REQUIRED_DB_PATH) + 2 + strlen(lcmaps_db_file_env));
    if (lcmaps_db_file==NULL)	{
	lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	return 1;
    }
    sprintf(lcmaps_db_file, "%s/%s", LCMAPS_REQUIRED_DB_PATH, lcmaps_db_file_env);
#else /* DONT_ALLOW_ABSOLUTE_LCMAPS_DB_FILE */
    lcmaps_db_file_env = getenv("LCMAPS_DB_FILE");
    lcmaps_db_file_env = (lcmaps_db_file_env ? lcmaps_db_file_env : LCMAPS_DEFAULT_DB_FILE );
    lcmaps_db_file = strdup(lcmaps_db_file_env);
    if (lcmaps_db_file==NULL)	{
	lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	return 1;
    }
#endif /* DONT_ALLOW_ABSOLUTE_LCMAPS_DB_FILE */

    /*
     * Check the lcmaps_db_file.
     * Again, if we're very paranoid we set CHECK_LCMAPS_DB_PERMISSIONS, but normally
     * this is not needed, because the calling application (glexec) should do that.
     */
#if CHECK_LCMAPS_DB_PERMISSIONS
    /*
     * Check lcmaps.db file:
     * - Should be in a directory owned by root, which non-writable for others.
     * - Should be owned by root and non-writable for others.
     * If checks are ok reset environment variable
     */
    /*
     * Stat the directory of lcmaps.db and verify it is a directory, or error out.
     */
    if (((lstat(LCMAPS_REQUIRED_DB_PATH, &dir_info)) != 0) || !(S_ISDIR(dir_info.st_mode)))
    {
        lcmaps_log(LOG_ERR, "%s: cannot stat directory: (%s)\n", __func__, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is not owned by root
     */
    if ((dir_info.st_uid != 0) || (dir_info.st_gid != 0))
    {
        lcmaps_log(LOG_ERR, "%s: directory is not owned by root: (%s)\n", __func__, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the directory is writable by others.
     */
    if (dir_info.st_mode & S_IWOTH)
    {
        lcmaps_log(LOG_ERR, "%s: directory is writable by others: (%s)\n", __func__, LCMAPS_REQUIRED_DB_PATH);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if we cannot stat the lcmaps.db file.
     */
    if (((lstat(lcmaps_db_file, &prg_info)) != 0) || (S_ISLNK(prg_info.st_mode)))
    {
        lcmaps_log(LOG_ERR, "%s: cannot stat program: (%s)\n", __func__, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcmaps.db file is not owned by root
     */
    if ((prg_info.st_uid != 0) || (prg_info.st_gid != 0))
    {
        lcmaps_log(LOG_ERR, "%s: file is not owned by root: (%s)\n", __func__, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
    /*
     * Error out if the lcmaps.db file is writable by others.
     */
    if (prg_info.st_mode & S_IWOTH)
    {
        lcmaps_log(LOG_ERR, "%s: file is writable by others: (%s)\n", __func__, lcmaps_db_file);
        if (lcmaps_db_file) {free(lcmaps_db_file); lcmaps_db_file = NULL;}
        return 1;
    }
#endif /* CHECK_LCMAPS_DB_PERMISSIONS*/

    /* Everything's fine, so set (overwrite!) the environment variable. */
    setenv("LCMAPS_DB_FILE", lcmaps_db_file, 1);
    if (lcmaps_db_file)
    {
        free(lcmaps_db_file);
        lcmaps_db_file = NULL;
    }

    /* Parse arguments from va_list */
    va_start(ap, narg);
    if (narg == 3)
    {
        pem_string = va_arg(ap, char *);
        mapcounter = va_arg(ap, int);
        plcmaps_account = va_arg(ap, lcmaps_account_info_t *);
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: The number of arguments (%d) should be 3\n", __func__, narg);
        return 1;
    }
    va_end(ap);

#if 0
    if (start_logging())
    {
        fprintf(stderr, "%s: Cannot start logging\n", lcmaps_log_string);
        return 1;
    }
#endif

    /* Parse policy environment variable */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++) lcmaps_policies[jj] = 0;
    lcmaps_policy_string = getenv("LCMAPS_POLICY_NAME");
    if ((rc = lcmaps_tokenize(lcmaps_policy_string, lcmaps_policies, &lcmaps_npols, ":")) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: Cannot parse LCMAPS_POLICY_NAME environment variable, because\n", __func__);
        switch (rc)
        {
            case -1:
                lcmaps_log(LOG_ERR, "%s: of a malloc error\n", __func__);
                break;
            case -2:
                lcmaps_log(LOG_ERR, "%s: the policy list is too large (max = %d)\n",
                    __func__, LCMAPS_MAX_POLICIES);
                break;
            case -3:
                lcmaps_log(LOG_ERR, "%s: of a non-matching quote\n", __func__);
                break;
            case -4:
                lcmaps_log(LOG_ERR, "%s: invalid input\n", __func__);
                break;
            default:
                lcmaps_log(LOG_ERR, "%s: of an unknown error\n", __func__);
                break;
        }
        goto return_account_from_pem_error;
    }

    /* First initialize LCMAPS (used to use lcmaps_log_stream but that's always
     * NULL */
    if (lcmaps_init_and_log(NULL, DO_USRLOG|DO_SYSLOG))
    {
        lcmaps_log(LOG_ERR, "%s: LCMAPS initialization failure\n", __func__);
        goto return_account_from_pem_error;
    }


    /*
     * Now that we have the credential let's run (and terminate) LCMAPS !
     */
    rc = lcmaps_run_with_pem_and_return_account(
        NULL,
        pem_string,
        mapcounter,
        NULL,
        lcmaps_npols,
        lcmaps_policies,
        &uid,
        &pgid_list,
        &npgid,
        &sgid_list,
        &nsgid,
        &poolindex
    );
    if (rc != 0)
    {
        /* lcmaps_log (1, "LCMAPS failed to do mapping and return account information\n"); */
        lcmaps_log (LOG_ERR, "LCMAPS failed to do mapping and return account information\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_account_from_pem_error;
        }
        goto return_account_from_pem_error;
    }
    /*
     * Fill the lcmaps account information
     */
    rc = lcmaps_account_info_fill(
        &uid,
        &pgid_list,
        &npgid,
        &sgid_list,
        &nsgid,
        &poolindex,
        plcmaps_account
    );
    if (rc != 0)
    {
        lcmaps_log (LOG_ERR, "LCMAPS failed to do mapping and return account information\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_account_from_pem_error;
        }
        goto return_account_from_pem_error;
    }
    /* Now we're done with the poolindex (has been dupped into plcmaps_account),
     * so free it */
    free(poolindex);

    rc = lcmaps_term();
    if (rc)
    {
        lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
        goto return_account_from_pem_error;
    }

#if 0
    if (end_logging())
    {
        fprintf(stderr, "%s: Cannot cleanly end lcmaps logging\n", lcmaps_log_string);
        goto return_account_from_pem_error;
    }
#endif

    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }

    return 0;

 return_account_from_pem_error:
    /* Clean policies */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++)
    {
        if ((lcmaps_policies[jj]) != NULL)
        {
            free(lcmaps_policies[jj]);
            lcmaps_policies[jj] = NULL;
        }
    }
#if 0
    if (end_logging())
    {
        fprintf(stderr, "%s: Cannot cleanly end lcmaps logging\n", lcmaps_log_string);
    }
#endif

    return 1;
}

/******************************************************************************
Function:   lcmaps_return_account_from_pem
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.

Parameters:
    mapcounter:    :  The counter which will be added to the poolindex, effectively enabling
                      multiple account mappings
    pem_string:    :  The PEM-encoded string containing the user proxy
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_return_account_from_pem(
        char * pem_string,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
)
{
    return (lcmaps_return_account_from_pem_va(3, pem_string, mapcounter, plcmaps_account));
}

#endif /* LCMAPS_RETURN_ACCOUNT_FROM_PEM_C */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_return_account_from_pem.c,v $
    $Date: 2012-03-16 16:16:23 +0100 (Fri, 16 Mar 2012) $
    $Revision: 16177 $
    $Author: msalle $
******************************************************************************/
