#ifndef _ALBERTA_INTERN_H_
#define _ALBERTA_INTERN_H_
/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file: alberta_intern.h                                                   */
/*                                                                          */
/*                                                                          */
/* description: private header file of the ALBERTA package                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2005)                          */
/*  (c) by D. Koester (2002-2005)                                           */
/*--------------------------------------------------------------------------*/

#include <ctype.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <stdlib.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

/*--------------------------------------------------------------------------*/
/* information about size of leaf data, which will be hidden at child[1] and*/
/* pointers to functions which will be used for transferring such           */
/* information during refinement/coarsening                                 */
/*--------------------------------------------------------------------------*/
typedef struct leaf_data_info   LEAF_DATA_INFO;

struct leaf_data_info
{
  size_t           leaf_data_size;
  void            (*refine_leaf_data)(EL *parent, EL *child[2]);
  void            (*coarsen_leaf_data)(EL *parent, EL *child[2]);
};

/*--------------------------------------------------------------------------*/
/* interpolation of dof vectors during refinement                           */
/* using DOF_VEC_LIST structure                                             */
/*--------------------------------------------------------------------------*/

typedef struct dof_vec_list  DOF_VEC_LIST;

struct dof_vec_list
{
  int             size;                       /* current size of total list */
  void            **list;          /* total list, distributed to ones below */

  int             n_dof_int_vec, n_dof_dof_vec, n_dof_uchar_vec,
                  n_dof_schar_vec, n_dof_real_vec, n_dof_real_d_vec,
                  n_dof_ptr_vec, n_dof_matrix, n_dof_dowb_matrix;

  DOF_INT_VEC     **dof_int_vec;
  DOF_DOF_VEC     **dof_dof_vec;
  DOF_UCHAR_VEC   **dof_uchar_vec;
  DOF_SCHAR_VEC   **dof_schar_vec;
  DOF_REAL_VEC    **dof_real_vec;
  DOF_REAL_D_VEC  **dof_real_d_vec;
  DOF_PTR_VEC     **dof_ptr_vec;
  DOF_MATRIX      **dof_matrix;
  DOF_DOWB_MATRIX **dof_dowb_matrix;
};

/*--------------------------------------------------------------------------*/
/* the structure stored in the entry mem_info of mesh.                      */
/* It stores information for memory management and macro elements.          */
/*--------------------------------------------------------------------------*/
typedef struct mesh_mem_info    MESH_MEM_INFO;

struct mesh_mem_info
{
  void *dof_ptrs;
  void *dofs[N_NODE_TYPES];
  void *element;

  void *rc_list;
  void *real_d;

  DOF_VEC_LIST   *dvlist;

  void           *leaf_data;
  LEAF_DATA_INFO  leaf_data_info[1];

/*--------------------------------------------------------------------------*/
/* special entries reserved for the use of master/slave grids.              */
/*--------------------------------------------------------------------------*/

/* pointers on slave grids */
  MESH         *master;                           /* pointer to master mesh */
                                                  /* binding of master mesh */
  int         (*binding_method)(MESH *master, MACRO_EL *el, int subsimplex,
				void *data);
  DOF_PTR_VEC  *master_binding;        /* pointers to master mesh simplices */
  DOF_PTR_VEC  *slave_binding;          /* pointers to slave mesh simplices */

/* pointers on master grids */
  int           n_slaves;                     /* no. of slaves of this mesh */
  MESH        **slaves;                     /* vector of pointers to slaves */

/*---8<---------------------------------------------------------------------*/
/*--- These last two entries are not really for memory management.       ---*/
/*--- They describe the array used to store macro element coordinate     ---*/
/*--- information, which is filled in macro.c and read_mesh.c.           ---*/
/*--------------------------------------------------------------------->8---*/
  int   count;  
  REAL_D *coords;
};



/***   file memory.c   ******************************************************/
extern void      *AI_get_leaf_data(MESH *mesh);
extern DOF_ADMIN *AI_get_dof_admin(MESH *mesh, const char *name, 
			    const int n_dof[N_NODE_TYPES]);
extern void       AI_free_leaf_data(void *leaf_data, MESH *mesh);
extern void       AI_reactivate_dof(MESH *mesh, const EL *el);
extern DOF       *AI_get_dof_memory(MESH *mesh, int position);
extern void       AI_get_dof_list(MESH *mesh, int i);
extern void       AI_get_dof_ptr_list(MESH *mesh);
extern void       AI_fill_missing_dofs(MESH *mesh);
extern void       AI_advance_cookies_rec(MESH *mesh);

/***   file macro.c   *******************************************************/
extern S_CHAR AI_get_orientation(MACRO_EL *mel);

/***   file dof_admin.c   ***************************************************/
extern void  AI_add_dof_matrix_to_admin(DOF_MATRIX *, DOF_ADMIN *);
extern void  AI_add_dof_dowb_matrix_to_admin(DOF_DOWB_MATRIX *, DOF_ADMIN *);
extern void  AI_add_dof_int_vec_to_admin(DOF_INT_VEC *, DOF_ADMIN *);
extern void  AI_add_int_dof_vec_to_admin(DOF_DOF_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_dof_vec_to_admin(DOF_DOF_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_uchar_vec_to_admin(DOF_UCHAR_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_schar_vec_to_admin(DOF_SCHAR_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_real_vec_to_admin(DOF_REAL_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_real_d_vec_to_admin(DOF_REAL_D_VEC *, DOF_ADMIN *);
extern void  AI_add_dof_ptr_vec_to_admin(DOF_PTR_VEC *, DOF_ADMIN *);
extern void  AI_remove_dof_matrix_from_admin(DOF_MATRIX *);
extern void  AI_remove_dof_dowb_matrix_from_admin(DOF_DOWB_MATRIX *);
extern void  AI_remove_dof_int_vec_from_admin(DOF_INT_VEC *);
extern void  AI_remove_dof_dof_vec_from_admin(DOF_DOF_VEC *);
extern void  AI_remove_int_dof_vec_from_admin(DOF_DOF_VEC *);
extern void  AI_remove_dof_uchar_vec_from_admin(DOF_UCHAR_VEC *);
extern void  AI_remove_dof_schar_vec_from_admin(DOF_SCHAR_VEC *);
extern void  AI_remove_dof_real_vec_from_admin(DOF_REAL_VEC *);
extern void  AI_remove_dof_real_vec_from_admin(DOF_REAL_VEC *);
extern void  AI_remove_dof_real_d_vec_from_admin(DOF_REAL_D_VEC *);
extern void  AI_remove_dof_ptr_vec_from_admin(DOF_PTR_VEC *);

/***   file read_mesh.c   ***************************************************/
extern bool_t  AI_xdr_REAL(XDR *xdr, void *rp);
extern bool_t  AI_xdr_U_CHAR(XDR *xdr, void *ucp);
extern bool_t  AI_xdr_S_CHAR(XDR *xdr, void *cp);
extern bool_t  AI_xdr_DOF(XDR *xdr, void *dp);
extern XDR    *AI_xdr_open_file(const char *fn, enum xdr_op mode);
extern int     AI_xdr_close_file(XDR *xdr);

/***   file refine.c   ******************************************************/
extern void  AI_refine_fct_1d(const EL_INFO *el_info, void *data); 
extern void  AI_post_refine_1d(MESH *mesh);
extern int   AI_coarse_recursive_1d(MESH *mesh, EL *parent);

#if DIM_OF_WORLD > 1
extern void  AI_bisect_element_2d(MESH *mesh, EL *el, DOF *dof[3]); 
extern void  AI_bisect_patch_2d(MESH *mesh,RC_LIST_EL ref_list[],int n_neighs);
extern void  AI_coarse_patch_2d(MESH *mesh,RC_LIST_EL coarse_list[],
				int n_neigh);
extern void  AI_post_refine_2d(MESH *mesh);
#endif
extern DOF_VEC_LIST *AI_get_dof_vec_list(MESH *mesh);
extern void  AI_free_dof_vec_list(MESH *mesh);
extern void  AI_get_neigh_on_patch_3d(RC_LIST_EL ref_list[], int n_neigh,
				      int bound);

/***   file submesh.c   *****************************************************/
extern void AI_check_slavery(MESH *master);

/***   file traverse_nr.c ***************************************************/
extern void AI_test_traverse_nr(MESH *mesh, int level, FLAGS fill_flag);
extern void AI_update_elinfo_stack_3d(TRAVERSE_STACK *stack); 

/***   file traverse_r.c  ***************************************************/
extern void AI_test_traverse(MESH *mesh, int level, FLAGS fill_flag);
extern void AI_update_elinfo_3d(EL_INFO *elinfo);
extern void AI_run_traverse_hooks(MESH *mesh, const EL_INFO *el_info);

#ifdef __cplusplus
}
#endif

#endif /* !_ALBERTA_INTERN_H_ */
