/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * Playlist Editor
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>

#include "common.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/inputtext.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/button.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/button_list.h"
#include "xine-toolkit/browser.h"
#include "playlist.h"
#include "file_browser.h"
#include "menus.h"
#include "mrl_browser.h"
#include "panel.h"
#include "stream_infos.h"
#include "actions.h"
#include "event.h"
#include "mediamark.h"
#include "errors.h"
#include "oxine/oxine.h"

typedef enum {
  _W_playlist = 0,
  _W_list_title,
  _W_autoplay_buttons,
  _W_winput,
  _W_add,
  _W_load,
  _W_close,
  /* keep order */
  _W_move_up,
  _W_move_down,
  _W_play,
  _W_delete,
  _W_delete_all,
  _W_save,
  /* /keep order */
  _W_LAST
} _W_t;

struct xui_playlist_st {
  gui_new_window_t      nw;

  xitk_widget_t        *w[_W_LAST];

  int                 (*exit) (xui_playlist_t *pl);

  mmk_string_list_t     list;
  uint32_t              flags;
  /* while user has selected something manually for editing, moving, deleting etc.,
   * do not auto select the currently played item. */
  int                   user_mode;
  int                   sel;
  int                   mrlident;
};

/* neither "no selection" nor "Total time: ". */
#define _playlist_is_item(_i) ((unsigned int)(_i) < pl->list.used)

static void _playlist_flags (xui_playlist_t *pl) {
  unsigned int oneless = pl->list.used - (pl->list.used > 0);
  pl->flags = 0;
  pl->flags |= XITK_0_TO_MAX_MINUS_1 (pl->sel - 1, oneless) * PLAYLIST_MENU_FLAG_UP;
  pl->flags |= XITK_0_TO_MAX_MINUS_1 (pl->sel, oneless) * PLAYLIST_MENU_FLAG_DOWN;
  pl->flags |= _playlist_is_item (pl->sel) * PLAYLIST_MENU_FLAG_PLAY;
  pl->flags |= (pl->list.used > 0) * PLAYLIST_MENU_FLAG_CLEAR;
}

static void _playlist_able_buttons (xui_playlist_t *pl) {
  _playlist_flags (pl);
  xitk_widgets_state (pl->w + _W_move_up, 1,
    XITK_WIDGET_STATE_ENABLE, xitk_bitmove (pl->flags, PLAYLIST_MENU_FLAG_UP, XITK_WIDGET_STATE_ENABLE));
  xitk_widgets_state (pl->w + _W_move_down, 1,
    XITK_WIDGET_STATE_ENABLE, xitk_bitmove (pl->flags, PLAYLIST_MENU_FLAG_DOWN, XITK_WIDGET_STATE_ENABLE));
  xitk_widgets_state (pl->w + _W_play, 2,
    XITK_WIDGET_STATE_ENABLE, xitk_bitmove (pl->flags, PLAYLIST_MENU_FLAG_PLAY, XITK_WIDGET_STATE_ENABLE));
  xitk_widgets_state (pl->w + _W_delete_all, 2,
    XITK_WIDGET_STATE_ENABLE, xitk_bitmove (pl->flags, PLAYLIST_MENU_FLAG_CLEAR, XITK_WIDGET_STATE_ENABLE));
}

/*
 *
 */
static void _playlist_update_browser_list (xui_playlist_t *pl, int start, int newsel) {
  const char * const * list = (const char * const *)(pl->mrlident ? pl->list.mrl : pl->list.ident);

  if (start == -1) {
    start = pl->sel - (xitk_browser_get_num_entries (pl->w[_W_playlist]) >> 1);
    if (start < 0)
      start = 0;
  }
  if (newsel >= 0)
    pl->sel = newsel;
  if (pl->sel >= 0)
     start = -2 - pl->sel;
  pl->sel = xitk_browser_update_list (pl->w[_W_playlist], list, (const char * const *)pl->list.len,
    /* see mediamark.h */
    pl->list.used + (pl->list.used > 1), start);
  _playlist_able_buttons (pl);
}


/*
 *
 */
static void _playlist_handle_selection (xitk_widget_t *w, void *data, int sel, unsigned int modifier) {
  xui_playlist_t *pl = data;

  (void)w;
  (void)modifier;
  do {
    if (sel < 0) { /* unselect */
      if (!pl->user_mode)
        break;
      pl->user_mode = 0;
      sel = pl->list.cur;
      if (sel < 0)
        break;
      sel = xitk_browser_set_select (pl->w[_W_playlist], sel);
    } else if (!_playlist_is_item (sel)) { /* "Total time: " */
      sel = -1;
      pl->user_mode = 1;
      break;
    } else { /* item */
      pl->user_mode = 1;
    }
    xitk_inputtext_change_text (pl->w[_W_winput], pl->list.mrl[sel]);
    mmk_edit_mediamark (pl->nw.gui, NULL, NULL, sel);
  } while (0);
  pl->sel = sel;
  _playlist_able_buttons (pl);
}

static int _playlist_xine_play (gGui_t *gui, int indx) {
  int ret = gui_playlist_play (gui, indx);
  if (ret) {
    if ((ret == 2) && (gui->flags & XUI_FLAG_start_quit))
      gui_exit (NULL, gui);
    else
      gui_display_logo (gui);
  }
  return ret;
}

/*
 * Start to play the selected stream on double click event in playlist.
 */
static void _playlist_play_on_dbl_click (xitk_widget_t *w, void *data, int sel, unsigned int modifier) {
  xui_playlist_t *pl = data;

  (void)w;
  (void)modifier;
  pl->sel = sel;
  if (!_playlist_is_item (sel))
    return;
  if (_playlist_xine_play (pl->nw.gui, sel) == 0) {
    pl->user_mode = 0;
    xitk_inputtext_change_text (pl->w[_W_winput], pl->list.mrl[sel]);
    pl->sel = xitk_browser_set_select (pl->w[_W_playlist], sel);
  }
}

/*
 * Delete a given entry from playlist
 */
void playlist_delete_entry (gGui_t *gui, int j) {
  xui_playlist_t *pl;
  int num, cur, sel;

  if (!gui)
    return;
  pl = gui->plwin;

  if (pl) {
    num = pl->list.used;
    cur = pl->list.cur;
    sel = pl->sel;
  } else {
    gui_playlist_lock (gui);
    num = gui->playlist.num;
    sel = cur = gui->playlist.cur;
    gui_playlist_unlock (gui);
  }

  if (j == GUI_MMK_CURRENT)
    j = sel;

  if (!XITK_0_TO_MAX_MINUS_1 (j, num))
    return;

  mmk_editor_end (gui);

  if ((cur == j) && (xine_get_status (gui->stream) != XINE_STATUS_STOP))
    gui_stop (NULL, gui);

  num = gui_playlist_remove (gui, j, GUI_PLAYLIST_REMOVE_1);
  if (num > 0) {
    gui_current_set_index (gui, GUI_MMK_CURRENT);
    if (pl && pl->user_mode && (j >= num))
      pl->sel = xitk_browser_set_select (pl->w[_W_playlist], num - 1);
    playlist_update_playlist (gui);
  } else {
    gui_current_set_index (gui, GUI_MMK_NONE);
    playlist_update_playlist (gui);
    panel_playback_ctrl (gui->panel, 0);
    if (xine_get_status (gui->stream) != XINE_STATUS_STOP)
      gui_stop (NULL, gui);
    if (pl) {
      pl->user_mode = 0;
      xitk_inputtext_change_text (pl->w[_W_winput], NULL);
    }
  }
}

static void _playlist_action_2 (xitk_widget_t *w, void *data) {
  xui_playlist_t *pl = data;

  playlist_action (pl->nw.gui, xitk_widget_user_id (w));
}
static void _playlist_action_3 (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xui_playlist_t *pl = data;

  (void)state;
  (void)modifier;
  playlist_action (pl->nw.gui, xitk_widget_user_id (w));
}

/*
 * Load playlist from $HOME/.xine/playlist
 */
static void _playlist_fb_cb (filebrowser_t *fb, void *data) {
  gGui_t *gui = data;
  char file[XITK_PATH_MAX + XITK_NAME_MAX + 2];

  if (filebrowser_get_string (fb, file, sizeof (file), 1)) {
    if (fb == gui->pl_fb[1]) {
      gui_playlist_save (gui, file);
    } else {
      mmk_editor_end (gui);
      gui_playlist_add_item (gui, file, 1, GUI_ITEM_TYPE_PLAYLIST, 1);
      gui_current_set_index (gui, GUI_MMK_CURRENT);
      playlist_update_playlist (gui);
      if ((xine_get_status (gui->stream) == XINE_STATUS_PLAY))
        gui_stop (NULL, gui);
      if (gui->playlist.num > 0)
        panel_playback_ctrl (gui->panel, 1);
    }
  }
  if (fb == gui->pl_fb[1])
    gui->pl_fb[1] = NULL;
  else
    gui->pl_fb[0] = NULL;
  if (gui->plwin) {
    xitk_widgets_state (gui->plwin->w, _W_close + 1, XITK_WIDGET_STATE_ENABLE, ~0u);
    _playlist_able_buttons (gui->plwin);
  }
}

static void _playlist_exit_cb (filebrowser_t *fb, void *data) {
  gGui_t *gui = data;

  if (fb == gui->pl_fb[1])
    gui->pl_fb[1] = NULL;
  else
    gui->pl_fb[0] = NULL;
  if (gui->plwin) {
    xitk_widgets_state (gui->plwin->w, _W_close + 1, XITK_WIDGET_STATE_ENABLE, ~0u);
    _playlist_able_buttons (gui->plwin);
  }
}

/*
 *
 */
static void _playlist_add_input(xitk_widget_t *w, void *data, const char *filename) {
  xui_playlist_t *pl = data;

  (void)w;
  if (filename)
    gui_dndcallback (pl->nw.gui, filename);
}

/*
 * Leaving playlist editor
 */
static int _playlist_deinit (xui_playlist_t *pl) {
  gGui_t *gui = pl->nw.gui;

  if (gui->pl_fb[0]) {
    filebrowser_end (gui->pl_fb[0]);
    gui->pl_fb[0] = NULL;
  }
  if (gui->pl_fb[1]) {
    filebrowser_end (gui->pl_fb[1]);
    gui->pl_fb[1] = NULL;
  }
  mmk_editor_end (gui);

  gui_window_delete (&pl->nw);

  gui_playlist_free_strings (pl->nw.gui, &pl->list);

  gui->plwin = NULL;
  free (pl);
  return 1;
}

/*
 * Handle X events here.
 */
static int playlist_event (void *data, const xitk_be_event_t *e) {
  xui_playlist_t *pl = data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      return pl->exit (pl);
    case XITK_EV_BUTTON_UP:
      if (pl->w[_W_playlist]) {
        if (pl->sel < 0) {
          mmk_editor_end (pl->nw.gui);
          return 1;
        }
      }
      break;
    case XITK_EV_BUTTON_DOWN:
      if (e->code == 3) {
        xitk_widget_t *w = xitk_get_focused_widget (pl->nw.wl);

        if (xitk_get_widget_type (w) & WIDGET_GROUP_BROWSER) {
          playlist_menu (pl->nw.gui, pl->nw.wl, e->w, e->h, pl->flags);
          return 1;
        }
      } else if ((e->code == 4) || (e->code == 5)) {
        mmk_editor_end (pl->nw.gui);
        return 1;
      }
      break;
    case XITK_EV_KEY_UP:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        if (e->utf8[1] == XITK_KEY_ESCAPE)
          return xitk_widget_key_event (pl->w[_W_close], e, 1);
      }
      break;
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        xitk_widget_t *w;

        switch (e->utf8[1]) {
          case XITK_MOUSE_WHEEL_UP:
          case XITK_KEY_UP:
          case XITK_MOUSE_WHEEL_DOWN:
          case XITK_KEY_DOWN:
          case XITK_KEY_NEXT:
          case XITK_KEY_PREV:
            mmk_editor_end (pl->nw.gui);
            w = xitk_get_focused_widget (pl->nw.wl);
            if (!(xitk_get_widget_type (w) & WIDGET_GROUP_BROWSER))
              xitk_widget_key_event (pl->w[_W_playlist], e, 0);
            return 1;
          case XITK_KEY_ESCAPE:
            return xitk_widget_key_event (pl->w[_W_close], e, 1);
          case XITK_KEY_MENU:
            {
              xitk_widget_t *w = xitk_get_focused_widget (pl->nw.wl);

              if (xitk_get_widget_type (w) & WIDGET_GROUP_BROWSER) {
                playlist_menu (pl->nw.gui, pl->nw.wl, e->w, e->h, pl->flags);
                return 1;
              }
            }
          default: ;
        }
      }
    default: ;
  }
  return gui_handle_be_event (pl->nw.gui, e);
}

static void _playlist_apply_cb (void *data, int index) {
  gGui_t *gui = data;
  /* ee are called with freshly unlocked playlist,
   * so it should be safe to read playlist.cur here. */
  if (index == gui->playlist.cur) {
    /* this already calls gui_pl_updated (). */
    int changed = gui_current_set_index (gui, GUI_MMK_CURRENT);

    if (changed & MMK_CHANGED_AV_OFFS)
      xine_set_param (gui->stream, XINE_PARAM_AV_OFFSET, gui->mmk.av_offset);
    if (changed & MMK_CHANGED_SPU_OFFS)
      xine_set_param (gui->stream, XINE_PARAM_SPU_OFFSET, gui->mmk.spu_offset);
  } else {
    /* assume MMK_CHANGED_IDENT || MMK_CHANGED_MRL || MMK_CHANGED_LENGTH */
    playlist_update_playlist (gui);
  }
}

void playlist_get_input_focus (gGui_t *gui) {
  if (gui && gui->plwin)
    xitk_window_set_input_focus (gui->plwin->nw.xwin);
}

static void _scan_for_playlist_infos (xui_playlist_t *pl, xine_stream_t *stream, int n) {
  int state = 0;

  if (xine_open (stream, pl->list.mrl[n])) {
    char ident[2048];

    state = 1;
    if (stream_infos_get_ident_from_stream (stream, ident, sizeof (ident))
      && strcmp (ident, pl->list.ident[n])) {
      state = 2;
      gui_playlist_lock (pl->nw.gui);
      gui_playlist_set_str_val (pl->nw.gui, ident, MMK_VAL_IDENT, n);
      if (n == pl->nw.gui->playlist.cur) {
        mediamark_t *mmk = &pl->nw.gui->mmk;
        int changed = mediamark_set_str_val (&mmk, ident, MMK_VAL_IDENT);

        gui_playlist_unlock (pl->nw.gui);
        state = 3;
        if (changed)
          panel_message (pl->nw.gui->panel, NULL);
      } else {
        gui_playlist_unlock (pl->nw.gui);
      }
    }
    xine_close (stream);
  }

  if (pl->nw.gui->verbosity >= 2) {
    static const char names[4][20] = {
      "failed", "probed", "new ident", "new current ident"
    };
    printf ("playlist.scan_for_info (%d, %s) [%s].\n", n, pl->list.mrl[n], names[state]);
  }
}

/*
 *
 */
int playlist_update_playlist (gGui_t *gui) {
  xui_playlist_t *pl;
  int changed, oldsel;

  if (!gui)
    return -1;
  pl = gui->plwin;
  if (!pl)
    return -1;
  /* may be called from outside xitk_run (), eg seek thread. */
  if (xitk_lock (gui->xitk, 2))
    return -1;
  /* leads to a race when hitting ADD from mrl editor.
  if (!playlist_is_visible (gui)
    return; */
  oldsel = pl->sel;
  changed = gui_playlist_get_strings (pl->nw.gui, &pl->list);
  if (changed || (pl->mrlident != pl->nw.gui->is_display_mrl)) {
    pl->mrlident = pl->nw.gui->is_display_mrl;
    _playlist_update_browser_list (pl, -1, -1);
  }
  if (!pl->user_mode)
    pl->sel = pl->list.cur;
  if (!_playlist_is_item (pl->sel)) {
    mmk_editor_end (pl->nw.gui);
    pl->sel = -1;
  }
  if (changed || (pl->sel != oldsel)) {
    pl->sel = xitk_browser_set_select (pl->w[_W_playlist], pl->sel);
    /* Live streams keep growing their reported length. This triggers half second
     * updates here. Do not keep resetting the mrl field as well, while user is
     * trying to input something. */
    if (!xitk_inputtext_is_editing (pl->w[_W_winput])) {
      const char *oldtext = xitk_inputtext_get_text (pl->w[_W_winput]);
      const char *newtext = (pl->sel >= 0) ? pl->list.mrl[pl->sel] : "";

      if (strcmp (oldtext, newtext))
        xitk_inputtext_change_text (pl->w[_W_winput], newtext);
    }
    _playlist_able_buttons (pl);
    mmk_edit_mediamark (pl->nw.gui, NULL, NULL, pl->sel);
  }
  xitk_lock (gui->xitk, 0);
  return pl->sel;
}

/*
 * Handle autoplay buttons hitting (from panel and playlist windows)
 */
void playlist_scan_input_s (gGui_t *gui, const char *name) {
  xui_playlist_t *pl;
  int num_mrls = 0;
  const char * const *autoplay_mrls;

  if (!gui)
    return;
  pl = gui->plwin;
  autoplay_mrls = xine_get_autoplay_mrls (gui->xine, name, &num_mrls);

  if (autoplay_mrls) {
    xine_stream_t *stream;
    int j, cdda_mode = 0;

    if (!strcasecmp (name, "cd"))
      cdda_mode = 1;

    /* Flush playlist in newbie mode */
    if (gui->flags & XUI_FLAG_smart_mode) {
      gui_playlist_free (gui);
      playlist_update_playlist (gui);
      if (pl)
        xitk_inputtext_change_text (pl->w[_W_winput], NULL);
    }

    stream = xine_stream_new (gui->xine, gui->ao_none, gui->vo_none);
    for (j = 0; j < num_mrls; j++) {
      char ident[2048];
      int n = 0;

      if (cdda_mode && xine_open (stream, autoplay_mrls[j])) {
        n = stream_infos_get_ident_from_stream (stream, ident, sizeof (ident));
        xine_close (stream);
      }
      gui_playlist_append (gui, autoplay_mrls[j], n ? ident : autoplay_mrls[j], NULL, 0, -1, 0, 0);
    }
    xine_dispose (stream);

    gui->playlist.cur = gui->playlist.num ? 0 : -1;
    if (gui->playlist.cur == 0)
      gui_current_set_index (gui, GUI_MMK_CURRENT);
    /*
     * If we're in newbie mode, start playback immediately
     * (even ignoring if we're currently playing something
     */
    if (gui->flags & XUI_FLAG_smart_mode) {
      if (xine_get_status (gui->stream) == XINE_STATUS_PLAY)
        gui_stop (NULL, gui);
      gui_play (NULL, gui);
    }

    if (pl && gui_playlist_get_strings (pl->nw.gui, &pl->list))
      _playlist_update_browser_list (pl, 0, -1);

    panel_playback_ctrl (gui->panel, gui->playlist.num > 0);
  }
}
void playlist_scan_input_w (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  gGui_t *gui = data;

  (void)state;
  (void)modifier;
  playlist_scan_input_s (gui, xitk_labelbutton_get_label (w));
}

/*
 * Change the current skin.
 */
void playlist_change_skins (gGui_t *gui, int synthetic) {
  xui_playlist_t *pl;

  if (!gui)
    return;
  pl = gui->plwin;
  if (!pl)
    return;

  (void)synthetic;
  if (xitk_window_change_skin (pl->nw.xwin, pl->nw.gui->skin_config, "PlBG")) {
    gui_msg (pl->nw.gui, XUI_MSG_ERROR, _("%s(): couldn't find image for background\n"), __XINE_FUNCTION__);
    exit(-1);
  }
}

int playlist_action (gGui_t *gui, playlist_action_t action) {
  xui_playlist_t *pl;

  if (!gui)
    return 0;
  pl = gui->plwin;

  switch (action) {

    case PLAYLIST_SAVE:
      if (gui->playlist.num <= 0)
        break;
      /* fall through */
    case PLAYLIST_LOAD:
      action -= PLAYLIST_LOAD;
      if (gui->pl_fb[action]) {
        filebrowser_raise_window (gui->pl_fb[action]);
      } else {
        static const char _s1[2][8] = {N_("Load"), N_("Save")},
                          _s2[2][16] = {N_("Load a playlist"), N_("Save a playlist")};
        filebrowser_callback_button_t cbb[2] = {
          [0] = { .label = gettext (_s1[action]), .callback = _playlist_fb_cb, .userdata = gui, .need_a_file = 1 },
          [1] = { .callback = _playlist_exit_cb, .userdata = gui }
        };
        char buf[2048];
        snprintf (buf, sizeof (buf), "%s/.xine/playlist.tox", xine_get_homedir ());
        if (pl)
          xitk_widgets_state (pl->w, _W_LAST, XITK_WIDGET_STATE_ENABLE, 0);
        gui->pl_fb[action] = filebrowser_create (gui, pl ? pl->nw.xwin : NULL,
          gettext (_s2[action]), buf, cbb, 2, XUI_EXTS_PLAYLIST);
      }
      return 1;

    case PLAYLIST_CLEAR:
      mmk_editor_end (gui);
      gui_playlist_free (gui);
      playlist_update_playlist (gui);
      if (xine_get_status (gui->stream) != XINE_STATUS_STOP)
        gui_stop (NULL, gui);
      if (pl) {
        pl->user_mode = 0;
        xitk_inputtext_change_text (pl->w[_W_winput], NULL);
        pl->sel = xitk_browser_set_select (pl->w[_W_playlist], -1);
      }
      gui_current_set_index (gui, GUI_MMK_NONE);
      panel_playback_ctrl (gui->panel, 0);
      oxine_playlist_update (gui->oxine);
      return 1;

    case PLAYLIST_CURR_REMOVE:
      playlist_delete_entry (gui, GUI_MMK_CURRENT);
      oxine_playlist_update (gui->oxine);
      return 1;

    case PLAYLIST_CURR_UP:
    case PLAYLIST_CURR_DOWN:
      if (pl) {
        int diff = 2 * (int)(action - PLAYLIST_CURR_UP) - 1, j;

        mmk_editor_end (pl->nw.gui);
        j = pl->sel;
        if (_playlist_is_item (j)) {
          int start = xitk_browser_get_current_start (pl->w[_W_playlist]);
          int max_vis_len = xitk_browser_get_num_entries (pl->w[_W_playlist]);
          int new_pos = pl->sel = gui_playlist_move (pl->nw.gui, j, 1, diff);

          if ((new_pos < 0) || (new_pos == j))
            return 1;
          gui_playlist_get_strings (pl->nw.gui, &pl->list);
          _playlist_update_browser_list (pl,
            (new_pos < start) ? new_pos :
            (new_pos > start + max_vis_len - 1) ? new_pos + 1 - max_vis_len :
            -1,
            new_pos);
          oxine_playlist_update (gui->oxine);
        }
        return 1;
      }
      break;

    case PLAYLIST_CURR_PLAY:
      {
        int index;
        if (pl) {
          index = pl->sel;
          if (!_playlist_is_item (index))
            return 0;
        } else {
          index = gui->playlist.cur;
          if (!XITK_0_TO_MAX_MINUS_1 (index, gui->playlist.num))
            return 0;
        }
        if (!_playlist_xine_play (gui, index) && pl)
          pl->user_mode = 0;
      }
      oxine_playlist_update (gui->oxine);
      return 1;

    case PLAYLIST_CURR_EDIT:
      {
        int item;
        if (pl) {
          item = pl->sel;
          if (!_playlist_is_item (item))
            break;
        } else {
          gui_playlist_lock (gui);
          item = gui->playlist.cur;
          gui_playlist_unlock (gui);
        }
        mmk_edit_mediamark (gui, _playlist_apply_cb, gui, item);
      }
      return 1;

    case PLAYLIST_CURR_SCAN:
    case PLAYLIST_SCAN:
      if (pl) {
        int start, item, stop;
        xine_stream_t *stream = xine_stream_new (gui->xine, gui->ao_none, gui->vo_none);

        if (action == PLAYLIST_SCAN)
          start = 0, stop = pl->list.used;
        else
          start = pl->sel, stop = start + (start >= 0);
        for (item = start; item < stop; item++)
          _scan_for_playlist_infos (pl, stream, item);
        xine_dispose (stream);
        playlist_update_playlist (gui);
        return 1;
      }
      break;

    case PLAYLIST_LAST:
      if (pl)
        return pl->exit (pl);
      /* fall through */
    default: ;
  }
  return 0;
}

/*
 * Create playlist editor window
 */
void playlist_main (xitk_widget_t *mode, void *data) {
  gGui_t *gui = data;
  xui_playlist_t *pl;

  if (!gui)
    return;

  pl = gui->plwin;
  if (mode == XUI_W_OFF) {
    if (!pl)
      return;
    pl->exit (pl);
    return;
  } else if (mode == XUI_W_ON) {
    if (pl) {
      gui_raise_window (gui, pl->nw.xwin);
      mmk_editor_raise_window (pl->nw.gui);
      return;
    }
  } else { /* toggle */
    if (pl) {
      pl->exit (pl);
      return;
    }
  }

  pl = (xui_playlist_t *)calloc (1, sizeof (xui_playlist_t));
  if (!pl)
    return;

  pl->nw.gui = gui;
  pl->nw.gui->plwin = pl;
  pl->exit = _playlist_deinit;

  pl->user_mode = 0;
  pl->sel = -1;

  gui_playlist_init_strings (&pl->list);

  pl->nw.title = _("xine Playlist Editor");
  pl->nw.id = "playlist";
  pl->nw.skin = "PlBG";
  pl->nw.wfskin = "PlWF";
  pl->nw.adjust = NULL;
  pl->nw.wr.x = 200;
  pl->nw.wr.y = 200;
  {
    int r = gui_window_new (&pl->nw);
    if (r < 0) {
      gui_msg (pl->nw.gui, XUI_MSG_ERROR, _("playlist: couldn't find image for background\n"));
      free (pl);
      return;
    }
  }

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = pl->nw.wl,
        .userdata = pl->nw.gui,
        .add_state = XITK_WIDGET_STATE_KEEP,
        .skin_element_name = "PlAdd",
        .tips = _("Add one or more entries in playlist")
      },
      .button_type = CLICK_BUTTON,
      .align       = ALIGN_DEFAULT,
      .label       = _("Add"),
      .callback    = open_mrlbrowser_from_playlist
    };

    pl->w[_W_add] =  xitk_labelbutton_create (&lb, pl->nw.gui->skin_config);

    lb.nw.userdata = pl;
    lb.callback = _playlist_action_3;

    lb.nw.user_id = PLAYLIST_LAST;
    lb.nw.skin_element_name = "PlDismiss";
    lb.nw.tips = _("Close playlist window");
    lb.label = _("Dismiss");
    pl->w[_W_close] = xitk_labelbutton_create (&lb, pl->nw.gui->skin_config);

    lb.nw.user_id = PLAYLIST_LOAD;
    lb.nw.skin_element_name = "PlLoad";
    lb.nw.tips = _("Load saved playlist");
    lb.label = _("Load");
    pl->w[_W_load] = xitk_labelbutton_create (&lb, pl->nw.gui->skin_config);

    lb.nw.user_id = PLAYLIST_SAVE;
    lb.nw.skin_element_name = "PlSave";
    lb.nw.tips = _("Save playlist");
    lb.label = _("Save");
    pl->w[_W_save] = xitk_labelbutton_create (&lb, pl->nw.gui->skin_config);
  }

  /* optimized playlist_update_playlist (). */
  gui_playlist_get_strings (pl->nw.gui, &pl->list);
  pl->mrlident = pl->nw.gui->is_display_mrl;
  pl->sel = pl->list.cur;
  if (!_playlist_is_item (pl->sel))
    pl->sel = -1;
  _playlist_flags (pl);

  {
    xitk_browser_widget_t br = {
      .nw = {
        .wl = pl->nw.wl,
        .skin_element_name = "PlItemBtn",
        .userdata = pl,
        .add_state = XITK_WIDGET_STATE_KEEP
      },
      .arrow_up = { .skin_element_name = "PlUp" },
      .slider = { .skin_element_name = "SliderPl" },
      .arrow_dn = { .skin_element_name = "PlDn" },
      .arrow_left = { .skin_element_name  = "PlLeft" },
      .slider_h = { .skin_element_name = "SliderHPl" },
      .arrow_right = { .skin_element_name = "PlRight" },
      .browser = {
        /* see mediamark.h */
        .num_entries = pl->list.used + (pl->list.used > 1),
        .entries = (const char * const *)(pl->mrlident ? pl->list.mrl : pl->list.ident),
        .shortcuts = (const char * const *)pl->list.len
      },
      .callback = _playlist_handle_selection,
      .dbl_click_callback = _playlist_play_on_dbl_click
    };
    pl->w[_W_playlist] = xitk_browser_create (&br, pl->nw.gui->skin_config);
  }
  pl->sel = xitk_browser_set_select (pl->w[_W_playlist], pl->sel);

  {
    xitk_label_widget_t lbl = {
      .nw = { .wl = pl->nw.wl, .skin_element_name = "AutoPlayLbl", .add_state = XITK_WIDGET_STATE_KEEP },
      /* TRANSLATORS: only ASCII characters (skin) */
      .label = pgettext ("skin", "Scan for:")
    };
    pl->w[_W_list_title] = xitk_label_create (&lbl, pl->nw.gui->skin_config);
  }

  {
    xitk_inputtext_widget_t inp = {
      .nw = {
        .wl = pl->nw.wl,
        .userdata = pl,
        .add_state = XITK_WIDGET_STATE_KEEP,
        .skin_element_name = "PlInputText",
        .tips = _("Direct MRL entry")
      },
      .text       = (pl->sel >= 0) ? pl->list.mrl[pl->sel] : "",
      .max_length = 4095,
      inp.callback   = _playlist_add_input
    };
    pl->w[_W_winput] =  xitk_inputtext_create (&inp, pl->nw.gui->skin_config);
  }

  do {
    const char *tips[64];
    const char * const *autoplay_plugins = xine_get_autoplay_input_plugin_ids (pl->nw.gui->xine);
    unsigned int i;
    xitk_button_list_widget_t nbl = {
      .nw = {
        .wl = pl->nw.wl,
        .skin_element_name = "AutoPlayBG",
        .tips = _("More sources..."),
        .userdata = pl->nw.gui,
        .add_state = XITK_WIDGET_STATE_KEEP
      },
      .names = autoplay_plugins,
      .tips = tips,
      .callback = playlist_scan_input_w
    };

    if (!autoplay_plugins)
      break;

    for (i = 0; autoplay_plugins[i]; i++) {
      if (i >= sizeof (tips) / sizeof (tips[0]))
        break;
      tips[i] = xine_get_input_plugin_description (pl->nw.gui->xine, autoplay_plugins[i]);
    }

    pl->w[_W_autoplay_buttons] = xitk_button_list_new (&nbl, pl->nw.gui->skin_config);
  } while (0);

  {
    xitk_button_widget_t b = {
      .nw = {
        .wl = pl->nw.wl,
        .add_state = (pl->flags & PLAYLIST_MENU_FLAG_UP) ?
          (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE) : XITK_WIDGET_STATE_VISIBLE,
        .userdata = pl,
        .user_id = PLAYLIST_CURR_UP,
        .skin_element_name = "PlMoveUp",
        .tips = _("Move up selected MRL")
      },
      .callback = _playlist_action_2
    };

    pl->w[_W_move_up] =  xitk_button_create (&b, pl->nw.gui->skin_config);

    b.nw.add_state = (pl->flags & PLAYLIST_MENU_FLAG_DOWN) ?
      (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE) : XITK_WIDGET_STATE_VISIBLE;
    b.nw.skin_element_name = "PlMoveDn";
    b.nw.tips = _("Move down selected MRL");
    b.nw.user_id = PLAYLIST_CURR_DOWN;
    pl->w[_W_move_down] =  xitk_button_create (&b, pl->nw.gui->skin_config);

    b.nw.add_state = (pl->flags & PLAYLIST_MENU_FLAG_PLAY) ?
      (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE) : XITK_WIDGET_STATE_VISIBLE;

    b.nw.skin_element_name = "PlPlay";
    b.nw.tips = _("Start playback of selected MRL");
    b.nw.user_id = PLAYLIST_CURR_PLAY;
    pl->w[_W_play] =  xitk_button_create (&b, pl->nw.gui->skin_config);

    b.nw.skin_element_name = "PlDelete";
    b.nw.tips = _("Delete selected MRL from playlist");
    b.nw.user_id = PLAYLIST_CURR_REMOVE;
    pl->w[_W_delete] =  xitk_button_create (&b, pl->nw.gui->skin_config);

    b.nw.add_state = (pl->flags & PLAYLIST_MENU_FLAG_CLEAR) ?
      (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE) : XITK_WIDGET_STATE_VISIBLE;
    b.nw.skin_element_name = "PlDeleteAll";
    b.nw.tips = _("Delete all entries in playlist");
    b.nw.user_id = PLAYLIST_CLEAR;
    pl->w[_W_delete_all] = xitk_button_create (&b, pl->nw.gui->skin_config);
  }

  pl->nw.key = xitk_be_register_event_handler ("playlist", pl->nw.xwin, playlist_event, pl, NULL, NULL);
  xitk_window_flags (pl->nw.xwin, XITK_WINF_DND, XITK_WINF_DND);

  /* if that is open before. */
  mmk_editor_raise_window (pl->nw.gui);
  /* playlist_update_focused_entry (pl->nw.gui); */
  gui_raise_window (pl->nw.gui, pl->nw.xwin);
  xitk_set_focus_to_widget (pl->w[_W_winput]);
}

