#include <stdio.h>
#include <stdarg.h>
#include <math.h>
#include "dvi.h"
#include "dvilw.h"
#include "dvisplin.h"

static double linewidth, arrow_width = 0, arrow_length = 0;
static double arrow_t, arrow_l, arrow_d;
static double dotsize = 0;
static int fill_mode, fill_start, fill_error;
static int line_startstyle = 0, line_endstyle = 0;
static int last_x2, last_y2;

void setlinecap(int left, int right)
{
    line_startstyle = left;
    line_endstyle = right;
}

void setlinewidth(real width)
{
    linewidth = width;
}

void setarrowsize(real width, real length)
{
    arrow_length = length;
    arrow_width = width;
}

void setarrowshape(real t, real l)
{
    arrow_t = t;
    arrow_l = l;
}

void setarrowline(real d)
{
    arrow_d = d;
}

void setdotsize(real diameter)
{
    dotsize = diameter/2.0;
}

void setdash(real *dash, int len)
{
    if (len==0)
	psprint("[] V\n");
    else
    {
	int j;

	psprint("[");
	for (j=0; j<len; j++) psprint(" %f",dash[j]);
	psprint(" ] V\n");
    }
}

static void g_bezier(const coord *p1, const coord *p2, const coord *p3)
{
    psprint("%.1f %.1f %.1f %.1f %.1f %.1f C\n",
	p1->x, p1->y, p2->x, p2->y, p3->x, p3->y);
}

static void circle(const coord *p, double linewidth)
{
    psprint("%.1f %.1f %.2f E\n",p->x,p->y,linewidth);
}

static void arrow(coord *p1, coord *p2, int style)
{
    coord p[3], q[3], pa, pb;
    real x, y, n, px_m, py_m;

    if (arrow_length<1 || arrow_width<1) return;

    x = (p2->x - p1->x);
    y = (p2->y - p1->y);
    n = sqrt(x*x+y*y);
    if (n<0.5) return;
    x /= n;
    y /= n;

    p[0] = *p2;
    p[1].x = p[0].x - x*arrow_length-y*arrow_width;
    p[1].y = p[0].y - y*arrow_length+x*arrow_width;
    p[2].x = p[0].x - x*arrow_length+y*arrow_width;
    p[2].y = p[0].y - y*arrow_length-x*arrow_width;

    px_m = (p[0].x+p[1].x+p[2].x)/3;
    py_m = (p[0].y+p[1].y+p[2].y)/3;

    q[2].x = px_m*arrow_t + (p[1].x+p[0].x)*(1-arrow_t)/2;
    q[2].y = py_m*arrow_t + (p[1].y+p[0].y)*(1-arrow_t)/2;
    q[1].x = px_m*arrow_t + (p[2].x+p[0].x)*(1-arrow_t)/2;
    q[1].y = py_m*arrow_t + (p[2].y+p[0].y)*(1-arrow_t)/2;
    q[0].x = px_m*arrow_l + (p[1].x+p[2].x)*(1-arrow_l)/2;
    q[0].y = py_m*arrow_l + (p[1].y+p[2].y)*(1-arrow_l)/2;

    if (style!=2)
    {
	pa.x = (p[1].x+p[2].x)/8 + 0.75*q[0].x;
	pa.y = (p[1].y+p[2].y)/8 + 0.75*q[0].y;
	pb.x = (p[1].x+p[2].x)/2;
	pb.y = (p[1].y+p[2].y)/2;

	if (arrow_l>0) *p2 = pb;
	else           *p2 = pa;
	x = (p2->x - p->x);
	y = (p2->y - p->y);
	if (sqrt(x*x+y*y)>n) *p1 = *p2; 
	psprint("%.2f %.1f %.1f %.1f %.1f AX\n",linewidth,
	    pa.x,pa.y,pb.x,pb.y);
    }
    if (style==1) psprint("%.2f ",arrow_d);
    if (style==2) psprint("%.2f ",linewidth);
    else psprint("%.1f %.1f %.1f %.1f %.1f %.1f\n",
	q[0].x,q[0].y,q[0].x,q[0].y,p[2].x,p[2].y);
    psprint("%.1f %.1f %.1f %.1f %.1f %.1f\n",
	q[2].x,q[2].y,q[2].x,q[2].y,p[1].x,p[1].y);
    psprint("%.1f %.1f %.1f %.1f %.1f %.1f\n",
	q[1].x,q[1].y,q[1].x,q[1].y,p[0].x,p[0].y);
    psprint("%.1f %.1f A%c\n",p[2].x,p[2].y,style+'0');
}

void clip_init(void)
{
    fill_error = 0;
    fill_start = 1;
    fill_mode = 1;
    psprint("G ");
}

int clip_exec(unsigned char *cp)
{
    if (fill_mode)
    {
	if (cp) 
	{
	    psprint("currentpoint grestore gsave\n");
	    psprint("<%02x%02x%02x%02x%02x%02x%02x%02x> 8 1 300 32 div setpattern\n",
		cp[0],cp[1],cp[2],cp[3],cp[4],cp[5],cp[6],cp[7]);
	    psprint("fill grestore clip moveto\n");
	    /* psprint("I "); */
	}
	else psprint("H ");
	fill_mode = 0;
    }
    return fill_error;
}

void clip_end(void)
{
    if (fill_mode) psprint("currentpoint grestore moveto ");
    psprint("J ");
    fill_mode = 0;
}

static void startline(coord *p1, coord *p2, int style)
{
    if (fill_mode)
    {
	if (fill_start)
	{
	    psprint("O %.1f %.1f K\n",p1->x,p1->y);
	    fill_start = 0;
	}
	else
	{
	    if (iround(p1->x)!=last_x2 || iround(p1->y)!=last_y2)
		fill_error = 1;
	    psprint("O ");
	}
    }
    else
    {
	switch(style)
	{
	    case 1 : circle(p1,linewidth/2); break;
	    case 2 :
	    case 3 :
	    case 4 : arrow(p2,p1,style-2); break;
	}
	psprint("%.1f %.1f %.2f F\n",p1->x,p1->y,linewidth);
    }
}

static void endline(coord *p1, coord *p2, int style)
{
    last_x2 = iround(p2->x);
    last_y2 = iround(p2->y);
    if (!fill_mode)
    {
	switch(style)
	{
	    case 1 : circle(p2,linewidth/2); break;
	    case 2 :
	    case 3 :
	    case 4 : arrow(p1,p2,style-2); break;
	}
    }
}

static void finish(void)
{
    if (fill_mode) psprint("U ");
    else psprint("L ");
}

void g_dot(const coord *p)
{
    circle(p,dotsize);
}

void g_dpoly(const coord *p, const coord *edge, int n, int closed)
{
    if (n>2)
    {
	if (closed)
	{
	    int i;
	    coord r[2];
	    r[0] = p[0];
	    r[1] = p[1];
	    startline(r,r+1,0);
	    for (i=1; i<n; i++)
		psprint("%.1f %.1f N\n",p[i].x,p[i].y);
	    psprint("closepath\n");
	}
	else
	{
	    coord r[2];
	    r[0] = edge[0];
	    r[1] = *++p;
	    startline(r,r+1,line_startstyle);
	    for (n -= 2; n--; p++)
		psprint("%.1f %.1f N\n",p->x,p->y);
	    r[0] = edge[1];
	    r[1] = p[-1];
	    endline(r+1,r,line_endstyle);
	    psprint("%.1f %.1f N\n",r->x,r->y);
	}
    }
    else if (n==2)
    {
	coord r[2];
	r[0] = edge[0];
	r[1] = edge[1];
	startline(r,r+1,line_startstyle);
	endline(r,r+1,line_endstyle);
	psprint("%.1f %.1f N\n",r[1].x,r[1].y);
    }
    finish();
}

void g_draw(const coord *p, const coord *q, coord *r, int n, int closed)
{
    if (n-- > 0)
    {
	startline(r,r+1,closed ? 0:line_startstyle);
	g_bezier(r+1,r+2,p+1);
	p++, q++;
	while(n-- > 0)
	{
	    coord r;
	    r.x = 2*p[1].x-q[1].x;
	    r.y = 2*p[1].y-q[1].y;
	    g_bezier(q,&r,p+1);
	    p++, q++;
	}
	endline(r+4,r+5, closed ? 0:line_endstyle);
	g_bezier(r+3,r+4,r+5);
	if (closed) psprint("closepath\n");
    }
    else
    {
	startline(r,r+3,line_startstyle);
	endline(r+4,r+5,line_endstyle);
	g_bezier(r+3,r+4,r+5);
    }
    finish();
}

