/* This file contains the code that implements the handling of \special's
in Ln03DVI. 

Copyright (c) 1985, 1987 by Digital Equipment Corporation. 
Originial Author: Flavio Rose, (...!decwrl!dvinci.dec.com!rose)

This version of Ln03DVI is maintained by Matt Thomas.
Please send all bug reports to either:
    ...!decwrl!thebay.dec.com!mthomas (UUCP)
    mthomas@thebay.dec.com (Internet)

Right now, the following \special's are implemented: 

ln03:defpoint fixnum ( [dimension] , [dimension] )
ln03:connect fixnum [/ fixnum] fixnum [/ fixnum]
ln03:plotfile filename
ln03:resetpoints fixnum fixnum

A dimension is a number with optional decimal point, followed by one of pt,
in, pc, cm, mm, bp, dd, cc, and mi. 

We try to mimic the functionality of DVIAPS, a program that drives the
Autologic Micro-5 typesetter, written by Textset Inc. of Ann Arbor,
Michigan. This has been done on the basis of user-level documentation
supplied by Textset. This code does not contain any Textset-proprietary
information. 

\special's are parsed case-insensitively.

Revision history:

10/8/85		Created file, wrote, tested code for ln03:plotfile

10/16/85	ln03:resetpoints, ln03:defpoint, ln03:connect

12/12/85	Corrected bug: ln03:plotfile does not update LN03's
    		position to the latest from the DVI file.

*/

#ifdef vms
#include stdio
#include ctype
#include math
#include file
#else
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#endif

/* In VMS we declare external variables to be globalref. 
This is not really necessary, just an old habit. */

#ifdef vms
#define GLOBAL globaldef
#define EXTERN globalref
#else
#define GLOBAL 
#define EXTERN extern
#endif

#define NUMSPECIALS 4

GLOBAL char *specialnames[NUMSPECIALS] = { "defpoint",
    "connect", "plotfile", "resetpoints" } ;

EXTERN FILE *dvifile,*outfile;

/* There are two top-level routines for handling specials, one to be
invoked during pass1, the other to be invoked during pass2. The parameter
that is passed is the size of the \special string. */ 

int do_special_pass1 (p) 
int p;
{
    for (; p != 0; p--) getc(dvifile);
    return(0);
}

/* Specials get put in a buffer for easier parsing. The buffer is a little
longer than needed to make some things simpler. [[This ought to be
cleaner.]] */ 

#define MAXSPECIAL 1000

GLOBAL char sb[MAXSPECIAL+7];
GLOBAL int sbp,cstart;

/* Do_special_pass2 gets called from the main program to perform special
processing in pass2, the pass in which stuff actually gets written out to
the LN3 output file. The parameter p is the length of the special. */

int do_special_pass2 (p)
int p;
{
    char c;
    int i,j;

/* Skip whitespace in input */
        
    sbp = 0;
    while (p != 0) {
    	c = getc(dvifile);
    	p--;
    	if (!isspace(c)) {
	    sb[sbp] = c;
	    sbp = 1;
	    break;
	}
    }
    if (sbp == 0) return(1);

/* Now we have non-whitespace, read the rest of the special, or as much as
will fit in the buffer. Lowercase the stuff as it is read in. */ 

    for (; p != 0; p--) {
    	sb[sbp] = tolower(getc(dvifile));
    	sbp++;
    	if (sbp > MAXSPECIAL) break;
    }

/* Put a null as sentinel at the end of the special. */

    sb[sbp] = 0;

/* Determine whether the special pertains to the ln03, in which case sbpf
will contain "ln03" followed by whitespace or a colon. */ 

    if (strncmp(sb,"ln03",sizeof("ln03")-1) != 0) return(1);
    sbp = sizeof("ln03")-1;

/* Skip whitespace and one colon */

    while (isspace(sb[sbp])) sbp++;
    if (sb[sbp] != ':') return(1);
    sbp++;

/* Now, it might be that the special was too long for the special buffer,
so check that out and issue an error message. */ 

    if (p != 0) {
    	printf("\n \special too long (over %d bytes).",MAXSPECIAL);
    	for (; p != 0; p--) getc(dvifile);
    	return(1);
    }

/* Now determine if the special command is one of those which the driver is
supposed to recognize. As in the DVIAPS program, only the first six bytes
of the command are significant, so plotfile could be written plotfi and
defpoint could be written defpoi. */ 

    while (isspace(sb[sbp])) sbp++;
    cstart = sbp;
    while (!isspace(sb[sbp]) && sb[sbp] != '\0') sbp++;
    sb[sbp] = 0;
    sbp++;

    for (i = 0; i<NUMSPECIALS; i++)
    	if (strncmp(&sb[cstart],specialnames[i],6) == 0) break;

    if (i == NUMSPECIALS) {
    	printf("\n Unrecognized option '%s' ignored in \special command",
    		&sb[cstart]);
    	return(1);
    } else {
    	switch (i) {
    	case 0:
	    j = do_defpoint_pass2();
    	    break;
    	case 1:
    	    j = do_connect_pass2();
    	    break;
    	case 2:
    	    j = do_plotfile_pass2();
    	case 3:
    	    j = do_resetpoints();
    	}
    }
}

/* In certain \specials, dimensions may be specified in any of nine
different units of measure. */ 

#define MAXUNITS 9

GLOBAL char *unit_array[MAXUNITS] = { "pt", "in", "pc", "cm", "mm", "bp", 
    "dd", "cc", "mi" };

/* Unit_convert gives a floating point conversion factor from units to
pixels. [[The factor for micas is doubtful.]] */ 

#define PIXELS_PER_INCH (300.0)

GLOBAL float unit_convert[MAXUNITS] = { PIXELS_PER_INCH/72.27, 
    PIXELS_PER_INCH, 12.0*PIXELS_PER_INCH/72.27,
    PIXELS_PER_INCH/2.54, PIXELS_PER_INCH/25.4, 
    PIXELS_PER_INCH/72.0, (1238.0/1157.0)*PIXELS_PER_INCH/72.27,
    12.0*(1238.0/1157.0)*PIXELS_PER_INCH/72.27, PIXELS_PER_INCH/2540.0 };

/* There are 255 point variables, numbered 1 through 255. */

#define MAXPOINTS 255

GLOBAL int point_hh[MAXPOINTS], point_vv[MAXPOINTS];
GLOBAL int point_present[MAXPOINTS];

GLOBAL char *special_mess = "\n Error in \special{ln03:%s...} parameters";

/* For the purposes of defpoint, we need to know the current horizontal and
vertical positions in pixels. */ 

EXTERN int hh,vv,hoff,voff;

/* ln03:defpoint fixnum ( [dimension] , [dimension] ) */

int do_defpoint_pass2()
{
    int i,which,hh_val,vv_val;
    float xx_p,yy_p;

    which = 0;
    if (scan_fixnum(&which) != 0 || which < 1 || which > MAXPOINTS) {
    	printf(special_mess,"defpoint");
    	printf("\n Invalid point number (%d)",which);
    	return(1);
    }
    while (isspace(sb[sbp]) || sb[sbp] == ',') sbp++;
    hh_val = hh;
    vv_val = vv;
    if (sb[sbp] != '(') goto record_point;
    sbp++;
    while (isspace(sb[sbp])) sbp++;
    if (sb[sbp] == '\0') goto record_point;
    if (sb[sbp] == ',') sbp++;
    else {
    	scan_dimension(&hh_val);
        while (isspace(sb[sbp])) sbp++;
    	if (sb[sbp] == ',') sbp++;
    	else goto record_point;
    } 
    while(isspace(sb[sbp])) sbp++;
    if (sb[sbp] != ')' && sb[sbp] != '\0') scan_dimension(&vv_val);
record_point:
    point_hh[which-1] = hh_val;
    point_vv[which-1] = vv_val;
    point_present[which-1] = 1;
    return(0);
}

EXTERN long first_counter;

int do_connect_pass2()
{
    int a1,b1,a2,b2,w;
    
    scan_xpoint(&a1,&b1);
    while (isspace(sb[sbp]) || sb[sbp] == ',') sbp++;
    scan_xpoint(&a2,&b2);
    while (isspace(sb[sbp]) || sb[sbp] == ',') sbp++;
    w = 2; /* default width is two pixels ~ 0.4 points */
    if (sb[sbp] != '\0') scan_dimension(&w);

    if (first_counter%2 == 0) connect_points(b1,b2,w);
    else connect_points(a1,a2,w);

    return(0);
}

EXTERN int ln3p,vpset,hh_old;

int do_plotfile_pass2()
{
    char buf[512];
    int i,j;
    char c;

/* Read filename into array. */ 

    while (isspace(sb[sbp])) sbp++;
    i = sbp;
    while (!isspace(sb[sbp]) && sb[sbp] != '\0') sbp++;
    sb[sbp] = 0;

/* Try to open the file. [[What is the status of O_RDONLY under Unix? Is
that a 4.2bsd hack that it would be best to leave out?]] */ 

#ifdef vms
    j = open(&sb[i],O_RDONLY,0);
#else
    j = open(&sb[i],0,0);
#endif
    if (j == -1) {
    	printf("\n Unable to open plotfile %s",&sb[i]);
    	return(1);
    }

/* In executing the plotfile special, always emit escape sequences to place
the LN03 at the current position of the DVI file, even if it would seem
that the LN03 is already there. We do this in order to be able to emit a
newline at this point. Emitting a newline at this point keeps the line
length of the .ln3 file to <= 16 characters more than the line length of
the plotfile. */ 

    fprintf(outfile,"\n\033[%dd\033[%d`",vv+voff,hh+hoff);

/* Read stuff from the plotfile, write it to the output file. [[Could this
cause problem with one record being split into two?]] */ 

    while((i = read(j,buf,512)) > 0) 
    	fwrite(buf,i,1,outfile);
    close(j);

/* Now, set the variables ln3p, vpset and hh_old to indicate to the caller
that its recorded value of the current position of the LN03 is no longer
correct. This will make the caller issue absolute positioning commands
before doing any further output to the dvifile. */ 

    ln3p = 0;
    vpset = 0;
    hh_old = 30000;
    return(0);
}


/* Scan_fixnum reads an integer off sb[], starting at sbp and advancing sbp
to the first character past a valid integer. The integer is returned in i.
*/ 

int scan_fixnum(i)
long *i;
{
    int sbp_save;

    sbp_save = sbp;
    *i = atol(&sb[sbp]);
    while(isspace(sb[sbp])) sbp++;
    if (sb[sbp] == '+' || sb[sbp] == '-') sbp++;
    while (isdigit(sb[sbp])) sbp++;
    return(sbp == sbp_save);
}


/* Scan_flonum reads a flonum (without exponential notation) off sb[],
starting at sbp and advancing sbp to the first character past a valid
flonum. The flonum is returned in x. [[Unfortunately, it is impossible to
use atof or sscanf to implement this function, because they don't allow
flonums like ".3" which don't have any digits before the decimal point.]]
*/ 

int scan_flonum(x)
float *x;
{
    float j,frac;
    char negative;
    int sbp_save;

    sbp_save = sbp;
    negative = 0;
    j = 0.0;
    while (isspace(sb[sbp])) sbp++;
    if (sb[sbp] == '-') { negative = 1; sbp++; }
    else if (sb[sbp] == '+') sbp++;
    while (isdigit(sb[sbp])) { 
    	j = 10.0*j + ((float)(sb[sbp] - '0'));
    	sbp++;
    }
    if (sb[sbp] == '.') {
    	sbp++;
    	frac = 0.1;
    	while (isdigit(sb[sbp])) {
	    j += frac*((float)(sb[sbp] - '0'));
	    frac *= 0.1;
	    sbp++;
	}
    }
    if (sbp != sbp_save) {
    	*x = negative ? (-j) : j;
        return(0);
    } else return(1);
    
}


/* Scan_dimension tries to parse a dimension off sb[], advancing sbp over
what it can parse. The dimension is converted to pixels and returned in
val. If the unit isn't recognizable, scan_dimension prints an error
message, returns 1 and leaves val unchanged. */ 

int scan_dimension(val)
int *val;
{
    float x;
    int i;

    x = 0.0;
    scan_flonum(&x);
    while (isspace(sb[sbp])) sbp++;
    for (i = 0; i<MAXUNITS; i++)
    	if (strncmp(&sb[sbp],unit_array[i],2) == 0) break;
    if (i == MAXUNITS) {
    	printf(special_mess,&sb[cstart]);
    	printf("\n No such unit of measure: %.2s",&sb[sbp]);
    	return(1);
    }
    sbp += 2;
    x *= unit_convert[i];
#ifdef vms
    *val = (x < 0) ? floor(x-0.5) : floor(x+0.5);
#else
    *val = (x < 0) ? (x-0.5) : (x+0.5);
#endif
    return(0);
}

/* Scan_xpoint picks an xpoint off sb[], advancing sbp as usual. If both
parts are present they are returned in a,b; if only one is, it's returned
in both a and b. */ 

int scan_xpoint(a,b)
int *a,*b;
{

    *a = 0; *b = 0;
    scan_fixnum(a);
    while (isspace(sb[sbp])) sbp++;
    if (sb[sbp] == '/') {
    	sbp++;
    	scan_fixnum(b);
    } else *b = *a;
    return(0);
}

/* int do_rule(xx0,yy0,xx1,yy1) */

int connect_points(b1,b2,w) 
int b1,b2;
int w;
{
    int halfw;


    if (b1 < 1 || b1 > MAXPOINTS || !point_present[b1-1]) {
    	printf(special_mess,&sb[cstart]);
    	printf("\n Invalid point number (%d)",b1);
    	return(1);
    }
    if (b2 < 1 || b2 > MAXPOINTS || !point_present[b2-1]) {
    	printf(special_mess,&sb[cstart]);
    	printf("\n Invalid point number (%d)",b2);
    	return(1);
    }

    halfw = w/2;
    if (point_hh[b1-1] == point_hh[b2-1]) 
    	do_rule(point_hh[b1-1]-halfw,
    		point_vv[b1-1],point_hh[b1-1]+w-1-halfw,
    		point_vv[b2-1]);
    else if (point_vv[b1-1] == point_vv[b2-1])
    	do_rule(point_hh[b1-1],
    		point_vv[b1-1]-halfw,point_hh[b2-1],
    		point_vv[b1-1]+w-1-halfw);
    else {
    	printf(special_mess,&sb[cstart]);
    	printf("\n Can't connect along a diagonal.");
    	return(1);
    }        
    return(0);
}

int do_resetpoints ()
{
    int a,b;

    a = 0;
    scan_fixnum(&a);
    b = 0;
    scan_fixnum(&b);

    if (0 < a && a <= b && b <= MAXPOINTS) 
    	for (; a <= b; a++) point_present[a] = 0;
    else if (0 < a && a <= MAXPOINTS)
    	point_present[a] = 0;
    return(0);
}

