IMPLEMENTATION MODULE sysinterface;

(* Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   SYSDEP: Implements InitSysInterface, the routine that gets the
   DVItoVDU command line and extracts the DVI file name and option values.
*)

FROM parameters IMPORT
   NumParameters, GetParameter, GetEnvironment;

FROM io IMPORT
   SReadf;

FROM screenio IMPORT
   Write, WriteString, WriteLn, RestoreTerminal;

TYPE
   units = (ic,cm,mm,pc,pt,px);
   (* SYSDEP: changed in to ic; compiler confused with IN, even with -sk!!! *)

VAR
   option : CHAR;         (* current command option *)
   value : stringvalue;   (* current option's value *)
   length : INTEGER;      (* current value's length *)

(**********************************************************************)

PROCEDURE ExplicitExt (fname : ARRAY OF CHAR; len : INTEGER) : BOOLEAN;

(* SYSDEP: Check for an extension of the form ".*" where * is any string.
   len is length of fname.
   If "." found then TRUE is returned, otherwise FALSE.
*)

BEGIN
WHILE len > 0 DO   (* search backwards looking for . *)
   DEC(len);
   IF fname[len] = '.' THEN
      RETURN TRUE
   END;
END;
RETURN FALSE;
END ExplicitExt;

(**********************************************************************)

PROCEDURE GetCardinal (VAR n : CARDINAL);

(* If current value represents a positive integer then return via n. *)

BEGIN
IF (SReadf(value,'%d',n) <> 1) OR (n <= 0) THEN
   WriteString('Bad -'); Write(option);
   WriteString(' value: '); WriteString(value); WriteLn;
   WriteString('Specify a positive integer.'); WriteLn;
   RestoreTerminal; HALT;
END;
END GetCardinal;

(**********************************************************************)

PROCEDURE GetDimension (VAR r : REAL;
                        VAR un : units);

(* A valid dimension consists of a positive integer or real number followed
   by a two-letter unit: IN, CM, MM, PC, PT or PX (or in lowercase).
   If current value represents a valid dimension, we return number part in r
   and units part in un.
*)

VAR i : INTEGER;   ch1, ch2 : CHAR;

BEGIN
(* extract un *)
IF length > 1 THEN i := length-1 ELSE i := 1 END;
IF    (CAP(value[i-1]) = 'I') AND (CAP(value[i]) = 'N') THEN
   un := ic;
ELSIF (CAP(value[i-1]) = 'C') AND (CAP(value[i]) = 'M') THEN
   un := cm;
ELSIF (CAP(value[i-1]) = 'M') AND (CAP(value[i]) = 'M') THEN
   un := mm;
ELSIF (CAP(value[i-1]) = 'P') AND (CAP(value[i]) = 'C') THEN
   un := pc;
ELSIF (CAP(value[i-1]) = 'P') AND (CAP(value[i]) = 'T') THEN
   un := pt;
ELSIF (CAP(value[i-1]) = 'P') AND (CAP(value[i]) = 'X') THEN
   un := px;
ELSE
   WriteString('Bad units in -'); Write(option);
   WriteString(' value: '); WriteString(value); WriteLn;
   WriteString('Last two letters should be IN, CM, MM, PC, PT or PX.');
   WriteLn; RestoreTerminal; HALT;
END;
ch1 := value[i-1];                           (* remember letters in units *)
ch2 := value[i];
value[i]   := 0C;                            (* remove units *)
value[i-1] := 0C;
IF (SReadf(value,'%f',r) <> 1) OR (r <= 0.0) THEN
   value[i-1] := ch1;                        (* restore units *)
   value[i]   := ch2;
   WriteString('Bad -'); Write(option);
   WriteString(' value: '); WriteString(value); WriteLn;
   WriteString('Specify a positive dimension.'); WriteLn;
   RestoreTerminal; HALT;
END;
END GetDimension;

(**********************************************************************)

PROCEDURE InitSysInterface;

(* If an option appears more than once then we return the last value. *)

VAR i, num : CARDINAL;
    unitwd, unitht : units;
    realwd, realht : REAL;
    len : INTEGER;

BEGIN
(* SYSDEP: initialize option values with defaults *)
resolution := 300;                           (* LaserWriter resolution *)
unitwd     := ic;                            (* paper dimensions in inches *)
unitht     := ic;
realwd     := 8.3;                           (* A4 paper is 8.3" wide *)
realht     := 11.7;                          (* A4 paper is 11.7" high *)
mag        := 0;                             (* use DVI mag *)
GetEnvironment('TERM',vdu,len);              (* set vdu to value of TERM *)
IF len < 0 THEN vdu := 'unknown' END;
dummyfont  := '/usr/lib/tex/fonts/cmr10.1500pxl';
fontdir    := '/usr/lib/tex/fonts';
helpname   := '/usr/lib/tex/dv/dvitovdu.hlp';
DVIname[0] := 0C;                            (* SYSDEP: empty string *)
num := 1;
WHILE num < NumParameters DO
   GetParameter(num,value,length);
   INC(num);
   IF value[0] = '-' THEN
      IF length > 1 THEN option := value[1] ELSE option := ' ' END;
      IF (option <> 'r') AND (option <> 'm') AND (option <> 'x') AND
         (option <> 'y') AND (option <> 'v') AND (option <> 'd') AND
         (option <> 'h') AND (option <> 'f') THEN
         WriteString('Unknown option: -'); Write(option); WriteLn;
         RestoreTerminal; HALT;
      END;
      (* allow things like -vvis500 *)
      IF length > 2 THEN                 (* shift value left 2 places *)
         FOR i := 0 TO length - 1 DO
            value[i] := value[i+2];
         END;
         length := length - 2;
         value[length] := 0C;            (* SYSDEP: terminate with NULL *)
      ELSE                               (* value is next parameter *)
         GetParameter(num,value,length);
         INC(num);
         IF length <= 0 THEN
            WriteString('Missing value after -'); Write(option); WriteLn;
            RestoreTerminal; HALT;
         END;
      END;
      CASE option OF
        'r' : GetCardinal(resolution);
      | 'm' : GetCardinal(mag);
      | 'x' : GetDimension(realwd,unitwd);
      | 'y' : GetDimension(realht,unitht);
      | 'v' : vdu       := value;
      | 'd' : dummyfont := value;
      | 'h' : helpname  := value;
      | 'f' : fontdir   := value;
              (* bad string values will be detected in other modules *)
      END;
   ELSE
      DVIname := value;
      IF NOT ExplicitExt(DVIname,length) THEN   (* append .dvi *)
         IF length + 3 <= HIGH(DVIname) THEN
            DVIname[length]   := '.';
            DVIname[length+1] := 'd';
            DVIname[length+2] := 'v';
            DVIname[length+3] := 'i';
            IF length + 4 <= HIGH(DVIname) THEN   (* SYSDEP: append NULL *)
               DVIname[length + 4] := 0C;
            END;
         ELSE   (* user has given a mighty long file name! *)
            WriteString('DVI file name too long: ');
            WriteString(DVIname); WriteLn;
            RestoreTerminal; HALT;
         END;
      END;
      (* bad DVIname will be detected upon open in main module *)
   END;
END;
IF DVIname[0] = 0C THEN   (* no file name on command line *)
   WriteString('Missing DVI file name!'); WriteLn;
   RestoreTerminal; HALT;
END;
(* set paperwd and paperht only after resolution has been decided *)
CASE unitwd OF
   ic : paperwd := TRUNC(realwd * FLOAT(resolution) + 0.5) |
   cm : paperwd := TRUNC((realwd / 2.54) * FLOAT(resolution) + 0.5) |
   mm : paperwd := TRUNC((realwd / 25.4) * FLOAT(resolution) + 0.5) |
   pt : paperwd := TRUNC((realwd / 72.27) * FLOAT(resolution) + 0.5) |
   pc : paperwd := TRUNC((realwd / 72.27) * 12.0 * FLOAT(resolution) + 0.5) |
   px : paperwd := TRUNC(realwd + 0.5)
END;
CASE unitht OF
   ic : paperht := TRUNC(realht * FLOAT(resolution) + 0.5) |
   cm : paperht := TRUNC((realht / 2.54) * FLOAT(resolution) + 0.5) |
   mm : paperht := TRUNC((realht / 25.4) * FLOAT(resolution) + 0.5) |
   pt : paperht := TRUNC((realht / 72.27) * FLOAT(resolution) + 0.5) |
   pc : paperht := TRUNC((realht / 72.27) * 12.0 * FLOAT(resolution) + 0.5) |
   px : paperht := TRUNC(realht + 0.5)
END;
END InitSysInterface;

END sysinterface.
