/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This is the low-level device driver for the imPRESS language
 *	(IMAGEN laser printers).
 */

#include "types.h"
#include "drivutil.h"
#include "font.h"
#include "im.h"
#include "tracedef.h"

#include "arith.p"
#include "fio.p"
#include "font.p"
#include "graph.p"
#include "raster.p"
#include "strng.p"

/*
 *	The following definitions governs conditional use of obsolete
 *	imPRESS commands:
 */

#define EXPLICIT_DELETE  1	/* Use DELG to delete glyphs */

/*
 *	Some useful macros:
 */

#define Out_Byte(c) Write_Character_M (c, IM_File)
#define Out_String(str) Write_Block_M (str, sizeof(str)-1, IM_File)
#define Out_Scratch Write_Block_M (Scratch_String, strlen (Scratch_String), IM_File)
#define Out_Word(w) Out_Byte (((w) >> 8) & 0xFF); Out_Byte ((w) & 0xFF)
#define Round_Pixel(size) (short) (((size) + ((size) >= 0 ? 512 : -512)) >> 10)
#define Trunc_Pixel(size) (short) ((size) >> 10)
#define Ceil_Pixel(size)  (short) (((size) + ((size) >= 0 ? 1023 : -1023)) >> 10)
extern int strlen();

/*
 *	The 'Patchwork' structure list is used to keep track of bitmap
 *	descriptions. These bitmap descriptions are cached to allow
 *	the combination of two bitmaps into one, reducing the amount
 *	of material sent to the output file.
 */

struct Patchwork {
	struct Patchwork *Link;	/* Pointer to next one */
	struct Patchwork *BLink; /* Pointer to previous one */
	short Patch_X, Patch_Y; /* (h,v) of upper left corner */
	unsigned short Patch_Width, Patch_Height; /* Raster size, in bits */
	unsigned char Patch_Raster[]; /* The raster description */
} *IM_Patch_Head;

char Scratch_String[256];
pointer IM_File;		/* Output file pointer */
long IM_VMSize;			/* Amount of memory available */
long IM_Save_VMSize;		/* Saved amount of memory available */
long IM_Max_X, IM_Max_Y; 	/* Maximum x,y values for page, for checking if things go off the page */
short IM_Max_Dev_X, IM_Max_Dev_Y; /* Maximum x,y values in pixels */
short IM_X, IM_Y;		/* Current x, y */
unsigned short IM_Pen_Size;	/* Current pen diameter */
unsigned short IM_Next_Fam_Mem;	/* Next available Family/Member for downloading */
unsigned short IM_Save_Fam_Mem;	/* Saved Family/Member value at beginning of page */
unsigned char IM_Mem_Used[96];	/* Array tracking how many chars used in each family */
unsigned char IM_Save_Mem_Used[96];/* Saved version of above */
unsigned char IM_Current_Family;/* Current family number */
unsigned short IM_Flags;	/* Various flags */
#define IM_LANDSCAPE 0x01

/*
 *	Routine IM_Set_Layout opens the output file, initializes the device
 *	and sets the proper orientation (portrait/landscape):
 */

int IM_Set_Layout (Widest_Width, Tallest_Height, Layout_Mode, Two_Sided,
		   N_Fonts, Max_Size, Max_Primitives, Memory_Size, Max_Fonts,
		   Resolution, Page_Size, Copies, Input_Name, Output_Name)
unsigned long Widest_Width, Tallest_Height, N_Fonts, Max_Size,
	      Max_Primitives, Memory_Size, Max_Fonts, Layout_Mode,
	      Two_Sided;
struct Ratio *Resolution, Page_Size[2];
unsigned short Copies;
char *Input_Name, *Output_Name;
{
	auto   int Index;
	auto   unsigned short Page_Width, Page_Height;
/*
 *	Construct file name; open the output file:
 */
	stringcpy_m (Scratch_String, Input_Name, sizeof (Scratch_String));
	stringcat_m (Scratch_String, ".IM", sizeof (Scratch_String));
	if ((IM_File = Open_File_M ((Output_Name == 0) ? "" : Output_Name,
				     Scratch_String, "w", 0)) == 0)
		return (0);
/*
 *	Set flags word:
 */
	IM_Flags = 0;
	if (Layout_Mode != 0)
		IM_Flags |= IM_LANDSCAPE;
/*
 *	Initialize font control data, memory size:
 */
	for (Index = 0; Index < 96; Index++)
		IM_Mem_Used[Index] = 128;	/* Is this necessary? */
	IM_Next_Fam_Mem = TEXTURE_FAM_MEM - 1;
	IM_VMSize = Memory_Size << 10;
/*
 *	Output initial @document() stuff. This includes papersize,
 *	number of copies, document language:
 */
	Page_Width = XN_Div_D_R_M (Page_Size[0].Numerator, Resolution->Numerator,
				   Page_Size[0].Denominator * Resolution->Denominator);
	Page_Height = XN_Div_D_R_M (Page_Size[1].Numerator, Resolution->Numerator,
				    Page_Size[1].Denominator * Resolution->Denominator);
	sprintf (Scratch_String,
		 "@document(language imPRESS, papermargin zero, name \"%s\", copies %u, paperwidth %u, paperheight %u)",
		 Input_Name, Copies, Page_Width, Page_Height);
	Out_Scratch;
/*
 *	Initially, set the coordinate system so that the h,v axes align
 *	with the x,y axes and the main advance direction is along the
 *	h,v axes (the normal coordinate system). If the mode is landscape,
 *	move the origin to the top right corner of the physical page and
 *	rotate the h axes so it aligns with the y axis:
 */
	Out_Byte (SET_HV_SYSTEM);
	Out_Byte ((ORIGIN_TOP_LEFT << 5) | (AXES_POSITIVE << 3) | ORIENT_PHY_0);
	Out_Byte (PAGE);
	if ((IM_Flags & IM_LANDSCAPE) != 0) {
		Out_Byte (SET_ABS_H); Out_Word (Page_Width);
		Out_Byte ((ORIGIN_PHYSICAL << 5) | (AXES_POSITIVE << 3) | ORIENT_PHY_90);
		Out_Byte (PAGE);
		IM_Max_Dev_X = Page_Height;
		IM_Max_Dev_Y = Page_Width;
	} else {
		IM_Max_Dev_X = Page_Width;
		IM_Max_Dev_Y = Page_Height;
	}
	IM_Max_X = IM_Max_Dev_X << 10;
	IM_Max_Y = IM_Max_Dev_Y << 10;
	Out_Byte (SET_ADV_DIRS); Out_Byte (ROTATE_0 << 1);
	Out_Byte (SET_PUM); Out_Byte (0);
	Out_Byte (SET_PUSH_MASK); Out_Word (0x01FF);
	IM_Pen_Size = 1;
	Init_GState (&Current_State, IM_Pen_Size);
	IM_X = 0; IM_Y = 0;
	IM_Current_Family = 255;
	return (1);
}

/*
 *	Routine IM_Download_Font downloads a font to the IMAGEN.
 *	Limitations are 1) we cannot download more than 12287
 *	characters and 2) character sizes are limited to 16383
 *	pixels in advance width, width, height, x offset and
 *	y offset. Landscape mode requires that glyphs be provided
 *	rotated 90 degrees from the normal orientation (with a
 *	corresponding change in width, height and origin).
 *
 *	If a character is successfully downloaded, the 'Driver_Id'
 *	value for that character contains the character advance
 *	width in the high order 16 bits and the character's
 *	family/member number in the low order 16 bits. Family/member
 *	numbers will never be zero, so the Driver_Id will likewise
 *	never be zero.
 */

IM_Download_Font (Font_Ptr, Char_Vector, N_Chars)
struct Font_Definition *Font_Ptr;
struct Char_Definition **Char_Vector;
unsigned int N_Chars;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned char *Raster1, *Raster2;
	auto   int Index;
	auto   unsigned short Fam_Mem;
	auto   short Advance_Width;
	extern char *Mem_Alloc();
	extern unsigned long IM_Download_Glyph();
	extern unsigned short IM_Get_Next_Fam_Mem();
/*
 *	Allocate enough space to store 2 complete unpacked rasters
 *	(used for landscaping only):
 */
	if ((IM_Flags & IM_LANDSCAPE) != 0) {
		Raster1 = (unsigned char *) Mem_Alloc (Font_Ptr->Max_Unpacked);
		Raster2 = (unsigned char *) Mem_Alloc (Font_Ptr->Max_Unpacked);
	}
/*
 *	Download each character in the font:
 */
	Fam_Mem = 0;
	for (Index = 0; Index < N_Chars; Index++) {
		Char_Ptr = Char_Vector[Index];
		if (IM_VMSize <= 0 ||
		    Char_Ptr->Pixel_Width == 0 || Char_Ptr->Pixel_Height == 0 ||
		    Char_Ptr->Pixel_Width > 16383 || Char_Ptr->Pixel_Height > 16383 ||
		    Char_Ptr->X_Origin > 16383 || Char_Ptr->X_Origin < -16384 ||
		    Char_Ptr->Y_Origin > 16383 || Char_Ptr->Y_Origin < -16384 ||
		    (Advance_Width = Round_Pixel (Char_Ptr->H_Escapement)) > 16383 ||
		    Advance_Width < -16384 || (Fam_Mem = IM_Get_Next_Fam_Mem ()) == 0) {
			Char_Ptr->Driver_Id = 0;
			continue;
		}
		IM_VMSize -= IM_Download_Glyph (Fam_Mem, Char_Ptr, Advance_Width, Raster1, Raster2,
						Font_Ptr->Flags & TEMP_FONT);
		Char_Ptr->Driver_Id = (Advance_Width << 16) | Fam_Mem;
	}
	IM_Reset_Family (Fam_Mem);
	if ((IM_Flags & IM_LANDSCAPE) != 0) {
		Mem_Free (Raster1);
		Mem_Free (Raster2);
	}
}

unsigned long IM_Download_Glyph (Fam_Mem, Char_Ptr, Advance_Width, Scratch1, Scratch2, Temp_Font)
unsigned short Fam_Mem;
struct Char_Definition *Char_Ptr;
short Advance_Width;
unsigned char *Scratch1, *Scratch2;
unsigned char Temp_Font;
{
	auto   unsigned long Raster_Size;
	auto   int Index;
	auto   unsigned short Glyph_Identifier;
	auto   short S_Word;

	Out_Byte (BGLY);
	if ((IM_Flags & IM_LANDSCAPE) == 0) {
		Glyph_Identifier = Fam_Mem;
		Out_Word (Glyph_Identifier);
		Out_Word (Advance_Width);
		Out_Word (Char_Ptr->Pixel_Width);
		Out_Word (Char_Ptr->X_Origin);
		Out_Word (Char_Ptr->Pixel_Height);
		Out_Word (Char_Ptr->Y_Origin);
		Raster_Size = ((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height;
		Write_Block_M (Char_Ptr->Pixel_Array, Raster_Size, IM_File);
	} else {
		Unpack_Raster_Data_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
				      Char_Ptr->Pixel_Array, Scratch1);
		Rotate_Raster_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
				 Scratch1, Scratch2, 3);
		Pack_Raster_Data_M (Char_Ptr->Pixel_Height, Char_Ptr->Pixel_Width,
				    Scratch2, Scratch1);
		Glyph_Identifier = (ROTATE_90 << 14) | Fam_Mem;
		Out_Word (Glyph_Identifier);
		Out_Word (Advance_Width);
		Out_Word (Char_Ptr->Pixel_Height);
		S_Word = Char_Ptr->Pixel_Height - Char_Ptr->Y_Origin; Out_Word (S_Word);
		Out_Word (Char_Ptr->Pixel_Width);
		Out_Word (Char_Ptr->X_Origin);
		Raster_Size = ((Char_Ptr->Pixel_Height + 7) >> 3) * Char_Ptr->Pixel_Width;
		Write_Block_M (Scratch1, Raster_Size, IM_File);
	}
/*
 *	If the font is temporary, mark the glyph as disposible:
 */
#if EXPLICIT_DELETE
	if (Temp_Font != 0) {
		Out_Byte (DELG);
		Out_Word (Glyph_Identifier);
	}
#endif
	return (Raster_Size);
}

/*
 *	Routine IM_Get_Next_Fam_Mem returns the next Family/Member
 *	number that can be used for identifying a character. It returns
 *	the value zero if no more values are available. The idea here
 *	is to keep characters that are within the same TeX font within
 *	the same IMAGEN family, so as to not have to switch families too
 *	often. However, if all normally available slots are full, we have
 *	to go back and see if there are any vacant slots in any of the
 *	previously loaded families. When they are all used up, that's it.
 *
 *	The array IM_Mem_Used keeps track of the last used character
 *	in the respective family.
 *
 *	The usage of characters within families could, for example,
 *	look like this (if only 8 families and 64 character per family)
 *	after downloading 4 fonts:
 *
 *		+----------------------------------------------------------------+
 *	fam 0	|$                                                               |
 *		+----------------------------------------------------------------+
 *	fam 1	|                                                                |
 *		+----------------------------------------------------------------+
 *	fam 2	|                                                               #|
 *		+----------------------------------------------------------------+
 *	fam 3	|                         @**************************************| (font 4)
 *		+----------------------------------------------------------------+
 *	fam 4	|         @******************************************************| (font 3)
 *		+----------------------------------------------------------------+
 *	fam 5	|                                                      @*********| (font 2, continued)
 *		+----------------------------------------------------------------+
 *	fam 6	|****************************************************************| (font 2)
 *		+----------------------------------------------------------------+
 *	fam 7	|                                      @************************%| (font 1)
 *		+----------------------------------------------------------------+
 *
 *	The '$' is the reserved (0,0) position, which is not used, and
 *	'%' is the reserved texture glyph position. The '#' is the next
 *	available slot in normal sequence and the '@'s are the last-used
 *	character in each font. Sequencing is always done from the
 *	highest to the lowest.
 */

unsigned short IM_Get_Next_Fam_Mem ()
{
	auto   int Index;
	auto   unsigned short Fam_Mem;

	if ((Fam_Mem = IM_Next_Fam_Mem) > 0) {
		IM_Next_Fam_Mem--;
		IM_Mem_Used[Fam_Mem>>7] = Fam_Mem & 0x7F;
	} else {
		for (Index = 95; Index >= 0; Index--)
		if (IM_Mem_Used[Index] > 0) {
			Fam_Mem = (Index << 7) + --IM_Mem_Used[Index];
			break;
		}
	}
	return (Fam_Mem);
}

/*
 *	Routine IM_Reset_Family adjusts the next available Family/Member
 *	number to start at the next family. This is done after a font has
 *	been completely downloaded.
 */

IM_Reset_Family (Fam_Mem)
unsigned short Fam_Mem;
{
	auto   unsigned short Family;

	if (Fam_Mem == IM_Next_Fam_Mem + 1)
		IM_Next_Fam_Mem = ((Family = Fam_Mem >> 7) == 0) ? 0 : ((Family - 1) << 7) + 127;
}

/*
 *	Routine IM_Setup_New_Page is responsible for saving the
 *	current state of things and initializing for a new page:
 */

IM_Setup_New_Page (Page_Number)
long Page_Number[10];
{
	auto   int Index;

	Out_Byte (PAGE);
	IM_X = 0; IM_Y = 0;
	IM_Patch_Head = 0;
	for (Index = 0; Index < 96; Index++)
		IM_Save_Mem_Used[Index] = IM_Mem_Used[Index];
	IM_Save_Fam_Mem = IM_Next_Fam_Mem;
	IM_Save_VMSize = IM_VMSize;
}

/*
 *	Routine IM_Typeset_String outputs the characters specified
 *	in the character list. We must be careful here to make sure
 *	that the current (h,v) location is carefully maintained,
 *	since the rounding of the character widths to the nearest
 *	pixel can result in gradual divergence of the required
 *	(h,v) position from the actual position. We must also make sure
 *	that characters that are outside the page boundaries are not
 *	output, since the IMAGEN outputs error messages if this occurs.
 */

IM_Typeset_String (X, Y, Font_Ptr, Char_Vector, N_Chars)
long X, Y;
struct Font_Definition *Font_Ptr;
unsigned int N_Chars;
struct Char_Definition **Char_Vector;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   long Cur_X;
	auto   int Index;
	auto   short Advance_Width;
	auto   unsigned char Family, Member;

	Cur_X = X;
	IM_Move_Y (Round_Pixel (Y));
	for (Index = 0; Index < N_Chars; Index++) {
		Char_Ptr = Char_Vector[Index];
		IM_Move_X (Round_Pixel (Cur_X));
		if (Clip_Character_M (IM_X, IM_Y, Char_Ptr, 0, IM_Max_Dev_X, 0, IM_Max_Dev_Y) != 0) {
			Advance_Width = (long) Char_Ptr->Driver_Id >> 16;
			Family = (Char_Ptr->Driver_Id >> 7) & 0x7F;
			Member = Char_Ptr->Driver_Id & 0x7F;
			if (Family != IM_Current_Family) {
				Out_Byte (SET_FAMILY);
				Out_Byte (Family);
				IM_Current_Family = Family;
			}
			Out_Byte (Member);
			IM_X += Advance_Width;
		}
		Cur_X += Char_Ptr->H_Escapement;
	}
}

/*
 *	Routine IM_Typeset_Pixel outputs a bitmap image at a particular
 *	(x,y) position. Requirements of the IMAGEN are that we pad the
 *	image on the top, bottom, left and right so that an integer multiple
 *	of 'patches' (32 x 32 bit arrays), in both directions, is present,
 *	aligned on 32 bit boundaries. The image also needs to be clipped to
 *	the page bounds.
 */

IM_Typeset_Pixel (X, Y, Width, Height, Pixel_Array)
long X, Y;
unsigned short Width, Height;
unsigned char *Pixel_Array;
{
	auto   struct Patchwork *Patch_Ptr;
	auto   unsigned short Wid, Hgt;
	auto   short Dev_X, Dev_Y;
	extern struct Patchwork *IM_Create_Patch();

	if (Width == 0 || Height == 0)
		return;
/*
 *	Clip the image, if necessary; create the patch; merge the
 *	patch with the current patch cache:
 */
	Dev_X = Round_Pixel (X);
	Dev_Y = Round_Pixel (Y);
	Wid = (Dev_X + (short) Width <= IM_Max_Dev_X) ? Width : IM_Max_Dev_X - Dev_X;
	Hgt = (Dev_Y + (short) Height <= IM_Max_Dev_Y) ? Height : IM_Max_Dev_Y - Dev_Y;
	Patch_Ptr = IM_Create_Patch (Dev_X, Dev_Y, Width, Hgt, Wid, Pixel_Array);
	IM_Merge_Patch (Patch_Ptr);
}

/*
 *	Routine IM_Create_Patch creates a patch structure. It makes
 *	sure that the patch is an even multiple of 32 bits in x and
 *	y, then fills in the structure.
 */

struct Patchwork *IM_Create_Patch (Dev_X, Dev_Y, Width, Height, Trunc_Width, Pixel_Array)
short Dev_X, Dev_Y;
unsigned short Width, Height, Trunc_Width;
unsigned char *Pixel_Array;
{
	auto   struct Patchwork *Patch_Ptr;
	auto   unsigned short Wid, Hgt;
	auto   short X_Base, Y_Base;
	extern char *Mem_Alloc();

	X_Base = Dev_X & ~0x1F;
	Y_Base = Dev_Y & ~0x1F;
	Wid = (Dev_X - X_Base + Trunc_Width + 31) & ~0x1F;
	Hgt = (Dev_Y - Y_Base + Height + 31) & ~0x1F;
	Patch_Ptr = (struct Patchwork *) Mem_Alloc (sizeof (struct Patchwork) + (Wid >> 3) * Hgt);
	Patch_Ptr->Patch_X = X_Base;
	Patch_Ptr->Patch_Y = Y_Base;
	Patch_Ptr->Patch_Width = Wid;
	Patch_Ptr->Patch_Height = Hgt;
	Clear_Memory_M (Patch_Ptr->Patch_Raster, (Wid >> 3) * Hgt);
	Combine_Raster_M (Patch_Ptr->Patch_Raster, Wid, Hgt, Pixel_Array, Width, Height, Trunc_Width,
			  Dev_X-X_Base, Dev_Y-Y_Base, 1);
	return (Patch_Ptr);
}

/*
 *	Routine IM_Merge_Patch merges a new patch into the existing
 *	patch cache. If the new patch can be combined with an existing
 *	patch, this is done. Only patches that are (possibly proper)
 *	subsets of other patches are considered, since it is relatively
 *	inexpensive to output two mutually exclusive, but adjacent,
 *	patches as opposed to combining them into one longer or wider
 *	patch. Also, it is possible that patches may overlap in an
 *	astonishly large number of complex ways, optimization of which
 *	is beyond the scope of the intent of the patch cache.
 */

IM_Merge_Patch (New_Patch_Ptr)
struct Patchwork *New_Patch_Ptr;
{
	auto   struct Patchwork *New_Patch, *Patch_Ptr, *Temp_Patch;
	auto   int Match;
	auto   short Max_New_X, Max_New_Y, Max_X, Max_Y;
	extern struct Patchwork *IM_Create_Patch(), *IM_Combine_Patch();

	New_Patch = New_Patch_Ptr;
	Max_New_X = New_Patch->Patch_X + New_Patch->Patch_Width;
	Max_New_Y = New_Patch->Patch_Y + New_Patch->Patch_Height;
/*
 *	First, search for a patch that completely contains the
 *	new patch, then merge the new patch with it. This can occur
 *	at most once, with no further overlaps possible if it does.
 */
	for (Patch_Ptr = IM_Patch_Head; Patch_Ptr != 0; Patch_Ptr = Patch_Ptr->Link) {
		Max_X = Patch_Ptr->Patch_X + Patch_Ptr->Patch_Width;
		Max_Y = Patch_Ptr->Patch_Y + Patch_Ptr->Patch_Height;
		if (Patch_Ptr->Patch_X <= New_Patch->Patch_X && Max_New_X <= Max_X &&
		    Patch_Ptr->Patch_Y <= New_Patch->Patch_Y && Max_New_Y <= Max_Y) {
			Combine_Raster_M (Patch_Ptr->Patch_Raster, Patch_Ptr->Patch_Width,
					  Patch_Ptr->Patch_Height, New_Patch->Patch_Raster,
					  New_Patch->Patch_Width, New_Patch->Patch_Height,
					  New_Patch->Patch_Width, New_Patch->Patch_X - Patch_Ptr->Patch_X,
					  New_Patch->Patch_Y - Patch_Ptr->Patch_Y, 1);
			Mem_Free (New_Patch);
			return;
		}
	}
/*
 *	Next, scan for all patches that are contained within the
 *	new patch, merging them in. There can be an arbitrary number
 *	of these.
 */
	for (Patch_Ptr = IM_Patch_Head; Patch_Ptr != 0; Patch_Ptr = Patch_Ptr->Link) {
		Max_X = Patch_Ptr->Patch_X + Patch_Ptr->Patch_Width;
		Max_Y = Patch_Ptr->Patch_Y + Patch_Ptr->Patch_Height;
		if (New_Patch->Patch_X <= Patch_Ptr->Patch_X && Max_X <= Max_New_X &&
		    New_Patch->Patch_Y <= Patch_Ptr->Patch_Y && Max_Y <= Max_New_Y) {
			Combine_Raster_M (New_Patch->Patch_Raster, New_Patch->Patch_Width,
					  New_Patch->Patch_Height, Patch_Ptr->Patch_Raster,
					  Patch_Ptr->Patch_Width, Patch_Ptr->Patch_Height,
					  Patch_Ptr->Patch_Width, Patch_Ptr->Patch_X - New_Patch->Patch_X,
					  Patch_Ptr->Patch_Y - New_Patch->Patch_Y, 1);
			if ((Patch_Ptr->BLink->Link = Patch_Ptr->Link) != 0)
				Patch_Ptr->Link->BLink = Patch_Ptr->BLink;
			Temp_Patch = Patch_Ptr->BLink;
			Mem_Free (Patch_Ptr);
			Patch_Ptr = Temp_Patch;
		}
	}
/*
 *	Next, search for all patches that have the same y origin and
 *	height and overlap in x (or the same x origin and width that
 *	overlap in y), creating a new patch that contains both patches.
 *	At each step, there can be at most two such patches in the
 *	patch cache. Note that neither can be contained within each
 *	other, since that case was handled above.
 *
 *	This particular process can be extremely expensive to do if
 *	the patch organization is pathological. For example, consider
 *	the following structure:
 *
 *	+---------------------------------------+
 *	|					|
 *	+-------+-----------------------+-------+
 *	|	|			|	|
 *	|	+-------+-------+-------+	|
 *	|	|	|	|	|	|
 *	|	+-------+-------+-------+	|
 *	|	|			|	|
 *	+-------+-----------------------+-------+
 *	|					|
 *	+---------------------------------------+
 *
 *	with the patch in the center not yet defined (a hole).
 *	When the center patch is finally defined, all of the
 *	others will merge together, creating new, larger versions
 *	at each step. Under more normal circumstances, it is
 *	expected that, in general, patches will be constructed
 *	row-wise or column-wise, not randomly.
 */
	do {
		Match = 0;
		for (Patch_Ptr = IM_Patch_Head; Patch_Ptr != 0; Patch_Ptr = Patch_Ptr->Link) {
			Max_Y = Patch_Ptr->Patch_Y + Patch_Ptr->Patch_Height;
			Max_X = Patch_Ptr->Patch_X + Patch_Ptr->Patch_Width;
			if ((Max_Y == Max_New_Y && Patch_Ptr->Patch_Y == New_Patch->Patch_Y &&
			     ((New_Patch->Patch_X < Patch_Ptr->Patch_X && Max_New_X >= Patch_Ptr->Patch_X) ||
			      (Max_New_X > Max_X && New_Patch->Patch_X <= Max_X))) ||
			    (Max_X == Max_New_X && Patch_Ptr->Patch_X == New_Patch->Patch_X &&
			     ((New_Patch->Patch_Y < Patch_Ptr->Patch_Y && Max_New_Y >= Patch_Ptr->Patch_Y) ||
			      (Max_New_Y > Max_Y && New_Patch->Patch_Y <= Max_Y)))) {
				Temp_Patch = IM_Combine_Patch (Patch_Ptr, New_Patch);
				Mem_Free (New_Patch);
				New_Patch = Temp_Patch;
				if ((Patch_Ptr->BLink->Link = Patch_Ptr->Link) != 0)
					Patch_Ptr->Link->BLink = Patch_Ptr->BLink;
				Temp_Patch = Patch_Ptr->BLink;
				Mem_Free (Patch_Ptr);
				Patch_Ptr = Temp_Patch;
				Match++;
			}
		}
	} while (Match != 0);
/*
 *	Finally, link the resulting patch in to patch cache:
 */
	if ((New_Patch->Link = IM_Patch_Head) != 0)
		New_Patch->Link->BLink = New_Patch;
	IM_Patch_Head = New_Patch;
	New_Patch->BLink = (struct Patchwork *) &IM_Patch_Head;
}

/*
 *	Routine IM_Combine_Patch combines two patches into a single
 *	larger patch that contains them both.
 */

struct Patchwork *IM_Combine_Patch (Patch1, Patch2)
struct Patchwork *Patch1, *Patch2;
{
	auto   struct Patchwork *Patch_Ptr;
	auto   unsigned int Size;
	auto   short Max_1, Max_2, Low_X, High_X, Low_Y, High_Y;
	extern char *Mem_Alloc();

	Low_X = (Patch1->Patch_X < Patch2->Patch_X) ? Patch1->Patch_X : Patch2->Patch_X;
	Low_Y = (Patch1->Patch_Y < Patch2->Patch_Y) ? Patch1->Patch_Y : Patch2->Patch_Y;
	Max_1 = Patch1->Patch_X + Patch1->Patch_Width;
	Max_2 = Patch2->Patch_X + Patch2->Patch_Width;
	High_X = (Max_1 > Max_2) ? Max_1 : Max_2;
	Max_1 = Patch1->Patch_Y + Patch1->Patch_Height;
	Max_2 = Patch2->Patch_Y + Patch2->Patch_Height;
	High_Y = (Max_1 > Max_2) ? Max_1 : Max_2;
	Size = ((High_X - Low_X) >> 3) * (High_Y - Low_Y);
	Patch_Ptr = (struct Patchwork *) Mem_Alloc (sizeof (struct Patchwork) + Size);
	Patch_Ptr->Patch_X = Low_X;
	Patch_Ptr->Patch_Y = Low_Y;
	Patch_Ptr->Patch_Width = High_X - Low_X;
	Patch_Ptr->Patch_Height = High_Y - Low_Y;
	Clear_Memory_M (Patch_Ptr->Patch_Raster, Size);
	Combine_Raster_M (Patch_Ptr->Patch_Raster, Patch_Ptr->Patch_Width, Patch_Ptr->Patch_Height,
			  Patch1->Patch_Raster, Patch1->Patch_Width, Patch1->Patch_Height,
			  Patch1->Patch_Width, Patch1->Patch_X - Low_X, Patch1->Patch_Y - Low_Y, 1);
	Combine_Raster_M (Patch_Ptr->Patch_Raster, Patch_Ptr->Patch_Width, Patch_Ptr->Patch_Height,
			  Patch2->Patch_Raster, Patch2->Patch_Width, Patch2->Patch_Height,
			  Patch2->Patch_Width, Patch2->Patch_X - Low_X, Patch2->Patch_Y - Low_Y, 1);
	return (Patch_Ptr);
}

/*
 *	Routine IM_Typeset_Rule outputs a rule at a specified position.
 *	We must ensure the rule does not go outside the page bounds;
 *	if it does, it must be clipped at the top, bottom and/or right
 *	page boundaries.
 */

IM_Typeset_Rule (X, Y, Width, Height)
long X, Y;
unsigned long Width, Height;
{
	auto   unsigned short w, h;
	auto   short t;
	static unsigned long wh[2];
	static long xy[2];

	xy[0] = Round_Pixel (X);
	xy[1] = Round_Pixel (Y);
	wh[0] = Ceil_Pixel (Width);
	wh[1] = Ceil_Pixel (Height);
	if (Clip_Rule_M (xy, wh, 0, IM_Max_Dev_X, 0, IM_Max_Dev_Y) != 0) {
		w = (unsigned short) wh[0];
		h = (unsigned short) wh[1];
		t = -(short) h;
		IM_Move_X ((short) xy[0]);
		IM_Move_Y ((short) xy[1]);
		Out_Byte (BRULE); Out_Word (w); Out_Word (h); Out_Word (t);
	}
}

IM_Typeset_Filled (Vertex_Count, Poly_X, Poly_Y, Width, Height, Pixel_Array,
		   Do_Perimeter)
unsigned int Vertex_Count;
long Poly_X[], Poly_Y[];
unsigned short Width, Height;
unsigned char *Pixel_Array;
int Do_Perimeter;
{
	auto   long *Clipped_Poly_X, *Clipped_Poly_Y;
	auto   unsigned char *Scratch1 = 0, *Scratch2 = 0;
	auto   long Prev_X, Prev_Y;
	auto   unsigned int Index, Count;
	auto   short Word;
	static struct Char_Definition Char_Def;
	extern char *Mem_Alloc();
	extern unsigned long IM_Download_Glyph();
/*
 *	Clip the polygon, then create the polygon path:
 */
	Count = Vertex_Count * 2 + 4;
	Clipped_Poly_X = (long *) Mem_Alloc (Count * sizeof (long) * 2);
	Clipped_Poly_Y = &Clipped_Poly_X[Count];
	if ((Count = Clip_Polygon_M (Vertex_Count, Poly_X, Poly_Y, 0, IM_Max_X - 1024, 0, IM_Max_Y - 1024,
				     Clipped_Poly_X, Clipped_Poly_Y)) > 0) {
		Out_Byte (PUSH);
		Out_Byte (CREATE_PATH); Out_Word (Count);
		for (Index = 0; Index < Count; Index++) {
			Word = Round_Pixel (Clipped_Poly_X[Index]); Out_Word (Word);
			Word = Round_Pixel (Clipped_Poly_Y[Index]); Out_Word (Word);
		}
/*
 *	Download the texture glyph; set the texture pattern to that glyph:
 */
		if (Width == 0 || Height == 0) {	/* Solid, use glyph zero */
			Out_Byte (SET_TEXTURE);
			Out_Word (0);
		} else {
/*			Char_Def.Pixel_Width = Width;
			Char_Def.Pixel_Height = Height;		*/
			Char_Def.Pixel_Width = 32;
			Char_Def.Pixel_Height = 32;
			Char_Def.X_Origin = 0;
			Char_Def.Y_Origin = 0;
			Char_Def.Pixel_Array = Pixel_Array;
			if ((IM_Flags & IM_LANDSCAPE) != 0) {
				Scratch1 = (unsigned char *) Mem_Alloc (Width * Height);
				Scratch2 = (unsigned char *) Mem_Alloc (Width * Height);
			}
			IM_Download_Glyph (TEXTURE_FAM_MEM, &Char_Def, 0, Scratch1, Scratch2, 0);
			Out_Byte (SET_TEXTURE); Out_Word (TEXTURE_FAM_MEM);
			if ((IM_Flags & IM_LANDSCAPE) != 0) {
				Mem_Free (Scratch1);
				Mem_Free (Scratch2);
			}
		}
/*
 *	Fill the path with the texture:
 */
		Out_Byte (FILL_PATH); Out_Byte (GOP_OPAQUE);
		Out_Byte (POP);
#if EXPLICIT_DELETE
		if (Width != 0 && Height != 0) {
			Out_Byte (DELG);
			Out_Word (TEXTURE_FAM_MEM);
		}
#endif
/*
 *	If the perimeter is to be drawn, do that now:
 */
		if (Do_Perimeter != 0) {
			Prev_X = Poly_X[Vertex_Count-1];
			Prev_Y = Poly_Y[Vertex_Count-1];
			for (Index = 0; Index < Vertex_Count; Index++) {
				IM_Typeset_Line (Prev_X, Prev_Y, Poly_X[Index], Poly_Y[Index]);
				Prev_X = Poly_X[Index];
				Prev_Y = Poly_Y[Index];
			}
		}
	}
	Mem_Free (Clipped_Poly_X);
}

IM_Typeset_Line (X1, Y1, X2, Y2)
long X1, Y1, X2, Y2;
{
	auto   unsigned long Phase;
	auto   unsigned short Save_Pen_Size;
	extern int IM_Draw_Solid_Line();

	Save_Pen_Size = IM_Pen_Size;
	Out_Byte (PUSH);
	if (Current_State.Dashline_Len == 0)	/* Solid line */
		IM_Draw_Solid_Line (X1, Y1, X2, Y2, Current_State.Line_Width);
	else {
		Phase = ((Current_State.Flags & XY_DEFINED) != 0 && Current_State.Graphics_X == X1 &&
			 Current_State.Graphics_Y == Y1) ? Current_State.Scaled_Phase : 0;
		Current_State.Scaled_Phase = Segment_Line_M (X1, Y1, X2, Y2, Current_State.Dashline_Len,
							     Current_State.Scaled_Spec, Phase,
							     Current_State.Line_Width,
							     &IM_Draw_Solid_Line);
	}
	Out_Byte (POP);
	IM_Pen_Size = Save_Pen_Size;
	Current_State.Graphics_X = X2;
	Current_State.Graphics_Y = Y2;
	Current_State.Flags |= XY_DEFINED;
}

IM_Draw_Solid_Line (X1, Y1, X2, Y2, Width)
long X1, Y1, X2, Y2;
unsigned long Width;
{
	extern int IM_Draw_Line();

	if (Width <= MAX_PEN_SIZE << 10)
		IM_Draw_Line (X1, Y1, X2, Y2, Width);
	else {
		Build_Line_M (X1, Y1, X2, Y2, Width, MAX_PEN_SIZE << 10, &IM_Draw_Line);
		IM_Typeset_Point (X1, Y1, Width);
		IM_Typeset_Point (X2, Y2, Width);
	}
}

IM_Draw_Line (X1, Y1, X2, Y2, Width)
long X1, Y1, X2, Y2;
unsigned long Width;
{
	auto   unsigned short Pen_Size;
	auto   short Word;
	static long p1[2], p2[2];

	p1[0] = X1; p1[1] = Y1;
	p2[0] = X2; p2[1] = Y2;
	if (Clip_Line_M (p1, p2, 0, IM_Max_X - 1024, 0, IM_Max_Y - 1024) != 0) {
		if ((Pen_Size = Width >> 10) != IM_Pen_Size) {
			Out_Byte (SET_PEN); Out_Byte (Pen_Size);
			IM_Pen_Size = Pen_Size;
		}
		Out_Byte (CREATE_PATH); Out_Word (2);
		Word = Round_Pixel (p1[0]); Out_Word (Word);
		Word = Round_Pixel (p1[1]); Out_Word (Word);
		Word = Round_Pixel (p2[0]); Out_Word (Word);
		Word = Round_Pixel (p2[1]); Out_Word (Word);
		Out_Byte (DRAW_PATH); Out_Byte (GOP_BLACK);
	}
}

IM_Typeset_Point (X, Y, Diameter)
long X, Y;
unsigned long Diameter;
{
	auto   long Radius;
	auto   unsigned short U_Word;
	auto   short Save_X, Save_Y;

	Radius = (Diameter >> 1) & ~0x3FF;
	if (X >= Radius && X+Radius < IM_Max_X && Y >= Radius && Y+Radius < IM_Max_Y) {
		Save_X = IM_X; Save_Y = IM_Y;
		Out_Byte (PUSH);
		IM_Move_X (Round_Pixel (X)); IM_Move_Y (Round_Pixel (Y));
		Out_Byte (CIRC_ARC); U_Word = Radius >> 10; Out_Word (U_Word); Out_Word (0); Out_Word (16383);
		Out_Byte (FILL_PATH); Out_Byte (GOP_BLACK);
		Out_Byte (POP);
		IM_X = Save_X; IM_Y = Save_Y;
	}
}

IM_Typeset_Arc (X, Y, Radius, Start_Ang, Stop_Ang)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius;
{
	auto   unsigned long Phase;
	auto   long X0, Y0;
	auto   unsigned short Save_Pen_Size;
	auto   short Save_X, Save_Y;
	extern IM_Draw_Solid_Arc();

	Out_Byte (PUSH);
	Save_Pen_Size = IM_Pen_Size;
	Save_X = IM_X; Save_Y = IM_Y;
	if (Current_State.Dashline_Len == 0)	/* Solid line */
		IM_Draw_Solid_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Current_State.Line_Width);
	else {
		Compute_Arc_XY_M (X, Y, Radius, Start_Ang, &X0, &Y0);
		Phase = ((Current_State.Flags & XY_DEFINED) != 0 && Current_State.Graphics_X == X0 &&
			 Current_State.Graphics_Y == Y0) ? Current_State.Scaled_Phase : 0;
		Current_State.Scaled_Phase = Segment_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang,
							    Current_State.Dashline_Len,
							    Current_State.Scaled_Spec, Phase,
							    Current_State.Line_Width, &IM_Draw_Solid_Arc);
	}
	Compute_Arc_XY_M (X, Y, Radius, Stop_Ang, &Current_State.Graphics_X, &Current_State.Graphics_Y);
	Current_State.Flags |= XY_DEFINED;
	IM_Pen_Size = Save_Pen_Size;
	IM_X = Save_X; IM_Y = Save_Y;
	Out_Byte (POP);
}

IM_Draw_Solid_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width;
{
	auto   long X0, Y0;
	extern IM_Draw_Arc();

	if (Width <= MAX_PEN_SIZE << 10)
		IM_Draw_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width);
	else {
		Build_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang, Width, MAX_PEN_SIZE << 10, &IM_Draw_Arc);
		Compute_Arc_XY_M (X, Y, Radius, Start_Ang, &X0, &Y0);
		IM_Typeset_Point (X0, Y0, Width);
		Compute_Arc_XY_M (X, Y, Radius, Stop_Ang, &X0, &Y0);
		IM_Typeset_Point (X0, Y0, Width);
	}
}

IM_Draw_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width;
{
	auto   unsigned int Index, Count;
	auto   unsigned short Pen_Size, U_Word;
	auto   short Word;
	static long Arc_Start[4], Arc_Stop[4];

	if ((Count = Clip_Arc_M (X, Y, Radius, Start_Ang, Stop_Ang, 0, IM_Max_X - 1024, 0, IM_Max_Y - 1024,
				 Arc_Start, Arc_Stop)) > 0) {
		if ((Pen_Size = Width >> 10) != IM_Pen_Size) {
			Out_Byte (SET_PEN); Out_Byte (Pen_Size);
			IM_Pen_Size = Pen_Size;
		}
		IM_Move_X (Round_Pixel (X)); IM_Move_Y (Round_Pixel (Y));
		for (Index = 0; Index < Count; Index++) {
			Out_Byte (CIRC_ARC);
			U_Word = Radius >> 10; Out_Word (U_Word);
			Word = XN_Div_D_T_M (Arc_Start[Index], 1 << 14, 360000); Out_Word (Word);
			Word = XN_Div_D_R_M (Arc_Stop[Index], 1 << 14, 360000); Out_Word (Word);
			Out_Byte (DRAW_PATH); Out_Byte (GOP_BLACK);
		}
	}
}

IM_Set_Linewidth (Width)
unsigned long Width;
{
	auto   struct Graphics_State *State_Ptr;
	extern unsigned long Scale_Linestyle();

	State_Ptr = &Current_State;
	if (State_Ptr->Line_Width != Width) {
		State_Ptr->Line_Width = Width;
		State_Ptr->Scaled_Phase = Scale_Linestyle (State_Ptr->Dashline_Len, State_Ptr->Dashline_Spec,
							   State_Ptr->Line_Width, State_Ptr->Dashline_Phase,
							   State_Ptr->Scaled_Spec);
	}
}

IM_Set_Linestyle (Segment_Len, Segment_Desc, Start_Phase)
unsigned int Segment_Len, Segment_Desc[], Start_Phase;
{
	auto   struct Graphics_State *State_Ptr;
	auto   unsigned int Index;
	extern unsigned long Scale_Linestyle();
	extern int Compare_Linestyle();

	State_Ptr = &Current_State;
	if (Compare_Linestyle (Segment_Len, Segment_Desc, State_Ptr->Dashline_Len,
			       State_Ptr->Dashline_Spec) != 0) {
		State_Ptr->Dashline_Len = Segment_Len;
		for (Index = 0; Index < Segment_Len; Index++)
			State_Ptr->Dashline_Spec[Index] = Segment_Desc[Index];
		State_Ptr->Dashline_Phase = Start_Phase;
		State_Ptr->Scaled_Phase = Scale_Linestyle (State_Ptr->Dashline_Len, State_Ptr->Dashline_Spec,
							   State_Ptr->Line_Width, State_Ptr->Dashline_Phase,
							   State_Ptr->Scaled_Spec);
	}
}

IM_Set_Color (Red, Green, Blue)
unsigned int Red, Green, Blue;
{

}

IM_Save_Graphics (Identifier)
pointer Identifier;
{
	Save_GState (&Current_State, Identifier);
}

int IM_Restore_Graphics (Identifier)
pointer Identifier;
{
	extern int Restore_GState();

	return (Restore_GState (&Current_State, Identifier));
}

/*
 *	Routine IM_Eject_Page is responsible for outputting any cached
 *	patches, restoring saved states, then ending the page:
 */

IM_Eject_Page ()
{
	auto   struct Patchwork *Patch_Ptr;
	auto   int Index;

	while ((Patch_Ptr = IM_Patch_Head) != 0) {
		IM_Output_Patch (Patch_Ptr);
		IM_Patch_Head = Patch_Ptr->Link;
		Mem_Free (Patch_Ptr);
	}
	Out_Byte (ENDPAGE);
#if EXPLICIT_DELETE
	Out_Byte (FORCE_GLY_DELETE);
#endif
	for (Index = 0; Index < 96; Index++)
		IM_Mem_Used[Index] = IM_Save_Mem_Used[Index];
	IM_Next_Fam_Mem = IM_Save_Fam_Mem;
	IM_VMSize = IM_Save_VMSize;
}

/*
 *	Routine IM_Output_Patch outputs a bitmap descriptor to the
 *	output file:
 */

IM_Output_Patch (Patch_Ptr)
struct Patchwork *Patch_Ptr;
{
	auto   unsigned char *Ptr;
	auto   unsigned short w, h;

	IM_Move_X (Patch_Ptr->Patch_X);
	IM_Move_Y (Patch_Ptr->Patch_Y);
	Out_Byte (BITMAP);
	Out_Byte (GOP_OR);
	Out_Byte (Patch_Ptr->Patch_Width >> 5);
	Out_Byte (Patch_Ptr->Patch_Height >> 5);
	Ptr = &Patch_Ptr->Patch_Raster[0];
	for (h = Patch_Ptr->Patch_Height; h > 0; h--) {
		for (w = Patch_Ptr->Patch_Width >> 3; w > 0; w--)
			Out_Byte (*Ptr++);
	}
}

/*
 *	Routine IM_Terminate finishes off the output data (sends EOF
 *	char out, closes the file, and deallocates resources used
 *	herein:
 */

IM_Terminate ()
{
	Out_Byte (EOF);
	Close_File_M (IM_File);
}

/*
 *	Routines IM_Move_X and IM_Move_Y move the current position to
 *	the specified coordinate.
 */

IM_Move_X (X)
short X;
{
	auto   short Coord, Delta;

	if ((Coord = X) != IM_X) {
		if (Coord > 16383)
			Coord = 16383;
		else if (Coord < 0)
			Coord = 0;
		Delta = Coord - IM_X;
		if (Delta == 1)
			Out_Byte (MPLUS);
		else if (Delta == -1)
			Out_Byte (MMINUS);
		else {
			Out_Byte (SET_ABS_H);
			Out_Word (Coord);
		}
		IM_X = Coord;
	}
}

IM_Move_Y (Y)
short Y;
{
	auto   short Coord;

	if ((Coord = Y) != IM_Y) {
		if (Coord > 16383)
			Coord = 16383;
		else if (Coord < 0)
			Coord = 0;
		Out_Byte (SET_ABS_V);
		Out_Word (Coord);
		IM_Y = Coord;
	}
}
