/*
 *	This file contains the routines that handle input from the
 *	LaserWriter.
 */

#include "symbiont.h"
#include "ast.h"
#include opcdef

/*
 *	Routine Process_LW_Input is executed when input from the
 *	LaserWriter has been received.
 */

unsigned long Process_LW_Input (Dcb)
char *Dcb;
{
	auto   struct Thread *Thread_Ptr;
	auto   unsigned long Sys_Status;
	auto   unsigned short Length;
	extern struct Thread *Find_Thread_Using_Dcb();
	extern unsigned long Read_Line(), Process_LW_Message();
	globalvalue SS$_NORMAL, SS$_TIMEOUT;
/*
 *	Locate the associated thread; obtain lines from the LaserWriter
 *	and process them, until there are no more.
 */
	if ((Thread_Ptr = Find_Thread_Using_Dcb (Dcb)) == 0)
		Sys_Status = SS$_NORMAL;	/* (not really) */
	else {
		while ((Sys_Status = Read_Line (Thread_Ptr)) == SS$_NORMAL)
		if (Thread_Ptr->Input_Buffer[0] != '\0') {
			Sys_Status = Process_LW_Message (Thread_Ptr);
			if ((Sys_Status & 0x01) == 0)
				break;
		}
		if (Sys_Status == SS$_TIMEOUT)
			Sys_Status = SS$_NORMAL;
	}
	return (Sys_Status);
}

/*
 *	Routine Read_Line reads one message line from the device. The process
 *	of getting messages from the LaserWriter is rather complicated. When
 *	the symbiont is notified of unsolicited input, it is necessary to read
 *	the data, without hanging, from the LaserWriter. Since the VAX may
 *	be able to read data faster than the LaserWriter can generate it, we
 *	may come to the end of the input data before seeing the complete
 *	message. Therefore it is necessary to rathole any partial messages
 *	until a new interrupt is received (almost immediately) that more data
 *	is waiting. Also, it is possible for the process to occur in the
 *	reverse - the LaserWriter being faster than the Vax. In this case,
 *	more than one complete message may be waiting in the type-ahead.
 *
 *	Either way, it's hard.
 */

#include "iostatus.h"

unsigned long Read_Line (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   struct dsc$descriptor *Buf_Desc;
	auto   unsigned long Sys_Status;
	auto   int Length;
	auto   unsigned short Buffer_Size;
	auto   char *Buffer_Ptr;
	static struct Read_IO_Status_Block IO_Status;
	extern unsigned long Read_Device_Now();
	globalvalue SS$_NORMAL, SS$_TIMEOUT;
/*
 *	Set up for the read, then read more data into the buffer:
 */
	Buf_Desc = &Thread_Ptr->Input_Buffer_Desc;
	Buffer_Size = Buf_Desc->dsc$w_length;
	Buffer_Ptr = &Thread_Ptr->Input_Buffer[Buffer_Size];
	Buffer_Size = sizeof (Thread_Ptr->Input_Buffer) - Buffer_Size - 1;
	Sys_Status = Read_Device_Now (Buffer_Ptr, Buffer_Size, "\004\n", Thread_Ptr->Dcb);
	if (Sys_Status == SS$_NORMAL || Sys_Status == SS$_TIMEOUT) {
		Set_Device_Read_Status (Thread_Ptr->Dcb, &IO_Status);
		Buf_Desc->dsc$w_length += IO_Status.Byte_Count;
		if (Sys_Status == SS$_NORMAL) {
			if (IO_Status.Terminator == 4)	/* Control-D */
				EnQueue_AST_Locked (AST_CONTROL_D, Thread_Ptr, Thread_Ptr->Incarnation, PRI_CONTROL_D);
			Length = Buf_Desc->dsc$w_length;
			if (Length > 0 && Thread_Ptr->Input_Buffer[Length-1] == '\r')
				Length--;
			while (Length > 0 && Thread_Ptr->Input_Buffer[Length-1] == ' ')
				Length--;
			Thread_Ptr->Input_Buffer[Length] = '\0';
			while (Length > 0)
			if (Thread_Ptr->Input_Buffer[--Length] == '\0')
				Thread_Ptr->Input_Buffer[Length] = ' ';
			Buf_Desc->dsc$w_length = 0;
		}
	}
	return (Sys_Status);
}

/*
 *	Routine Process_LW_Message is called when an incoming message
 *	from the LaserWriter has been completely received:
 */

unsigned long Process_LW_Message (Thread_Ptr)
struct Thread *Thread_Ptr;
{
	auto   unsigned long Sys_Status, FAO_Status;
	auto   unsigned int Msg_Count, Page_Count;
	static struct dsc$descriptor Desc_Array[5], Key_Desc, Value_Desc;
	static struct dsc$descriptor FAO_Desc, Buf_Desc;
	static union {	/* forces OPC buffer large enough */
		struct OPC Opc_Buf;
		char Msg_Buf[88];
	} Opc_Msg;
	static char *FAO_String = "LaserWriter !AS (queue !AS) has reported a problem - !AS";
	static unsigned short Length;
	static char Key[20];
	extern unsigned long Str$Copy_DX(), Sys$FAO(), Sys$SndOpr();
	extern unsigned long Set_Timer(), Append_Output_String();
	extern unsigned int Parse_Error_Message();
	extern int Strip_Error_Message(), strcmp(), atoi();
	globalvalue SS$_NORMAL, SS$_DEVCMDERR;
/*
 *	Check if the output is in the error message form:
 */
	if (Strip_Error_Message (&Thread_Ptr->Input_Buffer[0], &Buf_Desc) == 0)
		Sys_Status = Append_Output_String (&Buf_Desc, &Thread_Ptr->Misc_Output_Head);
/*
 *	Parse the message and extract the message key and process accordingly.
 *	This mostly involves (possibly) cancelling I/O on the task, and
 *	setting up things to be printed on the trailer page(s).
 */
	else if ((Msg_Count = Parse_Error_Message (&Buf_Desc, &Desc_Array[0],
						   sizeof (Desc_Array) / sizeof (struct dsc$descriptor))) > 0) {
		Parse_Message_Code (&Desc_Array[0], &Key_Desc, &Value_Desc);
		Copy_Desc_to_String (&Key_Desc, &Key[0], sizeof (Key));
/*
 *	"Error" indicates an error in the PostScript - it can
 *	occur in library modules or the main file (it better not
 *	occur in our stuff!) - signal that no further attempts to
 *	output to device is necessary until Control-D is sent:
 */
		if (strcmp (&Key[0], "Error") == 0) {
			Sys_Status = Str$Copy_DX (&Thread_Ptr->Error_Name, &Value_Desc);
			if ((Sys_Status & 0x01) != 0 && Msg_Count > 1) {
				Parse_Message_Code (&Desc_Array[1], &Key_Desc, &Value_Desc);
				Sys_Status = Str$Copy_DX (&Thread_Ptr->Offending_Command, &Value_Desc);
			}
			Thread_Ptr->Task_Status |= TASK_M_USERERROR;
			Add_Condition_Value (SS$_DEVCMDERR, Thread_Ptr);
/*
 *	"Printing" is induced by a showpage or copypage command:
 */
		} else if (strcmp (&Key[0], "Printing") == 0) {
			Copy_Desc_to_String (&Value_Desc, &Key[0], sizeof (Key));
			if ((Page_Count = atoi (&Key[0])) < 0)	/* somebody screwing around */
				Page_Count = 1;
			Thread_Ptr->Accounting_Data.smbmsg$l_pages_printed += Page_Count;
			Thread_Ptr->Current_Page++;
			Sys_Status = SS$_NORMAL;
/*
 *	Notify Operator if a printer error is reported (jam,
 *	out of paper, etc.). Also, if we were waiting for a Control-D,
 *	re-issue the timeout wait for a longer period of time, to allow
 *	the operator to put in more paper, or whatever.
 */
		} else if (strcmp (&Key[0], "PrinterError") == 0) {
			Make_VMS_Descriptor (FAO_String, &FAO_Desc);
			Set_VMS_Descriptor (&Opc_Msg.Opc_Buf.opc$l_ms_text, sizeof (Opc_Msg) - 8, &Buf_Desc);
			FAO_Status = Sys$FAO (&FAO_Desc, &Length, &Buf_Desc, &Thread_Ptr->Device_Name,
					      &Thread_Ptr->Executor_Queue, &Value_Desc);
			if ((FAO_Status & 0x01) != 0) {
				Opc_Msg.Opc_Buf.opc$b_ms_type = OPC$_RQ_RQST;
				Opc_Msg.Opc_Buf.opc$b_ms_target = OPC$M_NM_PRINT;
				Set_VMS_Descriptor (&Opc_Msg, Length+8, &Buf_Desc);
				FAO_Status = Sys$SndOpr (&Buf_Desc, 0);
			}
			if ((Thread_Ptr->Task_Status & TASK_M_CONTROL_D_WAIT) != 0) {
				Cancel_Timer (Thread_Ptr);
				Sys_Status = Set_Timer (5*60*1000, Thread_Ptr);
			} else
				Sys_Status = SS$_NORMAL;
/*
 *	Ignore non-relevant messages:
 */
		} else if (strcmp (&Key[0], "Flushing") == 0 || strcmp (&Key[0], "exitserver") == 0)
			Sys_Status = SS$_NORMAL;
/*
 *	Otherwise, output should be saved for printing at the end
 *	of the job:
 */
		else
			Sys_Status = Append_Output_String (&Buf_Desc, &Thread_Ptr->Misc_Output_Head);
	}
	return (Sys_Status);
}

int Strip_Error_Message (String, Desc_Ptr)
char *String;
struct dsc$descriptor *Desc_Ptr;
{
	auto   char *Str_L, *Str_R, *Pat_L, *Pat_R;
	auto   int Str_Length;
	static char Pattern[] = "%%[  ]%%";
	extern int strlen();

	Str_Length = strlen (String);
	Str_L = String;
	Str_R = &String[Str_Length];
	Pat_L = Pattern;
	Pat_R = &Pattern[sizeof(Pattern)-1];
	while (Pat_R > Pat_L && Str_R > Str_L && *Str_L == *Pat_L && *--Str_R == *--Pat_R) {
		Str_L++;
		Pat_L++;
	}
	if (Pat_R == Pat_L && Str_L <= Str_R) {
		Set_VMS_Descriptor (Str_L, Str_R - Str_L, Desc_Ptr);
		return (1);
	}
	Set_VMS_Descriptor (String, Str_Length, Desc_Ptr);
	return (0);
}

/*
 *	Routine Parse_Error_Message parses the LaserWriter error message
 *	into semicolon-delimited sections. It returns an array of descriptors
 *	and the descriptor count, one for each error value.
 */

unsigned int Parse_Error_Message (Message_Desc, Desc_Array, Max_Desc)
struct dsc$descriptor *Message_Desc, Desc_Array[];
unsigned int Max_Desc;
{
	auto   unsigned int Index;
	static struct dsc$descriptor Word_Desc;
	extern int Next_Word();
/*
 *	Construct a descriptor for the middle part of the error message;
 *	extract each 'word' of the message as those parts delimited by
 *	a semicolon:
 */
	Set_VMS_Descriptor (Message_Desc->dsc$a_pointer, 0, &Word_Desc);
	for (Index = 0; Index < Max_Desc && Next_Word (&Word_Desc, Message_Desc, ';') != 0; Index++)
	if (Word_Desc.dsc$w_length > 0)
		Desc_Array[Index] = Word_Desc;
	return (Index);
}

/*
 *	Routine Parse_Message_Code parses an error message into
 *	a 'key' and a 'value':
 */

Parse_Message_Code (Code_Desc, Key_Desc, Value_Desc)
struct dsc$descriptor *Code_Desc, *Key_Desc, *Value_Desc;
{
	static struct dsc$descriptor Word_Desc;
	extern int Next_Word();

	Set_VMS_Descriptor (Code_Desc->dsc$a_pointer, 0, &Word_Desc);
	if (Next_Word (&Word_Desc, Code_Desc, ':') != 0) {
		*Key_Desc = Word_Desc;
		if (Next_Word (&Word_Desc, Code_Desc, ':') != 0)
			*Value_Desc = Word_Desc;
		else
			Set_VMS_Descriptor (Code_Desc->dsc$a_pointer, 0, Value_Desc);
	} else
		Set_VMS_Descriptor (Code_Desc->dsc$a_pointer, 0, Key_Desc);
}
