## -*-Tcl-*- (install)
 # ###################################################################
 #  Vince's Additions - an extension package to Alpha
 # 
 #  FILE: "latexSizes.tcl"
 #                                    created: 16/11/97 {7:32:04 pm} 
 #                                last update: 31/1/1999 {11:37:13 pm} 
 #  Author: Vince Darley
 #  E-mail: <darley@fas.harvard.edu>
 #    mail: Division of Engineering and Applied Sciences, Harvard University
 #          Oxford Street, Cambridge MA 02138, USA
 #     www: <http://www.fas.harvard.edu/~darley/>
 #  
 # ========================================================================
 #               Copyright (c) 1997-1998 Vince Darley
 # ========================================================================
 # Permission to use, copy, modify, and distribute this software and its
 # documentation for any purpose and without fee is hereby granted,
 # provided that the above copyright notice appear in all copies and that
 # both that the copyright notice and warranty disclaimer appear in
 # supporting documentation.
 # 
 # Vince Darley disclaims all warranties with regard to this software,
 # including all implied warranties of merchantability and fitness.  In
 # no event shall Vince Darley be liable for any special, indirect or
 # consequential damages or any damages whatsoever resulting from loss of
 # use, data or profits, whether in an action of contract, negligence or
 # other tortuous action, arising out of or in connection with the use or
 # performance of this software.
 # ========================================================================
 # ###################################################################
 ##

# extension declaration
alpha::feature latexSizes 0.1.5 {TeX} {
    alpha::package require -loose Alpha 7.0
    newPref binding makeSmaller "/,<U<I" TeX "" "TeX::sizeChange -1"
    newPref binding makeLarger "/.<U<I" TeX "" "TeX::sizeChange 1"
} "" "" maintainer {
    {Vince Darley} <darley@fas.harvard.edu> <http://www.fas.harvard.edu/~darley/>
} help {
    Let's you shift the size of variable-size items using shift-opt-<>.
    E.g. 
    	you can turn \SMALL into \Small into \small ...
    Or 
    	you can turn the AMS-LaTeX \bigl( into \Bigl( into ... \Biggl(
	
    Usage:
    
    Hit shift-opt-, or shift-opt-. and we look for the first '\' to
    the left of the cursor position.  If it's part of any of the
    usual LaTeX2e text or maths size commands, we adjust it's size.
    Repeated keypresses continue the process.  The cursor position
    doesn't change.
    
    Feel free to improve this: handling more maths delimiters, or
    updating the matching delimiter at the same time would be useful
    improvements.
    
    You can adjust the keybindings in the standard TeX preferences
    dialog.
} uninstall {this-file}

set textSizes [list Tiny tiny SMALL Small small normalsize \
  large Large LARGE huge Huge]
set mathSizes [list XXX "" big Big bigg Bigg XXX]
set mathDelims [list ( ) \[ \] \{ \} | \|]
set mathDelimsRegexp {([][()|]|\\[{}])}

proc TeX::sizeChange {inc} {
    set pos [getPos]
    if {[catch {search -s -f 0 -r 1 -m 0 {\\} [pos::math $pos - 1]} res]} {return}
    global mathDelimsRegexp
    if {![catch {search -s -f 0 -r 1 -l [pos::math [lindex $res 0] - 1] -m 0 $mathDelimsRegexp $pos} resd]} {
	# p is position of the brace
	set p [lindex $resd 0]
	# t may be '\bigg' etc.
	set t [getText [pos::math $p - 6] $p]
	if {[regexp -nocase {\\(big+)([lr]?)$} $t t]} {
	    set res [list [pos::math $p - [string length $t]] $p]
	} else {
	    # it's a normal delimiter (therefore minimum size)
	    set res [list $p $p]
	    set mathsidx 1
	    set lr "l"
	    # rudimentary check for right delimiter
	    if {[lsearch ") \} \]" [lookAt $p]] != -1} {set lr "r"}
	}
    } else {
	set res [search -s -f 1 -r 1 -m 0 {\\[a-zA-Z]+} [lindex $res 0]]
    }
    if {![info exists t]} {
	set t [eval getText $res]
    }
    if {[info exists mathsidx] || \
      ([regexp -nocase {\\(big+)([lr]?)} $t "" which lr])} {
	global mathSizes mathDelims
	# it's a maths delimiter
	#if {[lindex $res 1] +2 < $pos} { return }
	if {![info exists mathsidx]} {
	    set mathsidx [lsearch -exact $mathSizes $which]
	}
	set new [lindex $mathSizes [expr $mathsidx + $inc]]
	if {$new == "XXX"} {
	    message "Can't adjust size further."
	    return
	}
	if {$new != ""} {
	    append new "$lr"
	}
    } else {
	global textSizes
	set idx [lsearch -exact $textSizes [string range $t 1 end]]
	if {$idx == -1} { return }
	if {[pos::compare [pos::math [lindex $res 1] + 2] < $pos]} {
	    # check we're within the {} pair
	    if {![catch {matchIt "\}" [pos::math [lindex $res 1] + 1]} mtch]} {
		if {$mtch < $pos} {return}
	    }
	}
	set new [lindex $textSizes [incr idx $inc]]
	if {$new == ""} {
	    message "Can't adjust size further."
	    return
	}
    }
    pushPosition 0
    if {$new != ""} {
	eval replaceText $res [list "\\$new"]
    } else {
	eval deleteText $res
    }
    popPosition 0
    return    
}



