/* ResidualVM - A 3D game interpreter
 *
 * ResidualVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifndef GRAPHICS_OPENGL_TILED_SURFACE_H
#define GRAPHICS_OPENGL_TILED_SURFACE_H

#include "graphics/opengl/system_headers.h"

#include "common/array.h"
#include "common/rect.h"

#include "graphics/surface.h"

namespace OpenGL {

class Texture;
class SurfaceRenderer;

/**
 * Surface implementation using OpenGL texture tiles
 */
class TiledSurface {
public:
	TiledSurface(uint width, uint height, const Graphics::PixelFormat &pixelFormat);
	~TiledSurface();

	/**
	 * Copy image data to the surface.
	 *
	 * The format of the input data needs to match the format returned by
	 * getFormat.
	 * This does not immediatly updates the textures.
	 *
	 * @param x        X coordinate of upper left corner to copy data to.
	 * @param y        Y coordinate of upper left corner to copy data to.
	 * @param w        Width of the image data to copy.
	 * @param h        Height of the image data to copy.
	 * @param src      Pointer to image data.
	 * @param srcPitch The number of bytes in a row of the image data.
	 */
	void copyRectToSurface(const void *src, int srcPitch, int x, int y, int w, int h);

	/**
	 * Update the OpenGL textures from the backing surface
	 */
	void update();

	/**
	 * Draw as a 2D surface
	 *
	 * The destination rect is as follow :
	 * x: left [0.0, 1.0] right
	 * y: top [0.0, 1.0] bottom
	 *
	 * @param surfaceRenderer
	 */
	void draw(SurfaceRenderer *surfaceRenderer) const;

	/**
	 * Fill the surface with a fixed color.
	 *
	 * @param color Color value in format returned by getFormat.
	 */
	void fill(uint32 color);

	/**
	 * Get the backing surface
	 *
	 * This can be used to update the image data, but causes a full update.
	 * Using copyRectToSurface is preferred
	 */
	Graphics::Surface *getBackingSurface();

	/**
	 * Get the backing surface
	 *
	 * This const version can be used to read image data without causing
	 * a full texture invalidation.
	 */
	const Graphics::Surface *getBackingSurface() const;

private:
	static const uint maxTextureSize = 256;

	struct Tile {
		Texture *texture;
		Common::Rect rect;
		bool dirty;
	};

	Graphics::Surface _backingSurface;
	Common::Array<Tile> _tiles;

	void invalidateAllTiles();
};

} // End of namespace OpenGL

#endif
