;          CPUID.ASM                                             Agner Fog 2003
;
; This program identifies the microprocessor and prints information about
; vendor, processor family, model, instructions supported, cache sizes, etc.
; Assemble and link for 16-bit tiny or small model
;
;  2003 GNU General Public License www.gnu.org/copyleft/gpl.html
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
.686p
.xmm

_TEXT SEGMENT PARA PUBLIC use16 'CODE'

ASSUME  CS:_TEXT

ORG  100H
START:  JMP START1


; ********** DetectProcessor function **********
; C++ prototype:
; extern "C" int DetectProcessor (void);

; return value:
; The return value is 0 if microprocessor has no CPUID instruction
; bits 0-3 = model
; bits 4-7 = family:  4 for 80486, Am486, Am5x86
;                     5 for P1, PMMX, K6
;                     6 for PPro, P2, P3, Athlon, Duron
;                    15 for P4, Athlon64, Opteron
; bit    8 = vendor is Intel
; bit    9 = vendor is AMD
; bit   11 = XMM registers enabled by operating system
; bit   12 = floating point instructions supported
; bit   13 = time stamp counter supported
; bit   14 = CMPXCHG8 instruction supported
; bit   15 = conditional move and FCOMI supported (PPro, P2, P3, P4, Athlon, Duron, Opteron)
; bit   23 = MMX instructions supported (PMMX, P2, P3, P4, K6, Athlon, Duron, Opteron)
; bit   24 = FXSAVE / FXRSTOR supported
; bit   25 = SSE instructions supported (P3, P4, Athlon64, Opteron)
; bit   26 = SSE2 instructions supported (P4, Athlon64, Opteron)
; bit   27 = SSE3 instructions supported (forthcoming "Prescott")
; bit   28 = hyperthreading (P4)
; AMD specific instructions:
; bit   29 = 3DNow instructions
; bit   30 = 3DNow extension instructions
; bit   31 = MMX extension

DetectProcessor PROC NEAR
_DetectProcessor LABEL NEAR
PUBLIC  DetectProcessor, _DetectProcessor
        PUSH    EBX
        PUSH    ESI
        PUSH    EDI
        XOR     EDI, EDI         ; will contain result
; detect if CPUID instruction supported by microprocessor:
        PUSHFD
        POP     EAX
        XOR     EAX, 1 SHL 21    ; check if CPUID bit can toggle
        PUSH    EAX
        POPFD
        PUSHFD
        POP     EBX
        XOR     EAX, EBX
        AND     EAX, 1 SHL 21
        JNZ     DPEND            ; CPUID not supported
        XOR     EAX, EAX
        CPUID                    ; get number of CPUID functions
        TEST    EAX, EAX
        JZ      DPEND            ; function 1 not supported
        ; determine vendor
        XOR     EBX, EBX
        CMP     CH, 't'
        SETE    BL               ; 1 if vendor = Intel
        CMP     CH, 'A'
        SETE    DL   
        SHL     DL, 1            ; 2 if vendor = AMD
        OR      BL, DL
        SHL     EBX, 8           ; shift vendor info to bit 8-9
        OR      EDI, EBX         ; EDI now contains vendor information
        MOV     EAX, 1
        CPUID                    ; get family and features
        SHR     EAX, 4
        AND     EAX, 0FFH        ; family and model
        OR      EDI, EAX         ; combine bits in EDI
        AND     ECX, 1           ; SSE3 feature
        SHL     ECX, 27          ; move to bit 27
        OR      EDI, ECX         ; combine bits in EDI
        MOV     ECX, EDX         ; feature flags
        AND     ECX, 17FF8000H   ; these feature flags remain in same position
        OR      EDI, ECX         ; combine bits in EDI
        MOV     ECX, EDX         ; shuffle feature bits 0, 4, 8 to 12, 13, 14
        AND     ECX, 1           ; FPU feature
        SHL     ECX, 12-0        ; move to bit 12
        OR      EDI, ECX         ; combine bits in EDI
        MOV     ECX, EDX         ; 
        AND     ECX, 10H         ; time stamp counter feature
        SHL     ECX, 13-4        ; move to bit 13
        OR      EDI, ECX         ; combine bits in EDI        
        MOV     ECX, EDX         ; 
        AND     ECX, 100H        ; CMPXCHG8 feature
        SHL     ECX, 14-8        ; move to bit 14
        OR      EDI, ECX         ; combine bits in EDI
        TEST    EDI, 200H        ; is it an AMD?
        JZ      Not_AMD
        ; Get AMD-specific features
        MOV     EAX, 80000000H
        CPUID
        CMP     EAX, 80000001H
        JB      Not_AMD
        MOV     EAX, 80000001H
        CPUID   ; get AMD-specific features into EDX
        MOV     EAX, EDX
        AND     EAX, 0C0000000H  ; 3DNow instructions support bit 30 - 31
        OR      EDI, EAX         ; combine bits in EDI
        AND     EDX, 400000H     ; MMX extensions bit 22        
        SHL     EDX, 29-22       ; move to bit 29
        OR      EDI, EDX         ; combine bits in EDI
Not_AMD:
        ; test OS support for XMM registers
        TEST    EDI, 2000000H    ; SSE support
        JZ      no_SSE_test
        TEST    EDI, 1000000H    ; FXSAVE/FXRSTOR support
        JZ      no_SSE_test
        SMSW    AX               ; read lower part of CR0 without violating privilege
        TEST    AL, 4            ; test if FXSAVE is emulated
        JNZ     no_SSE_test        
        PUSH    EBP
        MOV     ESI, ESP         ; save stack pointer
        SUB     ESP, 200H        ; allocate space for FXSAVE
        AND     ESP, -10H        ; align by 16
        MOV     EBP, ESP
IF @WORDSIZE EQ 2
        @BP EQU <BP>             ; don't use upper word of pointer if 16-bit mode
ELSE
        @BP EQU <EBP>
ENDIF
TESTDATA = 0D95A34BEH            ; random test value
TESTPS   = 10CH                  ; position to write TESTDATA = upper part of XMM6 image
        FXSAVE  [@BP]            ; save FP/MMX and XMM registers
        MOV     EAX,[@BP+TESTPS] ; read part of XMM6 register
        XOR     DWORD PTR [@BP+TESTPS],TESTDATA  ; change value
        FXRSTOR [@BP]            ; load changed value into XMM6
        MOV     [@BP+TESTPS],EAX ; restore old value in buffer
        FXSAVE  [@BP]            ; save again
        MOV     EBX,[@BP+TESTPS] ; read changed XMM6 register
        MOV     [@BP+TESTPS],EAX ; restore old value
        FXRSTOR [@BP]            ; load restored value into XMM6
        XOR     EAX, EBX         ; get difference between old and new value
        CMP     EAX, TESTDATA    ; test if XMM6 was changed correctly
        SETE    AL               ; 1 if test successful
        AND     EAX, 1           ; clear rest of EAX
        SHL     EAX, 11          ; move to bit 11
        OR      EDI, EAX         ; combine bits in EDI        
        MOV     ESP, ESI         ; restore ESP
        POP     EBP
no_SSE_test:
DPEND:  
        MOV     EAX, EDI         ; return result bits
IF @WORDSIZE EQ 2
        MOV     EDX, EAX         ; get high word into DX if 16-bit mode
        SHR     EDX, 16
ENDIF
        POP     EDI
        POP     ESI
        POP     EBX
        RET
DetectProcessor ENDP


; ********** InstructionSet function **********
; C++ prototype:
; extern "C" int InstructionSet (void);

; return value:
;  0 = use 80386 instruction set
;  1 or above = MMX instructions can be used
;  2 or above = conditional move and FCOMI can be used
;  3 or above = SSE (XMM) supported by processor and enabled by O.S.
;  4 or above = SSE2 supported by processor and enabled by O.S.
;  5 or above = SSE3 supported by processor and enabled by O.S.

InstructionSet PROC NEAR
_InstructionSet LABEL NEAR
PUBLIC InstructionSet, _InstructionSet
        CALL    _DetectProcessor
        MOV     EDX, EAX       ; feature flags
        XOR     EAX, EAX       ; 0
        TEST    EDX, 1 SHL 12  ; floating point support
        JZ      I_S_end
        TEST    EDX, 1 SHL 23  ; MMX support
        JZ      I_S_end
        INC     EAX            ; 1
        TEST    EDX, 1 SHL 15  ; conditional move support
        JZ      I_S_end
        INC     EAX            ; 2
        TEST    EDX, 1 SHL 25  ; SSE support by microprocessor
        JZ      I_S_end
        TEST    EDX, 1 SHL 11  ; SSE enabled by operating system
        JZ      I_S_end
        INC     EAX            ; 3
        TEST    EDX, 1 SHL 26  ; SSE2 support by microprocessor
        JZ      I_S_end
        INC     EAX            ; 4
        TEST    EDX, 1 SHL 27  ; SSE3 support by microprocessor
        JZ      I_S_end
        INC     EAX            ; 5        
I_S_end:RET
InstructionSet ENDP


; ********** ProcessorName function **********
; C++ prototype:
; extern "C" void ProcessorName (char * text);

; This function finds the name of the microprocessor. The name is returned
; in the parameter text, which must be a character array of at least 68 bytes.

ProcessorName PROC NEAR
_ProcessorName LABEL NEAR
PUBLIC ProcessorName, _ProcessorName
        CALL    DetectProcessor
        PUSH    EBX
        PUSH    ESI
        PUSH    EDI
        PUSH    EBP
IF @WORDSIZE EQ 4        
        MOV     EDI, [ESP+20]      ; text pointer
ELSE
        MOV     BP, SP             ; 16-bit mode
        MOVZX   EDI, WORD PTR [BP+18] 
        PUSH    DS
        POP     ES
ENDIF
        TEST    EAX, EAX
        JNZ     IDENTIFYABLE
        ; processor has no CPUID
        MOV     DWORD PTR [EDI], '8408'   ; write text '80486 or lower'
        MOV     DWORD PTR [EDI+4], 'ro 6'
        MOV     DWORD PTR [EDI+8], 'wol '
        MOV     DWORD PTR [EDI+12], 're'  ; end with 0
        JMP     PNEND
IDENTIFYABLE:
        MOV     ESI, EAX           ; value returned by DetectProcessor
        MOV     EAX, 80000000H
        CPUID
        CMP     EAX, 80000004H     ; text if extended vendor string available
        JB      no_ext_vendor_string
        MOV     EBP, 80000002H
        PUSH    EDI
VLOOP:  MOV     EAX, EBP           ; repeat 3 times to get extended vendor string
        CPUID
        MOV     [EDI], EAX         ; store extended vendor string
        MOV     [EDI+4], EBX
        MOV     [EDI+8], ECX
        MOV     [EDI+12], EDX
        ADD     EDI, 16
        INC     EBP
        CMP     EBP, 80000004h
        JBE     VLOOP
        POP     EDI
        JMP     get_family_and_model
no_ext_vendor_string:
        ; get short vendor string
        XOR     EAX, EAX
        CPUID
        MOV     [EDI],EBX          ; store short vendor string
        MOV     [EDI+4],EDX
        MOV     [EDI+8],ECX
        MOV     BYTE PTR [EDI+12],0 ; terminate string
get_family_and_model:
IF 0
        XOR     EAX, EAX
        MOV     ECX, 48
        CLD
        REPNE   SCASB              ; find end of text
        DEC     EDI
        MOV     DWORD PTR [EDI], 'maF '    ; append text ' Family ? Model ?'
        MOV     DWORD PTR [EDI+4], ' yli'
        MOV     DWORD PTR [EDI+8], 'oM ?'
        MOV     DWORD PTR [EDI+12], ' led'
        MOV     EAX, ESI
        SHR     EAX, 4
        AND     EAX, 07H            ; family number
        ADD     AL, '0'             ; convert to ASCII
        MOV     BYTE PTR [EDI+8], AL  ; put into string
        MOV     EAX, ESI
        AND     EAX, 0FH            ; model number
        ADD     AL, '0'             ; convert to ASCII
        CMP     AL, '9'
        JNA     MOD09
        ADD     AL, 'A'-'9'+1       ; hexadecimal
MOD09:  MOV     DWORD PTR [EDI+16], EAX ; put into string, followed by 0
ENDIF
PNEND:  POP     EBP
        POP     EDI
        POP     ESI
        POP     EBX
        RET
ProcessorName ENDP

        
; *********** structure definitions ***********

feature struc  ; structure for defining feature
bits    dd  ?           ; feature bit
text    db  28 dup (?)  ; feature name
feature ends

descriptor struc  ; structure for cache descriptor
fvalue   db  ?          ; feature ID
ftext    db  63 dup (?) ; feature name
descriptor ends

        
; *********** macro definitions ***********

WRITETEXT MACRO T     ; outputs text string T
        MOV     DX, OFFSET DS:T
        MOV     AH,9
        INT     21H
ENDM


; program start
START1:  
; *********** initialize ***********
        MOV     AX,CS
        MOV     DS,AX
        MOV     ES,AX
ASSUME  CS:_TEXT, DS:_TEXT, ES:_TEXT 
        
        CLD
        WRITETEXT heading  ; write heading
        
; *********** test for 16-bit processors ***********
        push    sp
        pop     ax
        cmp     ax,sp
        je      not_8086
        writetext pr8086
        jmp     exit
not_8086:
        pushf
        pop     ax
        mov     bx,ax
        or      ax,0f000h
        push    ax
        popf
        pushf
        pop     ax
        test    ax,0f000h
        jnz     not_80286
        writetext p80286
        jmp     exit
not_80286:push  bx
        popf        
        
; *********** test if CPUID supported ***********
        call    DetectProcessor
        mov     detectflags, eax
        test    eax, eax
        jnz     CPUID_OK
        WRITETEXT  nocpuid  ; CPU does not have CPUID instruction
        jmp     exit
CPUID_OK:
        sub     eax, eax
        cpuid
        mov     maxfunc, eax
        
; *********** get processor name string ***********
        lea     di, vendor_string
        push    di
        call    ProcessorName
        pop     di
        cld
        mov     cx, 69
        sub     ax, ax
        repne   scasb                ; find end of string
        mov     byte ptr [di-1],'$'  ; end string with '$'
        WRITETEXT vendor_string

; *********** get family number etc. ***********       
        writetext line
        writetext line
        mov     eax,1
        CPUID
        mov     esi,eax ; family etc.
        mov     featureflags, edx
        writetext family
        mov     eax,esi
        shr     eax,8
        and     eax,0fh
        call    writenum         
        writetext pmodel
        mov     eax,esi
        shr     eax,4
        and     eax,0fh
        call    writenum
        writetext xfamily
        mov     eax,esi
        shr     eax,20
        and     eax,0ffh
        call    writenum        
        writetext xmodel
        mov     eax,esi
        shr     eax,16
        and     eax,0fh
        call    writenum

; *********** get features from CPUID(1)->EDX ***********
        writetext line
        writetext features        
        mov     si, offset featurelist
        mov     edi, featureflags
floop:  ; search through featurelist
        test    edi,[si]+feature.bits
        jz      not_supp
        lea     dx,[si]+feature.text  ; this feature supported, get text
        MOV     AH,9
        INT     21H        ; write feature name
        writetext space
not_supp:        
        add     si, type feature
        cmp     si, offset ds:featurelistend
        jb      floop
        
; *********** get features from CPUID(1)->ECX ***********
        test    detectflags, 1 shl 27
        jz      no_SSE3
        writetext feature_SSE3
no_SSE3:

; *********** get AMD-specific features ***********
        test    dword ptr detectflags, 1 shl 9
        jz      no_AMD_features
        cmp maxxfunc,80000001h
        jb      no_AMD_features
        mov     eax,80000001h
        CPUID
        mov     edi,edx ; AMD features        
        mov     si, offset AMDfeaturelist
floop2:  ; search through AMD featurelist        
        test    edi,[si]+feature.bits
        jz      not_supp2
        lea     dx,[si]+feature.text  ; this feature supported, get text
        MOV     AH,9
        INT     21H        ; write feature name
        writetext space
not_supp2:        
        add     si,type feature
        cmp     si, offset AMDfeaturelistend
        jb      floop2
no_AMD_features:

; ***********  operating support for XMM *********
        test    dword ptr detectflags, 1 shl 11
        jz      no_OS_XMM_support
        writetext OS_XMM_support
no_OS_XMM_support:
; ***********  get cache descriptors (Intel) *********
        test    dword ptr detectflags, 1 shl 8
        jz      nocachedesc
        cmp     dword ptr maxfunc, 2
        jb      nocachedesc  ; no cache descriptors
        writetext line
        mov     eax,2
        cpuid
        mov     vendor_string,eax     ; store 16 descriptor bytes
        mov     vendor_string+4,ebx
        mov     vendor_string+8,ecx
        mov     vendor_string+12,edx        
        writetext cachedesc                
        mov     si, offset vendor_string

dloop:  ; loop through descriptors
        lodsb
        mov     di, offset descriptorlist
dlloop: ; loop through descriptorlist
        cmp     al,[di]+descriptor.fvalue
        je      descfound
        add     di,type descriptor
        cmp     di, offset ds:descriptorlistend
        jb      dlloop
        ; not found in list
        test    al,al
        jz      nextdesc ; null descriptor
        push    ax ; unknown descriptor
        writetext unknowndesc
        pop     ax
        call    writebyte
        writetext space
        jmp     nextdesc
descfound: ; descriptor found, write its name
        lea     dx,[di]+descriptor.ftext  ; this feature supported, get text
        MOV     AH,9
        INT     21H        ; write feature name
        writetext space
nextdesc: 
        cmp     si, offset vendor_string + 16
        jb      dloop
nocachedesc:

; ***********  get cache descriptors (AMD) *********
        test    dword ptr detectflags, 1 shl 9
        jz      no_AMD_cache
        cmp     maxxfunc, 80000005h
        jb      no_AMD_cache
        writetext line
        writetext cachedesc
        mov     eax, 80000005h
        CPUID
        mov     edi, eax
        mov     esi, edx
        writetext instrTLBlarge
        mov     ax, di
        call    INTERPRET_TLB
        writetext dataTLBlarge
        shr     edi, 16
        mov     ax, di
        call    INTERPRET_TLB 
        writetext instrTLBsmall
        mov     ax, bx
        call    INTERPRET_TLB
        writetext dataTLBsmall
        shr     ebx, 16
        mov     ax, bx
        call    INTERPRET_TLB
        writetext L1InstructCache
        mov     eax,esi
        mov     bx,0
        call    INTERPRET_CACHE
        writetext L1DataCache
        mov     eax,ecx
        mov     bx,0
        call    INTERPRET_CACHE
        cmp     maxxfunc, 80000006h
        jb      no_AMD_cache
        mov     eax, 80000006h
        CPUID
        writetext L2cache        
        mov     eax, ecx
        mov     bx, 1
        call    INTERPRET_CACHE
no_AMD_cache:

; *********** end program ***********
exit:   mov     ah, 8
        int     21h                   ; wait for key pressure
        MOV     AX, 4C00H
        INT     21H                   ; exit


; *********** procedures ***********
WRITEDW PROC NEAR                     ; writes EAX as hexadecimal
        mov     bx,8
wdl:    rol     eax, 4
        push    bx
        push    eax
        call    writehex
        pop     eax
        pop     bx
        dec     bx
        jnz     wdl
        ret
writedw endp

WRITEHEX PROC NEAR     ; writes AL low nibble as Hex digit
        and     al,0fh
        add     al,'0'
        cmp     al,'9'
        jna     wrh2
        add     al,'A'-'9'-1
        wrh2:
        mov     dl,al
        mov     ah,2
        int     21h
        ret
writehex endp

writebyte proc near   ; writes AL as hexadecimal
        push    ax
        shr     al,4
        call    writehex
        pop     ax
        call    writehex
        ret
writebyte endp

WRITENUM PROC NEAR  ; writes number in EAX as decimal signed integer
        push    si
        push    di
        push    bp
        mov     bp,sp
        sub     sp,16
        xor     di,di
        test    eax,eax
        jns     short WN1
        neg     eax                    ; negative
        push    eax
        mov     dl,'-'
        mov     ah,2
        int     21h                    ; write '-'
        inc     di
        pop     eax
WN1:    mov     si, sp                 ; 10 byte temp. buffer
        xor     ecx,ecx
        mov     ss:[si],eax
        mov     ss:[si+4],ecx
        fild    qword ptr ss:[si]
        fbstp   tbyte ptr ss:[si]      ; convert to packed BCD
        add     si,6
        std
        wait
WN2:    lods    byte ptr ss:[si]       ; read two BCD digits
        mov     ah,al                  ; unpack two BCD digits
        and     al,0fh
        shr     ah,4
        mov     dl,ah
        or      ah,cl
        jz      short WN3              ; skip leading zeros
        inc     cx
        add     dl,'0'
        push    ax
        mov     ah,2
        int     21h                    ; print digit
        inc     di
        pop     ax
WN3:    mov     dl,al
        or      al,cl
        jz      short WN4              ; skip leading zeros
        inc     cx
        add     dl,'0'
        mov     ah,2
        int     21h                    ; print digit
        inc     di
WN4:    cmp     si, sp
        jnb     WN2                    ; next two digits
        test    cl,cl
        jnz     short WN5
        mov     dl,'0'                 ; no digits have been written
        mov     ah,2
        int     21h                    ; write a '0'
        inc     di
WN5:
IF 0    ; use this if you want to left-justify number
        cmp     di,11
        jnb     WN9
        mov     dl,' '                 ; fill with spaces
        mov     ah,2
        int     21h                    ; write a ' '
        inc     di
        jmp     WN5
ENDIF
WN9:    cld
        mov     sp,bp
        pop     bp
        pop     di
        pop     si
        ret
WRITENUM        ENDP

INTERPRET_TLB PROC NEAR
; interpret AMD TLB information in AX
        pushad
        mov     di, ax
        mov     ax, di
        and     eax, 0ffh
        call    writenum                
        writetext entries
        mov     ax, di
        shr     ax, 8
        and     eax, 0ffh
        call    writenum                
        writetext ways
        popad
        ret
INTERPRET_TLB endp

INTERPRET_CACHE PROC NEAR
; interpret AMD cache information in EAX
; BX = 1 if big format
        pushad
        mov     edi,eax
        test    bx,bx
        jnz     ic1
        shr     eax,24
        jmp     ic2
ic1:    shr     eax,16
ic2:    call    writenum
        writetext sizekb
        mov     eax,edi
        test    bx,bx
        jnz     ic3
        shr     eax,16
        and     eax,0ffh
        jmp     ic4
ic3:    shr     eax,12
        and     eax,0fh
ic4:    call    writenum
        writetext ways        
        mov     eax,edi
        shr     eax,8
        test    bx,bx
        jnz     ic5
        and     eax,0ffh
        jmp     ic6
ic5:    and     eax,0fh
ic6:    call    writenum
        writetext linespertag         
        mov     eax,edi
        and     eax,0ffh
        call    writenum
        writetext bytesperline
        popad
        ret
INTERPRET_CACHE ENDP        


; *********** data ***********
align 4
featureflags dd 0 ; feature flags
detectflags  dd 0 ; feature flags from DetectProcessor function
maxfunc dd 0   ; maximum cpuid function
maxxfunc dd 0  ; maximum cpuid extended function

featurelist label dword  ; list of possible features
feature <1, 'Floating point$'>
feature <800000h, '64-bit MMX instructions$'>
feature <2000000h, '128-bit SSE$'>
feature <4000000h, 'SSE2$'>
feature <8000h, 'Conditional move$'>
feature <10h, 'Time stamp counter$'>
feature <20h, 'Model specific registers$'>
feature <1000000h, 'FXSAVE$'>
feature <100h, 'CMPXCHG8$'>
feature <2, 'Virtual mode$'>
feature <4, 'Debugging$'>
feature <8, 'Page size ext.$'>
feature <40h, 'Phys. addr ext.$'>
feature <80h, 'Machine check except.$'>
feature <200h, 'APIC$'>
feature <800h, 'SYSENTER$'>
feature <1000h, 'Memory type range$'>
feature <2000h, 'Page global$'>
feature <4000h, 'Machine check architecture$'>
feature <10000h, 'Page attr. table$'>
feature <20000h, 'Page size ext.$'>
feature <40000h, 'Processor serial number$'>
feature <80000h, 'CLFLUSH$'>
feature <200000h, 'Debug store$'>
feature <400000h, 'Termal monitor$'>
; feature <8000000h, 'Self snoop$'>
feature <10000000h, 'Hyper-threading$'>
;feature <20000000h, 'Thermal monitor 2$'>
feature <80000000h, 'Signal break FERR$'>
featurelistend label dword

AMDfeaturelist label dword  ; list of AMD-specific features
feature <80000000h, '3DNow instructions$'>
feature <40000000h, '3DNow extension$'>
feature <400000h, 'AMD MMX extension$'>
feature <20000000h, 'AMD Long mode supported$'>
feature <0A0000000h, 'Prefetch instruction$'>
feature <100000h, 'No-execute page protection$'>
AMDfeaturelistend label dword

descriptorlist label dword ; list of possible cache descriptors
descriptor <01h, 'Instruction TLB: 4kb pages, 4-way, 32 entries$'>
descriptor <02h, 'Instruction TLB: 4Mb pages, 2 entries$'>
descriptor <03h, 'Data TLB: 4kb pages, 4-way, 64 entries$'>
descriptor <04h, 'Data TLB: 4Mb pages, 4-way, 8 entries$'>
descriptor <06h, 'L1 Instruction cache: 8kb, 4-way, 32 bytes line size$'>
descriptor <08h, 'L1 Instruction cache: 16kb, 4-way, 32 bytes line size$'>
descriptor <0ah, 'L1 Data cache: 8kb, 2-way, 32 bytes line size$'>
descriptor <0ch, 'L1 Data cache: 16kb, 4-way, 32 bytes line size$'>
descriptor <22h, 'L3 cache: 512kb, 4-way, 64 bytes sector size$'>
descriptor <23h, 'L3 cache: 1Mb, 8-way, 64 bytes sector size$'>
descriptor <25h, 'L3 cache: 2Mb, 8-way, 64 bytes sector size$'>
descriptor <29h, 'L3 cache: 4Mb, 8-way, 64 bytes sector size$'>
descriptor <39h, 'L2 cache: 128kb, 4-way, 64 bytes sector size$'>
descriptor <3bh, 'L2 cache: 128kb, 2-way, 64 bytes sector size$'>
descriptor <3ch, 'L2 cache: 256kb, 4-way, 64 bytes sector size$'>
descriptor <40h, 'No L3 cache$'>
descriptor <41h, 'L2 cache: 128kb, 4-way, 32 bytes line size$'>
descriptor <42h, 'L2 cache: 256kb, 4-way, 32 bytes line size$'>
descriptor <43h, 'L2 cache: 512kb, 4-way, 32 bytes line size$'>
descriptor <44h, 'L2 cache: 1Mb, 4-way, 32 bytes line size$'>
descriptor <45h, 'L2 cache: 2Mb, 4-way, 32 bytes line size$'>
descriptor <50h, 'Instruction TLB 4k, 2M or 4M pages, 64 entries$'>
descriptor <51h, 'Instruction TLB 4k, 2M or 4M pages, 128 entries$'>
descriptor <52h, 'Instruction TLB 4k, 2M or 4M pages, 256 entries$'>
descriptor <5bh, 'Data TLB 4k, 4M pages, 64 entries$'>
descriptor <5ch, 'Data TLB 4k, 4M pages, 128 entries$'>
descriptor <5dh, 'Data TLB 4k, 4M pages, 256 entries$'>
descriptor <66h, 'L1 Data cache: 8kb, 4-way, 64 bytes sector size$'>
descriptor <67h, 'L1 Data cache: 16kb, 4-way, 64 bytes sector size$'>
descriptor <68h, 'L1 Data cache: 32kb, 4-way, 64 bytes sector size$'>
descriptor <70h, 'Trace cache: 12k uops, 8-way$'>
descriptor <71h, 'Trace cache: 16k uops, 8-way$'>
descriptor <72h, 'Trace cache: 32k uops, 8-way$'>
descriptor <79h, 'L2 cache: 128kb, 8-way, 64 bytes sector size$'>
descriptor <7ah, 'L2 cache: 256kb, 8-way, 64 bytes sector size$'>
descriptor <7bh, 'L2 cache: 512kb, 8-way, 64 bytes sector size$'>
descriptor <7ch, 'L2 cache: 1Mb, 8-way, 64 bytes sector size$'>
descriptor <82h, 'L2 cache: 256kb, 8-way, 32 bytes line size$'>
descriptor <83h, 'L2 cache: 512kb, 8-way, 32 bytes line size$'>
descriptor <84h, 'L2 cache: 1Mb, 8-way, 32 bytes line size$'>
descriptor <85h, 'L2 cache: 2Mb, 8-way, 32 bytes line size$'>
descriptor <82h, 'L2 cache: 256kb, 8-way, 32 bytes line size$'>
descriptorlistend label dword

; *********** text strings ***********
vendor_string  LABEL DWORD  ; temporary storage of vendor string and other features
        DB 68 DUP (0)
        DB 13,10,36
line    db 13,10,36
heading db 13,10,'                       CPU identification:',13,10,10,36
pr8086  db '8086 or 8088$'
p80286  db '80186 or 80286$'
nocpuid db 'CPUID not supported by microprocessor',13,10,36
family  db 'Family: $';
pmodel   db '. Model: $';
xfamily db '. Extended family: $';
xmodel  db '. Extended model: $';
feature_SSE3 db 'SSE3; $'
OS_XMM_support db 13,10,10,'Operating system supports XMM.',36
features db 13,10,10,'Features:',13,10,36
cachedesc db 13,10,'Cache descriptions:',13,10,36
space   db '; $'
unknowndesc db 'Unknown descriptor: $'
entries db ' entries $'
ways    db ' ways $'
sizekb  db ' kb $'
linespertag db ' lines per tag $'
bytesperline db ' bytes per line $'
instrTLBlarge db 'instruction TLB large pages $'
instrTLBsmall db '; instruction TLB 4k pages $'
dataTLBlarge db '; data TLB large pages $'
dataTLBsmall db '; data TLB 4k pages $'
L1DataCache db '; L1 data cache $'
L1InstructCache db '; L1 instruction cache $'
L2cache db '; L2 cache $'

_TEXT   ENDS

END     START
