/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  Substantially modified from:
  hostapd-0.5.7
  Copyright (c) 2002-2007, Jouni Malinen <jkmaline@cc.hut.fi> and
  contributors

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "includes.h"
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/un.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "common.h"
#include "eloop.h"
#include "dcbd.h"
#include "dcb_protocol.h"
#include "dcbd_shm.h"
#include "ctrl_iface.h"
#include "lldp/agent.h"
#include "lldp/ports.h"
#include "event_iface.h"
#include "messages.h"
#include "config.h"
#include "version.h"

extern u8 gdcbx_subtype;

char *cfg_file_name = NULL;

static const char *dcbd_version =
"dcbd v" VERSION_STR "\n"
"Copyright (c) 2007-2009, Intel Corporation\n"
"\nPortions used and/or modified from:  hostapd v 0.5.7\n"
"Copyright (c) 2004-2007, Jouni Malinen <j@w1.fi> and contributors";

static void handle_term(int sig, void *eloop_ctx, void *signal_ctx)
{
	printf("Signal %d received - terminating\n", sig);
	eloop_terminate();
}


static void usage(void)
{
	fprintf(stderr,
		"\n"
		"usage: dcbd [-hdksv] [-f configfile]"
		"\n"
		"options:\n"
		"   -h  show this usage\n"
		"   -f  use configfile instead of default\n"
		"   -d  run daemon in the background\n"
		"   -k  terminate current running dcbd\n"
		"   -s  remove dcbd state records\n"
		"   -v  show version\n");

	exit(1);
}

void send_event(int level, char *ebuf)
{
	struct dcbd_user_data *dud = NULL;
	dud = (struct dcbd_user_data *) eloop_get_user_data();

	if (dud != NULL && dud->dcbdd != NULL) {
		ctrl_iface_send(dud->dcbdd, level, ebuf, strlen(ebuf));
	}
}


int main(int argc, char *argv[])
{
	int c, daemonize = 0;
	struct dcbd_user_data *dud;
	int fd;
	char buf[32];
	int i;
	int dcbx_version;
	int shm_remove = 0;
	int killme = 0;
	int rval;

	for (;;) {
		c = getopt(argc, argv, "dhkvsf:");
		if (c < 0)
			break;
		switch (c) {
		case 'f':
			if (cfg_file_name) {
				usage();
				break;
			}
			cfg_file_name = strdup(optarg);
			break;
		case 'd':
			daemonize++;
			break;
		case 'k':
			killme = 1;
			break;
		case 's':
			shm_remove = 1;
			break;
		case 'v':
			printf("%s\n", dcbd_version);
			exit(0);
		case 'h':
		default:
			usage();
			break;
		}
	}
	if (cfg_file_name == NULL) {
		cfg_file_name = DEFAULT_CFG_FILE;
	}

	if (shm_remove) {
		mark_dcbx_shm_for_removal();
		exit(0);
	}

	if (killme) {
		rval = 0;
		fd = open(PID_FILE, O_RDWR , S_IRUSR | S_IWUSR);
		if (fd < 0) {
			printf("%s was not running\n", argv[0]);
		} else if (read(fd, buf, sizeof(buf)) > 0) {
			printf("killing %s", buf);
			if (kill(atoi(buf), SIGINT)) {
				perror("dcbd kill failed");
				rval = 1;
			}
		}
		exit(rval);
	}

	fd = open(PID_FILE, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
	if (fd < 0) {
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "error opening dcbd lock file");
		exit(1);
	}

	if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
		if (errno == EWOULDBLOCK) {
			fprintf(stderr, "dcbd is already running\n");
			if (read(fd, buf, sizeof(buf)) > 0) {
				fprintf(stderr, "pid of existing dcbd is %s\n",
					buf);
			}
			log_message(MSG_ERR_SERVICE_START_FAILURE,
				"%s", "dcbd already running");
		} else {
			perror("error locking dcbd lock file");
			log_message(MSG_ERR_SERVICE_START_FAILURE,
				"%s", "error locking dcbd lock file");
		}
		exit(1);
	}

	if (daemonize && os_daemonize(PID_FILE)) {
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "error daemonizing dcbd");
		goto out;
	}

	if (lseek(fd, 0, SEEK_SET) < 0) {
		if (!daemonize)
			fprintf(stderr, "error seeking dcbd lock file\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "error seeking dcbd lock file");
		exit(1);
	}

	memset(buf, 0, sizeof(buf));
	sprintf(buf, "%u\n", getpid());
	if (write(fd, buf, sizeof(buf)) < 0)
		perror("error writing to dcbd lock file");
	if (fsync(fd) < 0)
		perror("error syncing dcbd lock file");

	openlog("dcbd", LOG_CONS | LOG_PID, LOG_DAEMON);

	if (check_cfg_file()) {
		exit(1);
	}

	/* Get the DCBX version */
	if (get_dcbx_version(&dcbx_version)) {
		gdcbx_subtype = dcbx_version;
	} else {
		if (!daemonize)
			fprintf(stderr, "failed to get DCBX version\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to get DCBX version");
		exit(1);
	}

	/* initialize dcbd user data */
	dud = malloc(sizeof(struct dcbd_user_data));
	if (dud == NULL) {
		if (!daemonize)
			fprintf(stderr, "failed to malloc user data\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to malloc user data");
		exit(1);
	}
	dud->dcbdd = malloc(sizeof(struct dcbd_data));
	if (dud->dcbdd == NULL) {
		if (!daemonize)
			fprintf(stderr, "failed to malloc dcbd data\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to malloc dcbd data");
		exit(1);
	}
	dud->dcbdd->ctrl_interface = (char *) CLIF_IFACE_DIR;
	strcpy(dud->dcbdd->iface, CLIF_IFACE_IFNAME);
	dud->dcbdd->ctrl_interface_gid_set = 0;
	dud->dcbdd->ctrl_interface_gid = 0;

	if (eloop_init(dud)) {
		if (!daemonize)
			fprintf(stderr, "failed to initialize event loop\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to initialize event loop");
		exit(1);
	}

	eloop_register_signal_terminate(handle_term, NULL);

	/* setup LLDP agent */
	if (!start_lldp_agent()) {
		if (!daemonize)
			fprintf(stderr, "failed to initialize LLDP agent\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to initialize LLDP agent");
		exit(1);
	}

	/* store pg defaults */
	if (!add_pg_defaults()) {
		if (!daemonize)
			fprintf(stderr, "failed to add default PG data\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to add default PG data");
		exit(1); 
	}

	/* store pg defaults */
	if (!add_pfc_defaults()) {
		if (!daemonize)
			fprintf(stderr, "failed to add default PFC data\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to add default PFC data");
		exit(1);
	} 

	/* store app defaults */
	for (i = 0; i < DCB_MAX_APPTLV; i++) {
		if (!add_app_defaults(i)) {
			if (!daemonize)
				fprintf(stderr,
					"failed to add default APP data\n");
			log_message(MSG_ERR_SERVICE_START_FAILURE,
				"%s:%d", "failed to add default APP data", i);
			exit(1);
		} 
	} 


	for (i = 0; i < DCB_MAX_APPTLV; i++) {
		if (!add_llink_defaults(i)) {
			if (!daemonize)
				fprintf(stderr,
					"failed to add default APP data\n");
			log_message(MSG_ERR_SERVICE_START_FAILURE,
				"%s:%d", "failed to add default APP data", i);
			exit(1);
		} 
	} 

	if (ctrl_iface_init(dud->dcbdd) < 0) {
		if (!daemonize)
			fprintf(stderr,
				"failed to register client interface\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to register client interface");
		exit(1);
	}

	/* Find available interfaces, read in the dcbd.conf file and
	 * add adapters */
	init_ports();

	/* setup event RT netlink interface */
	if (dcbd_event_iface_init() < 0) {
		if (!daemonize)
			fprintf(stderr, "failed to register event interface\n");
		log_message(MSG_ERR_SERVICE_START_FAILURE,
			"%s", "failed to register event interface");
		exit(1);
	}

	eloop_run();

	remove_all_adapters();
	ctrl_iface_deinit(dud->dcbdd);  /* free's dud->dcbdd */
	dcbd_event_iface_deinit();
	stop_lldp_agent();
	free(dud);

 out:
	
	closelog();
	unlink(PID_FILE);
	eloop_destroy();
	exit(1);
}
