#!/bin/sh
#
# SPDX-FileCopyrightText: 2022-2025 Helmut Grohne <helmut@subdivi.de>
# SPDX-License-Identifier: MIT
#
# This is a mmdebstrap customize hook that installs a kernel image. The name
# of the kernel image depends on the architecture, derivative and release.

set -eu

if [ "${MMDEBSTRAP_VERBOSITY:-1}" -ge 3 ]; then
        set -x
fi

TARGET="$1"

ARCHITECTURES=$(xargs < "$TARGET/var/lib/dpkg/arch")

affected_by_1079443() {
	case "${ARCHITECTURES%% *}" in
		armel|armhf)
		;;
		*)
			return 1
		;;
	esac
	test "${1:-}" = --maybe && return 0
	test -d "$TARGET/etc/initramfs-tools/hooks" || return 1
	dpkg-query --root="$TARGET" --showformat='${db:Status-Status}\n' --show dracut-install 2>/dev/null | grep -q '^installed$'
}
work_around_1079443() {
	# FTS or its usage in dracut-install is broken on some 32bit architectures.
	echo "Warning: working around #1079443"
	cat >"$TARGET/etc/initramfs-tools/hooks/work_around_1079443" <<'EOF'
#!/bin/sh

test "${1-}" = prereqs && exit 0

. /usr/share/initramfs-tools/hook-functions

# work around #1079443
manual_add_modules virtio_blk
EOF
	chmod +x "$TARGET/etc/initramfs-tools/hooks/work_around_1079443"
	if test "${1:-}" = --update && test -x "$TARGET/usr/bin/update-initramfs"; then
		chroot "$TARGET" update-initramfs -u
	fi
}

if dpkg-query --root="$TARGET" --showformat='${db:Status-Status}\n' --show 'linux-image-*' 2>/dev/null | grep -q '^installed$'; then
	if affected_by_1079443; then
		work_around_1079443 --update
	fi
	exit 0
fi

KERNEL_ARCH="${ARCHITECTURES%% *}"
case "$KERNEL_ARCH" in
	armel)
		KERNEL_ARCH=rpi
		case "$ARCHITECTURES " in
			*" arm64 "*) KERNEL_ARCH=arm64:arm64 ;;
			*" armhf "*) KERNEL_ARCH=armmp:armhf ;;
		esac
	;;
	armhf)
		KERNEL_ARCH=armmp
		case "$ARCHITECTURES " in
			*" arm64 "*) KERNEL_ARCH=arm64:arm64 ;;
		esac
	;;
	hppa)
		KERNEL_ARCH=parisc
	;;
	i386)
		KERNEL_ARCH=686-pae
		case "$ARCHITECTURES " in
			*" amd64 "*) KERNEL_ARCH=amd64:amd64 ;;
		esac
	;;
	mips64el)
		KERNEL_ARCH=5kc-malta
	;;
	mipsel)
		KERNEL_ARCH=4kc-malta
		case "$ARCHITECTURES " in
			*" mips64el "*) KERNEL_ARCH=5kc-malta:mips64el
		esac
	;;
	ppc64)
		KERNEL_ARCH=powerpc64
	;;
	ppc64el)
		KERNEL_ARCH=powerpc64le
	;;
esac

export APT_CONFIG="$MMDEBSTRAP_APT_CONFIG"

if test "${MMDEBSTRAP_MODE:-}" = chrootless; then
	set -- \
		-oDPkg::Options::=--force-not-root \
		-oDPkg::Options::=--force-script-chrootless \
		-oDPkg::Options::=--root="$TARGET" \
		-oDPkg::Options::=--log="$TARGET/var/log/dpkg.log"
else
	set -- -oDPkg::Chroot-Directory="$TARGET"
fi

# On some derivatives such as Ubuntu, linux image does not depend on an initramfs.
KERNEL_SATISFY="linux-image-cloud-$KERNEL_ARCH | linux-image-$KERNEL_ARCH | linux-image-generic"
INITRD_SATISFY="initramfs-tools | linux-initramfs-tool"

if affected_by_1079443 --maybe; then
	apt-get --yes satisfy "$@" "$INITRD_SATISFY"
	if affected_by_1079443; then
		work_around_1079443
	fi
	apt-get --yes satisfy "$@" "$KERNEL_SATISFY"
else
	apt-get --yes satisfy "$@" "$KERNEL_SATISFY" "$INITRD_SATISFY"
fi
