/* $Id: configuration.c,v 1.9 2003/06/13 17:17:47 sjoerd Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <SDL.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>

#include "global.h"
#include "list.h"
#include "vector.h"
#include "configuration.h"

int
cmp_config_player(void *data, void *user_data) {
  Config_player *cplayer = (Config_player *)data;
  uint32_t *player = (uint32_t *)user_data;

  return cplayer->player == *player;
}

Config_key *
new_config_key(Config_action action, SDLKey key) {
  Config_key *result;
  result = malloc(sizeof(Config_key));
  result->action = action;
  result->key = key;
  return result;
}

void
del_config_key(Config_key *ckey) {
  free(ckey);
}

Config_player *
new_config_player(uint32_t player,char *name) {
  Config_player *result;
  result = malloc(sizeof(Config_player));
  result->name = strdup(name);
  result->player = player;
  result->keys = new_list();
  return result;
}

void
del_config_player(Config_player *cplayer) {
  Config_key *ckey;  
  while ((ckey = list_pop(cplayer->keys)) != NULL) {
    del_config_key(ckey);
  }
  del_list(cplayer->keys);
  free(cplayer->name);
  free(cplayer);
}

/**** configuration file parsing *****/
typedef int (*parse_subnode) (Config *conf,
                              xmlDocPtr doc,xmlNodePtr node, 
                              void *user_data); 

typedef struct { 
  char *nodename;
  parse_subnode func;
} nodeparsers;


static int
parse_xmlnodes(Config *conf, xmlDocPtr doc, xmlNodePtr node, 
               nodeparsers *nodedef, int nrtags, void *user_data) {
  int x;

  while (node != NULL) {
    if (node->type == XML_ELEMENT_NODE) {
      /* try to find the right parser */
      for (x = 0 ; x < nrtags; x++) {
        if (!xmlStrcmp(node->name, (xmlChar *)nodedef[x].nodename)) {
          if (!((nodedef[x].func)(conf,doc,node,user_data))) {
            WARN(_("Parsing of xml node %s failed"),node->name);
            return FALSE;
          }
          break;
        }
      }
      if (x == nrtags) {
        WARN(_("Unknown xml node: %s"),node->name);
        return FALSE;
      }
    }
    node = node->next;
  }
  return TRUE;
}

static int 
parse_hostname (Config *conf, xmlDocPtr doc, xmlNodePtr node, void *user_data) {
  xmlChar *ret;

  ret = xmlNodeListGetString(doc, node->xmlChildrenNode, 1);
  if (ret != NULL) {
    free(conf->hostname);
    conf->hostname = strdup((char *)ret);
    xmlFree(ret);
    return TRUE;
  }
  return FALSE;
}

static int 
parse_resolution(Config *conf, xmlDocPtr doc, xmlNodePtr node, void *user_data){
  char *x,*y;
  x = (char *)xmlGetProp(node, (const xmlChar *)"x");
  y = (char *)xmlGetProp(node, (const xmlChar *)"y");

  if (!strcmp(x,"") || !strcmp(y,"")) {
    xmlFree(x);
    xmlFree(y);
    return FALSE;
  }
  conf->resolution->x = atoi(x);
  conf->resolution->y = atoi(y);
  xmlFree(x);
  xmlFree(y);
  return TRUE;
}


static int 
parse_player_action(Config *conf, xmlDocPtr doc, xmlNodePtr node, 
                    void *user_data) {
  
  Config_key *ckey;
  Config_player *cplay = (Config_player *) user_data;
  char *actionstr[CONFIG_NRKEYS] = {"throw","left","right","jump"}; 
  xmlChar *str;
  int action;
  str = xmlGetProp(node, (const xmlChar *)"name");
  for (action = CONFIG_ACTION_FIRST; action < CONFIG_NRKEYS; action++) {
    if (!xmlStrcmp(str,(const xmlChar *)actionstr[action])) {
      break;
    }
  }
  if (action == CONFIG_NRKEYS) {
    WARN(_("Action tag has unknown name attribute: %s"),str);
    return FALSE;
  }
  xmlFree(str);
  str = xmlGetProp(node, (const xmlChar *)"key");
  ckey = new_config_key(action,atoi((char *)str));
  list_append(cplay->keys,ckey);
  xmlFree(str);
  return TRUE;
}

static int 
parse_player(Config *conf, xmlDocPtr doc, xmlNodePtr node, void *user_data) {
  int id;
  xmlChar *str;
  Config_player *cplay;
  nodeparsers playertags[] = { {"action",parse_player_action} };

  str = xmlGetProp(node, (const xmlChar *)"id");
  if (str == NULL) {
    WARN(_("Player tag without id found"));
    xmlFree(str);
    return FALSE;
  }
  id = atoi((char *)str);
  xmlFree(str);
  str = xmlGetProp(node, (const xmlChar *)"name");
  if (str == NULL) {
    WARN(_("Player tag without name found"));
    xmlFree(str);
    return FALSE;
  }
  cplay = list_search(conf->players, cmp_config_player,&id);
  if (cplay != NULL) {
    list_del(conf->players,cplay);
    del_config_player(cplay);
  }
  cplay = new_config_player(id, (char *)str);
  xmlFree(str);
  list_append(conf->players,cplay);
  id = parse_xmlnodes(conf,doc,node->xmlChildrenNode,playertags,1,cplay);
  return id;
}



static int
parse_config_file(Config *conf, char *filename) {
  xmlDocPtr doc = xmlParseFile(filename);
  xmlNodePtr cur;
  nodeparsers mainnodes[] = { {"hostname",parse_hostname},
                             {"resolution",parse_resolution},
                             {"player",parse_player}
                           }; 
  int nrtags = (sizeof(mainnodes)/(sizeof(void*)*2));
  if (doc == NULL) return FALSE;

  cur = xmlDocGetRootElement(doc);
  if (cur == NULL ||  xmlStrcmp(cur->name, (const xmlChar *) "ffrenzyconfig")) {
    WARN(_("Invalid configuration file: %s"),filename);
    xmlFreeDoc(doc);
    return FALSE;
  }
  cur = cur->xmlChildrenNode;
  if (!parse_xmlnodes(conf, doc, cur, mainnodes, nrtags,NULL)) {
    xmlFreeDoc(doc);
    return FALSE;
  }
  xmlFreeDoc(doc);
  return TRUE;
}

/**** public stuff ***/

Config *new_config(char *configfile) {
  Config *result;
  Config_player *cplay;
  char path[MAXPATHLEN];
  int ret;
  result = malloc(sizeof(Config));
  result->players = new_list();
  result->hostname = NULL;
  result->resolution = new_vector(800,600);

  if (configfile == NULL) {
    snprintf(path,MAXPATHLEN,"%s/.ffrenzy/config.xml",getenv("HOME"));
    ret = parse_config_file(result,path);
  } else {
    ret = parse_config_file(result,configfile);
  }
  if (!ret) {
    WARN(_("Parsing config file failed, using defaults"));
    cplay = new_config_player(0,"player 1");
    list_append(result->players,cplay);
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_SPACE));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_RALT));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_RCTRL));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_RIGHT,SDLK_RIGHT));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_JUMP,SDLK_UP));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_LEFT,SDLK_LEFT));

    cplay = new_config_player(1,"player 2");
    list_append(result->players,cplay);
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_LALT));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_LCTRL));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_THROW,SDLK_TAB));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_RIGHT,SDLK_d));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_JUMP,SDLK_w));
    list_append(cplay->keys,new_config_key(CONFIG_ACTION_LEFT,SDLK_a));
  }
  return result;
}

void
del_config(Config *config) {
  Config_player *cplayer;

  while ((cplayer = list_pop(config->players)) == NULL) {
    del_config_player(cplayer);
  }
  del_list(config->players);
  free(config->hostname);
  del_vector(config->resolution);
  free(config);
}

char *
config_get_pname(Config *config,uint32_t player) {
  Config_player *cplayer = list_search(config->players,
                                      cmp_config_player,&player);

  return cplayer != NULL ? cplayer->name : "Unknown player";
}

List *
config_get_keys(Config *config,uint32_t player) {
  Config_player *cplayer = list_search(config->players,
                                      cmp_config_player,&player);

  return cplayer != NULL ? cplayer->keys : NULL;
}
