/*
  ==============================================================================

   This file is part of the JUCE library.
   Copyright (c) 2017 - ROLI Ltd.

   JUCE is an open source library subject to commercial or open-source
   licensing.

   By using JUCE, you agree to the terms of both the JUCE 5 End-User License
   Agreement and JUCE 5 Privacy Policy (both updated and effective as of the
   27th April 2017).

   End User License Agreement: www.juce.com/juce-5-licence
   Privacy Policy: www.juce.com/juce-5-privacy-policy

   Or: You may also use this code under the terms of the GPL v3 (see
   www.gnu.org/licenses).

   JUCE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES, WHETHER
   EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR PURPOSE, ARE
   DISCLAIMED.

  ==============================================================================
*/

namespace juce
{

//==============================================================================
// This byte-code is generated from native/java/com/roli/juce/JuceWebView.java with min sdk version 16
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const unsigned char JuceWebView16ByteCode[] =
{31,139,8,8,167,106,229,91,0,3,74,117,99,101,87,101,98,86,105,101,119,49,54,46,100,101,120,0,125,150,93,108,84,69,20,199,
207,253,216,189,187,183,237,178,109,161,31,20,74,91,177,84,164,44,88,145,210,5,172,124,83,22,63,186,80,100,53,145,219,
221,177,189,101,123,239,114,239,221,182,38,68,81,155,168,4,19,99,225,141,23,19,36,38,248,136,9,15,62,16,35,209,24,141,47,
70,125,224,193,23,141,15,62,136,49,132,104,31,252,207,157,217,101,129,98,55,191,61,103,206,57,51,115,102,230,116,118,10,
108,206,220,52,176,133,222,95,60,244,253,183,71,91,94,92,240,223,75,222,186,115,227,231,111,110,221,252,44,186,176,245,
181,235,117,68,37,34,154,27,123,178,137,228,223,162,73,52,76,194,190,156,75,133,168,1,242,58,164,14,249,178,74,212,12,
121,26,82,227,49,248,202,196,136,2,200,250,40,218,160,7,172,7,253,224,105,176,7,188,2,78,131,15,193,85,240,35,88,4,45,6,
209,83,224,4,152,7,31,129,27,224,87,16,195,184,107,192,0,216,15,70,192,179,32,11,142,131,19,160,0,108,224,2,15,204,129,
55,192,57,176,0,46,130,143,193,21,112,13,124,1,190,3,63,129,223,192,31,224,95,80,23,39,234,0,235,193,78,112,16,88,96,26,
204,129,215,193,219,224,44,248,0,92,1,95,129,95,192,29,208,104,138,253,192,146,8,169,19,134,36,152,9,102,194,54,83,61,
137,125,76,128,101,32,9,26,1,223,248,102,185,215,43,64,11,104,5,107,64,68,142,119,41,34,108,149,67,106,147,250,167,176,
183,75,253,42,244,14,169,127,14,125,165,212,191,134,190,74,234,63,64,239,148,250,77,232,171,165,126,169,198,254,123,141,
254,55,244,46,153,31,31,167,91,234,60,41,190,182,222,112,141,73,234,147,235,236,13,165,104,71,72,33,17,106,134,50,38,219,
113,82,165,140,82,127,40,235,105,99,40,53,26,146,50,29,142,35,226,76,244,91,23,202,24,165,66,25,167,77,161,52,104,179,
156,119,32,148,17,218,30,202,58,218,17,74,157,118,134,123,47,230,77,86,231,167,80,139,200,189,228,53,29,160,113,89,164,
25,142,167,200,243,171,248,231,225,255,82,250,235,164,63,89,227,63,15,255,95,210,207,179,158,135,126,214,188,171,47,152,
162,207,69,147,199,107,161,222,110,138,122,40,37,185,175,7,227,149,146,124,207,95,74,42,148,107,18,117,162,99,4,62,126,
175,41,234,32,139,195,40,13,71,73,221,156,192,234,35,161,175,223,20,245,38,124,6,124,77,97,125,85,230,217,90,157,71,189,
111,30,13,243,168,225,60,226,172,20,218,99,138,58,61,242,140,70,171,149,22,164,159,219,165,82,167,146,192,8,157,202,58,
89,143,10,62,113,204,169,133,237,195,166,168,231,236,176,74,188,7,206,68,221,6,95,34,180,148,198,18,164,191,208,247,15,
223,79,61,140,31,51,197,218,106,227,7,49,154,136,94,134,232,4,246,88,15,215,123,194,20,245,150,45,61,48,182,167,146,113,
202,152,55,46,24,151,103,162,252,108,251,110,243,179,225,57,169,52,133,126,143,240,26,86,142,188,133,149,168,217,121,244,
199,128,155,53,61,58,168,213,19,111,151,114,203,232,192,5,147,6,49,87,167,218,168,116,170,61,106,148,86,106,91,176,139,
10,61,209,104,116,247,221,110,128,117,157,188,247,154,49,71,119,184,75,252,211,37,37,238,26,83,248,197,174,38,194,123,
176,246,239,204,125,237,115,247,181,121,141,24,184,9,148,154,54,183,232,85,169,146,38,245,70,89,123,252,188,181,170,183,
162,139,49,184,222,136,207,50,89,155,6,50,111,134,53,186,221,118,236,96,39,213,239,158,244,220,105,182,187,104,51,39,160,
168,148,202,8,37,71,202,121,118,140,141,143,217,108,118,227,148,53,99,145,150,201,100,168,61,99,57,5,207,181,11,169,9,
207,42,77,218,121,63,181,203,14,166,173,82,154,58,170,46,135,5,169,201,32,40,165,178,126,113,175,231,185,94,154,150,87,
157,174,159,58,204,124,223,154,96,105,234,170,90,103,217,248,73,59,168,118,56,0,123,145,121,75,68,32,165,218,148,211,180,
118,137,136,81,230,187,101,47,207,32,75,174,227,99,166,182,37,162,248,210,210,212,249,16,79,101,252,190,76,222,157,78,
121,110,209,78,77,97,75,82,53,251,178,246,222,76,122,254,47,82,198,116,60,60,134,15,80,176,138,51,246,201,148,229,56,110,
96,5,182,235,164,246,58,249,162,235,219,206,196,238,162,229,251,60,221,7,99,14,58,14,243,164,191,123,9,255,97,54,61,46,3,
24,66,86,100,248,121,166,108,23,29,75,229,32,27,120,204,154,78,83,147,48,23,45,103,34,245,220,248,20,203,7,247,218,16,
135,52,210,164,140,145,58,54,66,218,216,72,134,116,124,101,40,194,191,51,176,102,96,205,112,43,111,42,57,210,115,161,59,
151,201,229,50,84,103,229,243,56,248,125,69,107,194,167,8,227,199,76,198,171,214,140,157,119,29,50,38,197,137,147,62,233,
250,1,213,241,239,61,172,200,2,86,160,24,111,100,220,252,73,138,115,237,136,123,212,103,20,179,253,61,182,85,116,39,168,
193,246,97,240,246,51,63,40,123,140,116,199,154,102,212,224,58,187,177,111,236,152,237,20,220,89,74,160,137,85,6,53,237,
231,81,129,251,240,79,224,79,98,138,6,209,206,6,150,199,103,108,114,157,81,150,103,246,12,43,84,42,146,226,30,243,203,
197,224,176,63,65,45,254,164,91,46,22,14,58,1,67,145,149,130,81,118,170,140,217,201,20,246,140,107,21,40,30,176,57,254,
95,48,93,36,61,152,180,125,210,202,94,145,34,51,86,177,140,28,103,112,222,212,62,91,169,180,106,162,149,145,86,86,92,53,
73,87,124,173,210,199,19,230,83,85,23,209,114,159,163,178,154,74,135,7,150,20,157,21,187,49,170,172,55,18,106,107,90,157,
154,237,167,227,202,176,145,200,209,9,125,232,241,13,3,92,107,14,189,219,210,234,1,120,219,201,72,236,56,212,185,138,186,
213,161,65,35,113,118,21,110,209,161,193,71,141,196,59,57,122,76,27,234,93,27,218,182,114,231,234,29,239,78,105,180,101,
121,127,119,132,58,214,44,224,50,52,18,164,214,43,131,109,117,106,131,218,163,199,55,180,42,21,69,85,19,202,224,42,181,
45,222,134,31,122,77,37,85,105,210,222,60,163,159,55,180,183,240,59,5,116,229,186,161,40,55,241,131,166,71,84,120,99,240,
46,26,81,233,229,196,149,79,98,136,0,231,226,138,114,13,252,25,231,247,99,35,34,111,154,149,223,103,165,70,14,147,120,
215,242,59,179,242,182,229,247,101,237,251,182,242,198,141,208,221,119,110,148,238,190,117,181,164,208,249,61,175,116,
137,247,194,121,232,209,46,105,71,71,37,41,236,252,93,165,118,137,121,249,219,88,147,241,252,183,95,239,146,111,11,110,
144,125,195,55,72,82,228,202,223,225,255,1,121,75,63,47,192,11,0,0};

//==============================================================================
// This byte-code is generated from native/javacore/app/com/roli/juce/JuceWebView21.java with min sdk version 21
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const unsigned char JuceWebView21ByteCode[] =
{31,139,8,8,20,250,226,91,0,3,74,117,99,101,87,101,98,86,105,101,119,50,49,46,100,101,120,0,149,151,93,108,28,87,21,199,
207,124,236,206,174,119,118,179,222,117,28,219,113,236,181,235,54,78,147,116,227,36,37,78,54,169,204,58,113,237,48,81,
193,78,220,104,145,144,198,187,131,61,206,122,102,51,51,187,142,196,3,161,141,0,209,10,169,130,190,81,41,15,41,170,10,21,
66,66,34,60,240,134,74,16,47,149,120,160,128,84,1,2,169,15,17,42,15,84,81,37,36,254,247,99,54,27,199,68,176,171,223,158,
115,207,57,115,239,61,247,158,153,189,211,112,110,244,29,59,241,60,93,89,253,237,71,111,149,22,172,113,253,199,127,126,
238,195,175,252,238,212,236,173,175,189,250,151,87,111,127,146,37,106,17,209,141,213,147,5,146,159,35,176,45,145,176,15,
130,239,43,68,204,121,31,82,135,252,149,74,52,4,249,55,72,13,242,14,126,238,167,225,131,179,101,160,47,240,77,240,93,240,
6,120,27,188,3,222,3,247,192,159,192,3,144,75,17,29,3,203,96,27,188,5,126,14,126,15,52,244,119,8,44,130,38,120,29,252,4,
188,15,254,8,254,1,254,9,254,5,62,3,212,71,100,128,12,200,131,65,48,12,198,193,51,224,4,56,3,150,192,50,168,129,58,112,
65,7,220,4,175,129,55,193,29,240,46,248,5,248,13,248,3,248,24,252,27,244,103,136,70,192,211,224,28,88,2,151,65,13,52,128,
11,90,160,3,222,0,239,129,247,193,71,224,1,232,55,197,154,33,125,66,154,36,167,78,112,17,92,132,165,167,28,216,3,242,160,
159,196,218,23,193,0,216,43,247,100,31,137,61,24,6,35,96,18,36,128,42,247,48,41,251,255,56,249,80,255,36,41,98,138,50,
102,68,246,201,62,251,165,254,0,49,163,177,29,147,28,147,122,170,71,31,128,126,64,234,37,232,227,82,63,2,125,66,234,39,
123,244,57,232,37,169,179,57,196,118,171,39,230,42,244,167,100,126,172,207,41,169,55,12,177,54,135,249,26,21,232,168,92,
167,195,92,138,182,138,21,125,129,231,172,147,156,54,29,226,57,239,225,237,62,105,207,240,236,153,236,163,227,92,246,211,
9,46,147,84,149,114,158,247,43,226,76,92,119,132,203,44,157,228,50,71,207,115,105,210,231,184,204,208,41,46,21,58,195,
165,42,101,154,206,115,153,167,11,92,166,104,129,75,131,94,228,123,44,230,83,232,206,139,208,131,216,39,246,97,214,187,
104,252,218,36,57,15,225,239,235,241,223,131,255,239,210,159,149,254,66,143,255,67,248,199,179,162,205,106,226,109,196,
254,200,124,168,255,204,20,215,252,210,100,241,26,215,135,77,209,87,43,175,160,61,137,254,90,121,86,103,95,70,187,86,16,
117,169,163,7,214,255,51,166,152,239,10,54,182,53,151,34,117,38,135,236,18,220,119,212,20,123,32,124,105,248,10,124,39,
226,113,78,117,199,209,119,140,163,97,28,149,143,147,224,145,10,45,152,34,255,203,159,215,232,128,50,136,233,215,170,42,
141,41,57,244,48,166,28,228,59,149,36,54,223,52,198,212,120,251,37,83,220,63,43,115,42,177,43,102,144,246,105,248,114,
220,210,90,197,186,127,105,250,51,86,31,58,143,191,106,138,220,122,227,103,209,155,136,46,34,58,135,61,210,121,190,107,
166,184,127,86,90,143,245,29,168,100,92,55,110,25,111,26,63,236,36,7,48,163,233,79,169,123,93,243,127,188,110,111,247,58,
150,139,74,29,83,212,122,65,185,252,10,86,64,93,185,133,235,209,225,140,166,39,103,181,1,98,237,32,175,162,54,76,205,203,
179,103,139,169,181,86,138,180,136,7,217,44,198,30,83,251,149,49,117,82,77,209,136,118,22,187,161,209,241,126,99,98,250,
211,44,172,7,229,243,124,63,198,60,196,87,155,125,167,187,99,191,102,138,123,114,247,177,77,62,118,171,246,255,141,85,
196,8,19,221,177,74,82,18,221,54,133,95,84,66,142,255,151,244,126,242,59,218,163,59,218,172,174,89,5,8,89,228,125,198,
118,102,209,165,76,32,183,88,55,80,1,154,212,139,242,30,98,207,99,173,27,25,235,6,127,126,106,61,125,107,60,151,34,127,
62,235,220,190,23,95,236,245,89,215,115,163,23,200,156,223,8,252,45,103,190,233,58,94,68,73,41,149,139,84,184,216,174,59,
47,59,107,171,174,179,125,124,230,185,77,187,99,147,98,145,102,89,22,13,91,182,215,8,124,183,81,94,15,236,214,134,91,15,
203,85,55,218,178,91,21,234,239,186,60,39,42,95,9,220,10,237,127,196,180,17,69,173,242,74,216,188,16,4,126,80,161,129,
174,211,15,203,151,156,48,180,215,157,10,149,186,214,109,103,237,154,27,117,47,88,132,189,233,4,187,68,96,170,189,169,84,
232,169,93,34,150,157,208,111,7,117,103,217,185,222,118,66,4,77,61,49,40,108,249,94,136,233,12,237,18,197,214,165,66,99,
255,197,19,79,226,89,171,238,111,149,3,191,233,150,55,177,158,229,71,22,117,234,209,9,79,61,57,86,70,29,120,82,84,133,38,
173,134,221,236,184,215,202,182,231,249,145,29,185,190,87,190,224,213,155,126,232,122,235,243,77,59,12,217,164,31,143,89,
242,60,39,144,254,137,93,252,151,156,173,53,25,224,32,100,175,197,10,162,236,250,184,176,213,142,86,162,192,177,183,42,
84,16,230,166,237,173,151,95,90,219,116,234,209,163,54,196,97,26,21,82,86,73,93,189,72,218,234,69,139,116,252,88,148,96,
191,22,172,40,177,85,139,89,89,83,169,145,94,227,238,154,85,171,89,148,177,235,117,212,200,66,211,94,15,41,225,176,138,
160,44,23,241,102,145,241,85,187,227,214,125,143,146,235,78,116,37,104,146,177,33,106,134,244,13,63,140,40,195,126,207,
59,77,39,114,26,148,98,13,203,175,95,163,52,211,46,251,87,208,67,202,13,207,187,118,211,95,167,172,27,194,16,188,136,82,
105,7,14,233,158,189,229,80,214,247,230,177,156,206,203,174,215,240,183,41,135,38,146,143,122,218,95,68,13,47,224,246,10,
55,48,68,86,180,87,34,59,96,35,22,125,111,217,169,59,110,199,105,44,226,78,224,69,77,133,135,198,184,208,201,8,68,141,82,
58,112,194,118,51,186,20,174,211,96,184,225,183,155,141,37,47,114,80,159,173,72,150,49,245,9,187,229,219,13,74,71,206,13,
118,151,109,53,73,143,54,220,144,82,145,47,150,157,180,54,150,35,209,177,155,109,228,210,65,193,208,240,118,92,174,221,
132,226,62,71,98,87,79,114,177,111,159,244,177,196,216,160,221,100,7,119,56,226,172,135,164,253,241,212,247,237,240,116,
243,79,110,139,245,220,84,202,70,78,29,172,168,215,182,143,210,117,101,201,200,213,232,134,94,61,54,115,134,105,3,220,
251,1,85,212,159,126,29,254,17,50,114,231,190,48,54,74,147,106,117,206,200,125,103,148,142,107,213,185,67,70,238,91,53,
186,170,85,79,79,115,219,179,90,245,240,65,174,45,170,213,211,198,129,115,223,254,171,70,103,7,143,78,36,104,255,248,247,
104,150,93,11,227,102,175,113,15,122,38,53,167,204,13,101,213,61,234,211,122,122,102,159,18,43,170,154,87,230,70,213,161,
204,16,78,86,154,74,170,82,72,124,227,166,126,39,165,189,162,146,2,146,202,7,41,69,185,143,211,71,194,80,225,237,131,247,
245,116,74,122,99,50,202,221,52,162,192,15,250,20,229,30,184,153,81,148,187,64,28,130,6,113,213,109,118,30,201,203,255,
12,165,71,198,239,65,236,127,36,126,23,98,207,254,222,247,161,248,157,136,157,37,226,247,162,36,61,124,55,210,242,66,103,
255,105,74,73,156,103,166,160,39,75,194,206,206,122,74,94,156,137,216,121,93,45,201,113,113,136,210,100,60,59,155,233,37,
49,22,59,191,145,188,150,159,17,243,98,174,236,189,237,63,5,143,187,196,240,13,0,0};


#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor,         "<init>",              "(Landroid/content/Context;)V") \
  METHOD (getSettings,         "getSettings",         "()Landroid/webkit/WebSettings;") \
  METHOD (canGoBack,           "canGoBack",           "()Z") \
  METHOD (goBack,              "goBack",              "()V") \
  METHOD (goForward,           "goForward",           "()V") \
  METHOD (loadDataWithBaseURL, "loadDataWithBaseURL", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V") \
  METHOD (loadUrl,             "loadUrl",             "(Ljava/lang/String;Ljava/util/Map;)V") \
  METHOD (postUrl,             "postUrl",             "(Ljava/lang/String;[B)V") \
  METHOD (reload,              "reload",              "()V") \
  METHOD (setWebChromeClient,  "setWebChromeClient",  "(Landroid/webkit/WebChromeClient;)V") \
  METHOD (setWebViewClient,    "setWebViewClient",    "(Landroid/webkit/WebViewClient;)V") \
  METHOD (stopLoading,         "stopLoading",         "()V")

DECLARE_JNI_CLASS (AndroidWebView, "android/webkit/WebView")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor, "<init>", "()V")

DECLARE_JNI_CLASS (AndroidWebChromeClient, "android/webkit/WebChromeClient")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor, "<init>", "()V")

DECLARE_JNI_CLASS (AndroidWebViewClient, "android/webkit/WebViewClient")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  STATICMETHOD (getInstance, "getInstance", "()Landroid/webkit/CookieManager;")

DECLARE_JNI_CLASS (AndroidCookieManager, "android/webkit/CookieManager")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (setBuiltInZoomControls,    "setBuiltInZoomControls",    "(Z)V") \
  METHOD (setDisplayZoomControls,    "setDisplayZoomControls",    "(Z)V") \
  METHOD (setJavaScriptEnabled,      "setJavaScriptEnabled",      "(Z)V") \
  METHOD (setSupportMultipleWindows, "setSupportMultipleWindows", "(Z)V")

DECLARE_JNI_CLASS (WebSettings, "android/webkit/WebSettings")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (toString, "toString", "()Ljava/lang/String;")

DECLARE_JNI_CLASS (SslError, "android/net/http/SslError")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  STATICMETHOD (encode, "encode", "(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;")

DECLARE_JNI_CLASS (URLEncoder, "java/net/URLEncoder")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class WebBrowserComponent::Pimpl    : public AndroidViewComponent,
                                      public AsyncUpdater
{
public:
    Pimpl (WebBrowserComponent& o)
        : owner (o)
    {
        auto* env = getEnv();

        setView (env->NewObject (AndroidWebView, AndroidWebView.constructor, getMainActivity().get()));

        auto settings = LocalRef<jobject> (env->CallObjectMethod ((jobject) getView(), AndroidWebView.getSettings));
        env->CallVoidMethod (settings, WebSettings.setJavaScriptEnabled, true);
        env->CallVoidMethod (settings, WebSettings.setBuiltInZoomControls, true);
        env->CallVoidMethod (settings, WebSettings.setDisplayZoomControls, false);
        env->CallVoidMethod (settings, WebSettings.setSupportMultipleWindows, true);

        juceWebChromeClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebChromeClient, JuceWebChromeClient.constructor,
                                                                            reinterpret_cast<jlong> (this))));
        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebChromeClient, juceWebChromeClient.get());

        auto sdkVersion = getAndroidSDKVersion();

        if (sdkVersion >= 21)
            juceWebViewClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebViewClient21, JuceWebViewClient21.constructor,
                                                                              reinterpret_cast<jlong> (this))));
        else
            juceWebViewClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebViewClient16, JuceWebViewClient16.constructor,
                                                                              reinterpret_cast<jlong> (this))));

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebViewClient, juceWebViewClient.get());
    }

    ~Pimpl()
    {
        auto* env = getEnv();

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.stopLoading);

        auto defaultChromeClient = LocalRef<jobject> (env->NewObject (AndroidWebChromeClient, AndroidWebChromeClient.constructor));
        auto defaultViewClient   = LocalRef<jobject> (env->NewObject (AndroidWebViewClient,   AndroidWebViewClient  .constructor));

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebChromeClient, defaultChromeClient.get());
        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebViewClient,   defaultViewClient  .get());

        masterReference.clear();

        // if other Java thread is waiting for us to respond to page load request
        // wake it up immediately (false answer will be sent), so that it releases
        // the lock we need when calling hostDeleted.
        responseReadyEvent.signal();

        env->CallVoidMethod (juceWebViewClient, getAndroidSDKVersion() >= 21 ? JuceWebViewClient21.hostDeleted
                                                                             : JuceWebViewClient16.hostDeleted);
    }

    void goToURL (const String& url,
                  const StringArray* headers,
                  const MemoryBlock* postData)
    {
        auto* env = getEnv();

        if (headers == nullptr && postData == nullptr)
        {
            env->CallVoidMethod ((jobject) getView(), AndroidWebView.loadUrl, javaString (url).get(), 0);
        }
        else if (headers != nullptr && postData == nullptr)
        {
            auto headersMap = LocalRef<jobject> (env->NewObject (JavaHashMap,
                                                                 JavaHashMap.constructorWithCapacity,
                                                                 headers->size()));

            for (const auto& header : *headers)
            {
                auto name  = header.upToFirstOccurrenceOf (":", false, false).trim();
                auto value = header.fromFirstOccurrenceOf (":", false, false).trim();

                env->CallObjectMethod (headersMap, JavaMap.put,
                                       javaString (name).get(),
                                       javaString (value).get());
            }

            env->CallVoidMethod ((jobject) getView(), AndroidWebView.loadUrl,
                                 javaString (url).get(), headersMap.get());
        }
        else if (headers == nullptr && postData != nullptr)
        {
            auto dataStringJuce = postData->toString();
            auto dataStringJava = javaString (dataStringJuce);
            auto encodingString = LocalRef<jobject> (env->CallStaticObjectMethod (URLEncoder, URLEncoder.encode,
                                                                                  dataStringJava.get(), javaString ("utf-8").get()));

            auto bytes = LocalRef<jbyteArray> ((jbyteArray) env->CallObjectMethod (encodingString, JavaString.getBytes));

            env->CallVoidMethod ((jobject) getView(), AndroidWebView.postUrl,
                                 javaString (url).get(), bytes.get());
        }
        else if (headers != nullptr && postData != nullptr)
        {
            // There is no support for both extra headers and post data in Android WebView, so
            // we need to open URL manually.

            URL urlToUse = URL (url).withPOSTData (*postData);
            connectionThread.reset (new ConnectionThread (*this, urlToUse, *headers));
        }
    }

    void stop()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.stopLoading);
    }

    void goBack()
    {
        connectionThread = nullptr;

        auto* env = getEnv();
        auto view = (jobject) getView();

        if (env->CallBooleanMethod (view, AndroidWebView.canGoBack))
            env->CallVoidMethod (view, AndroidWebView.goBack);
        else
            owner.reloadLastURL();
    }

    void goForward()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.goForward);
    }

    void refresh()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.reload);
    }

    void handleAsyncUpdate()
    {
        jassert (connectionThread != nullptr);

        if (connectionThread == nullptr)
            return;

        auto& result = connectionThread->getResult();

        if (result.statusCode >= 200 && result.statusCode < 300)
        {
            auto url = javaString (result.url);
            auto data = javaString (result.data);
            auto mimeType = javaString ("text/html");
            auto encoding = javaString ("utf-8");

            getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.loadDataWithBaseURL,
                                      url.get(), data.get(), mimeType.get(),
                                      encoding.get(), 0);
        }
        else
        {
            owner.pageLoadHadNetworkError (result.description);
        }
    }

    bool handlePageAboutToLoad (const String& url)
    {
        if (MessageManager::getInstance()->isThisTheMessageThread())
            return owner.pageAboutToLoad (url);

        WeakReference<Pimpl> weakRef (this);

        if (weakRef == nullptr)
            return false;

        responseReadyEvent.reset();

        bool shouldLoad = false;

        MessageManager::callAsync ([weakRef, url, &shouldLoad]
        {
            if (weakRef == nullptr)
                return;

            shouldLoad = weakRef->owner.pageAboutToLoad (url);

            weakRef->responseReadyEvent.signal();
        });

        responseReadyEvent.wait (-1);

        return shouldLoad;
    }

    WebBrowserComponent& owner;

private:
    class ConnectionThread  : private Thread
    {
    public:
        struct Result
        {
            String url;
            int statusCode = 0;
            String description;
            String data;
        };

        ConnectionThread (Pimpl& ownerToUse,
                          URL& url,
                          const StringArray& headers)
            : Thread ("WebBrowserComponent::Pimpl::ConnectionThread"),
              owner (ownerToUse),
              webInputStream (new WebInputStream (url, true))
        {
            webInputStream->withExtraHeaders (headers.joinIntoString ("\n"));
            webInputStream->withConnectionTimeout (10000);

            result.url = url.toString (true);

            startThread();
        }

        ~ConnectionThread() override
        {
            webInputStream->cancel();
            signalThreadShouldExit();
            waitForThreadToExit (10000);

            webInputStream = nullptr;
        }

        void run() override
        {
            if (! webInputStream->connect (nullptr))
            {
                result.description = "Could not establish connection";
                owner.triggerAsyncUpdate();
                return;
            }

            result.statusCode = webInputStream->getStatusCode();
            result.description = "Status code: " + String (result.statusCode);
            readFromInputStream();
            owner.triggerAsyncUpdate();
        }

        const Result& getResult() { return result; }

    private:
        void readFromInputStream()
        {
            MemoryOutputStream ostream;

            for (;;)
            {
                if (threadShouldExit())
                    return;

                char buffer [8192];
                auto num = webInputStream->read (buffer, sizeof (buffer));

                if (num <= 0)
                    break;

                ostream.write (buffer, (size_t) num);
            }

            result.data = ostream.toUTF8();
        }

        Pimpl& owner;
        std::unique_ptr<WebInputStream> webInputStream;
        Result result;
    };

    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD (constructor, "<init>",      "(J)V") \
     METHOD (hostDeleted, "hostDeleted", "()V") \
     CALLBACK (webViewReceivedHttpError, "webViewReceivedHttpError", "(JLandroid/webkit/WebView;Landroid/webkit/WebResourceRequest;Landroid/webkit/WebResourceResponse;)V") \
     CALLBACK (webViewPageLoadStarted, "webViewPageLoadStarted", "(JLandroid/webkit/WebView;Ljava/lang/String;)Z") \
     CALLBACK (webViewPageLoadFinished, "webViewPageLoadFinished", "(JLandroid/webkit/WebView;Ljava/lang/String;)V") \
     CALLBACK (webViewReceivedSslError, "webViewReceivedSslError", "(JLandroid/webkit/WebView;Landroid/webkit/SslErrorHandler;Landroid/net/http/SslError;)V") \

     DECLARE_JNI_CLASS_WITH_BYTECODE (JuceWebViewClient21, "com/roli/juce/JuceWebView21$Client", 21, JuceWebView21ByteCode, sizeof (JuceWebView21ByteCode))
    #undef JNI_CLASS_MEMBERS

    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD (constructor, "<init>",      "(J)V") \
     METHOD (hostDeleted, "hostDeleted", "()V") \
     CALLBACK (webViewPageLoadStarted, "webViewPageLoadStarted", "(JLandroid/webkit/WebView;Ljava/lang/String;)Z") \
     CALLBACK (webViewPageLoadFinished, "webViewPageLoadFinished", "(JLandroid/webkit/WebView;Ljava/lang/String;)V") \
     CALLBACK (webViewReceivedSslError, "webViewReceivedSslError", "(JLandroid/webkit/WebView;Landroid/webkit/SslErrorHandler;Landroid/net/http/SslError;)V") \

     DECLARE_JNI_CLASS_WITH_BYTECODE (JuceWebViewClient16, "com/roli/juce/JuceWebView$Client", 16, JuceWebView16ByteCode, sizeof (JuceWebView16ByteCode))
    #undef JNI_CLASS_MEMBERS

    static jboolean JNICALL webViewPageLoadStarted (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jstring url)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            return myself->handlePageAboutToLoad (juceString (url));

        return 0;
    }

    static void JNICALL webViewPageLoadFinished (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jstring url)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.pageFinishedLoading (juceString (url));
    }

    static void JNICALL webViewReceivedHttpError (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jobject /*request*/, jobject errorResponse)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->webReceivedHttpError (errorResponse);
    }

    static void JNICALL webViewReceivedSslError (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jobject /*sslErrorHandler*/, jobject sslError)
    {
        auto* env = getEnv();

        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
        {
            auto errorString = LocalRef<jstring> ((jstring) env->CallObjectMethod (sslError, SslError.toString));

            myself->owner.pageLoadHadNetworkError (juceString (errorString));
        }
    }

    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
      METHOD (constructor, "<init>",      "(J)V") \
      CALLBACK (webViewCloseWindowRequest,  "webViewCloseWindowRequest",  "(JLandroid/webkit/WebView;)V") \
      CALLBACK (webViewCreateWindowRequest, "webViewCreateWindowRequest", "(JLandroid/webkit/WebView;)V") \

    DECLARE_JNI_CLASS (JuceWebChromeClient, "com/roli/juce/JuceWebView$ChromeClient")
    #undef JNI_CLASS_MEMBERS

    static void JNICALL webViewCloseWindowRequest (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.windowCloseRequest();
    }

    static void JNICALL webViewCreateWindowRequest (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.newWindowAttemptingToLoad ({});
    }

    //==============================================================================
    void webReceivedHttpError (jobject errorResponse)
    {
        auto* env = getEnv();

        LocalRef<jclass> responseClass (env->FindClass ("android/webkit/WebResourceResponse"));

        if (responseClass != nullptr)
        {
            jmethodID method = env->GetMethodID (responseClass, "getReasonPhrase", "()Ljava/lang/String;");

            if (method != nullptr)
            {
                auto errorString = LocalRef<jstring> ((jstring) env->CallObjectMethod (errorResponse, method));

                owner.pageLoadHadNetworkError (juceString (errorString));
                return;
            }
        }

        // Should never get here!
        jassertfalse;
        owner.pageLoadHadNetworkError ({});
    }

    //==============================================================================
    GlobalRef juceWebChromeClient;
    GlobalRef juceWebViewClient;
    std::unique_ptr<ConnectionThread> connectionThread;
    WaitableEvent responseReadyEvent;

    WeakReference<Pimpl>::Master masterReference;
    friend class WeakReference<Pimpl>;
};

//==============================================================================
WebBrowserComponent::WebBrowserComponent (const bool unloadWhenHidden)
    : blankPageShown (false),
      unloadPageWhenBrowserIsHidden (unloadWhenHidden)
{
    setOpaque (true);

    browser.reset (new Pimpl (*this));
    addAndMakeVisible (browser.get());
}

WebBrowserComponent::~WebBrowserComponent()
{
}

//==============================================================================
void WebBrowserComponent::goToURL (const String& url,
                                   const StringArray* headers,
                                   const MemoryBlock* postData)
{
    lastURL = url;

    if (headers != nullptr)
        lastHeaders = *headers;
    else
        lastHeaders.clear();

    if (postData != nullptr)
        lastPostData = *postData;
    else
        lastPostData.reset();

    blankPageShown = false;

    browser->goToURL (url, headers, postData);
}

void WebBrowserComponent::stop()
{
    browser->stop();
}

void WebBrowserComponent::goBack()
{
    browser->goBack();

    lastURL.clear();
    blankPageShown = false;
}

void WebBrowserComponent::goForward()
{
    lastURL.clear();

    browser->goForward();
}

void WebBrowserComponent::refresh()
{
    browser->refresh();
}

//==============================================================================
void WebBrowserComponent::paint (Graphics& g)
{
    g.fillAll (Colours::white);
}

void WebBrowserComponent::checkWindowAssociation()
{
    if (isShowing())
    {
        if (blankPageShown)
            goBack();
    }
    else
    {
        if (unloadPageWhenBrowserIsHidden && ! blankPageShown)
        {
            // when the component becomes invisible, some stuff like flash
            // carries on playing audio, so we need to force it onto a blank
            // page to avoid this, (and send it back when it's made visible again).

            blankPageShown = true;
            browser->goToURL ("about:blank", nullptr, nullptr);
        }
    }
}

void WebBrowserComponent::reloadLastURL()
{
    if (lastURL.isNotEmpty())
    {
        goToURL (lastURL, &lastHeaders, lastPostData.getSize() == 0 ? nullptr : &lastPostData);
        lastURL.clear();
    }
}

void WebBrowserComponent::parentHierarchyChanged()
{
    checkWindowAssociation();
}

void WebBrowserComponent::resized()
{
    browser->setSize (getWidth(), getHeight());
}

void WebBrowserComponent::visibilityChanged()
{
    checkWindowAssociation();
}

void WebBrowserComponent::focusGained (FocusChangeType)
{
}

void WebBrowserComponent::clearCookies()
{
    auto* env = getEnv();

    auto cookieManager = LocalRef<jobject> (env->CallStaticObjectMethod (AndroidCookieManager,
                                                                         AndroidCookieManager.getInstance));

    jmethodID clearCookiesMethod = nullptr;

    if (getAndroidSDKVersion() >= 21)
    {
        clearCookiesMethod = env->GetMethodID (AndroidCookieManager, "removeAllCookies", "(Landroid/webkit/ValueCallback;)V");
        env->CallVoidMethod (cookieManager, clearCookiesMethod, 0);
    }
    else
    {
        clearCookiesMethod = env->GetMethodID (AndroidCookieManager, "removeAllCookie", "()V");
        env->CallVoidMethod (cookieManager, clearCookiesMethod);
    }
}

WebBrowserComponent::Pimpl::JuceWebViewClient16_Class   WebBrowserComponent::Pimpl::JuceWebViewClient16;
WebBrowserComponent::Pimpl::JuceWebViewClient21_Class   WebBrowserComponent::Pimpl::JuceWebViewClient21;
WebBrowserComponent::Pimpl::JuceWebChromeClient_Class WebBrowserComponent::Pimpl::JuceWebChromeClient;
} // namespace juce
