package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.Visitor;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.transaction.GlobalTransaction;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Implements functionality defined by {@link org.jboss.cache.Cache#put(String, java.util.Map)}.
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class PutDataMapCommand extends AbstractVersionedDataCommand
{
   public static final int METHOD_ID = 1;
   public static final int ERASE_METHOD_ID = 2;
   public static final int VERSIONED_METHOD_ID = 37;
   public static final int ERASE_VERSIONED_METHOD_ID = 38;

   private static final Log log = LogFactory.getLog(PutDataMapCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /* parameters*/
   private Map data;
   private Map oldData;

   public PutDataMapCommand(GlobalTransaction globalTransaction, Fqn fqn, Map data)
   {
      this.globalTransaction = globalTransaction;
      this.fqn = fqn;
      this.data = data;
   }

   public PutDataMapCommand()
   {
   }

   /**
    * Adds the provided data map to the data map in the node referenced by the specified Fqn.
    */
   public Object perform(InvocationContext ctx)
   {
      if (trace)
      {
         log.trace("perform(" + globalTransaction + ", \"" + fqn + "\", " + data + ")");
      }
      NodeSPI nodeSPI = dataContainer.peekStrict(globalTransaction, fqn, false);
      Map existingData = nodeSPI.getDataDirect();
      if (!existingData.isEmpty())
      {
         oldData = new HashMap(existingData); // defensive copy
      }
      if (notifier.shouldNotifyOnNodeModified())
      {
         notifier.notifyNodeModified(fqn, true, NodeModifiedEvent.ModificationType.PUT_MAP, oldData == null ? Collections.emptyMap() : oldData, ctx);
      }

      nodeSPI.putAllDirect(data);
      if (notifier.shouldNotifyOnNodeModified())
      {
         notifier.notifyNodeModified(fqn, false, NodeModifiedEvent.ModificationType.PUT_MAP, nodeSPI.getDataDirect(), ctx);
      }
      return null;
   }

   public void rollback()
   {
      if (trace) log.trace("rollback(" + globalTransaction + ", " + fqn + ", " + data + ")");
      NodeSPI n = dataContainer.peek(fqn, false, true);
      if (n != null)
      {
         n.clearDataDirect();
         if (oldData != null) n.putAllDirect(oldData);
      }
   }

   public Object acceptVisitor(InvocationContext ctx, Visitor visitor) throws Throwable
   {
      return visitor.visitPutDataMapCommand(ctx, this);
   }

   public Map getData()
   {
      return data;
   }

   public void setData(Map data)
   {
      this.data = data;
   }

   public int getCommandId()
   {
      if (isVersioned())
      {
         return VERSIONED_METHOD_ID;
      } else
      {
         return METHOD_ID;
      }
   }

   @Override
   public Object[] getParameters()
   {
      if (isVersioned())
         return new Object[]{globalTransaction, fqn, data, false, dataVersion};
      else
         return new Object[]{globalTransaction, fqn, data, false};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      globalTransaction = (GlobalTransaction) args[0];
      fqn = (Fqn) args[1];
      data = (Map) args[2];
      if (isVersionedId(commandId)) dataVersion = (DataVersion) args[4];
   }

   @Override
   protected boolean isVersionedId(int id)
   {
      return id == VERSIONED_METHOD_ID || id == ERASE_VERSIONED_METHOD_ID;
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;
      PutDataMapCommand that = (PutDataMapCommand) o;
      if (data != null ? !data.equals(that.data) : that.data != null) return false;
      if (globalTransaction != null ? !globalTransaction.equals(that.globalTransaction) : that.globalTransaction != null)
         return false;

      return true;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + (globalTransaction != null ? globalTransaction.hashCode() : 0);
      result = 31 * result + (data != null ? data.hashCode() : 0);
      return result;
   }

   @Override
   public String toString()
   {
      return "PutDataMapCommand{" +
            "fqn=" + fqn +
            ", dataVersion=" + dataVersion +
            ", data=" + data +
            ", globalTransaction=" + globalTransaction +
            '}';
   }

   Map getOldData()
   {
      return oldData;
   }
}
