/*
 * Copyright 2010, 2011, 2012, 2013 mapsforge.org
 * Copyright 2014 Ludwig M Brinckmann
 * Copyright 2015-2018 devemux86
 *
 * This program is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.mapsforge.map.layer.overlay;

import org.mapsforge.core.graphics.Canvas;
import org.mapsforge.core.graphics.GraphicFactory;
import org.mapsforge.core.graphics.Paint;
import org.mapsforge.core.graphics.Path;
import org.mapsforge.core.model.BoundingBox;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.core.model.Point;
import org.mapsforge.core.util.LatLongUtils;
import org.mapsforge.core.util.MercatorProjection;
import org.mapsforge.map.layer.Layer;

import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * A {@code Polygon} draws a closed connected series of line segments specified by a list of {@link LatLong LatLongs}.
 * If the first and the last {@code LatLong} are not equal, the {@code Polygon} will be closed automatically.
 * <p/>
 * A {@code Polygon} holds two {@link Paint} objects to allow for different outline and filling. These paints define
 * drawing parameters such as color, stroke width, pattern and transparency.
 */
public class Polygon extends Layer {

    private BoundingBox boundingBox;
    private final GraphicFactory graphicFactory;
    private final boolean keepAligned;
    private final List<LatLong> latLongs = new CopyOnWriteArrayList<>();
    private Paint paintFill;
    private Paint paintStroke;

    /**
     * @param paintFill      the initial {@code Paint} used to fill this polygon (may be null).
     * @param paintStroke    the initial {@code Paint} used to stroke this polygon (may be null).
     * @param graphicFactory the GraphicFactory
     */
    public Polygon(Paint paintFill, Paint paintStroke, GraphicFactory graphicFactory) {
        this(paintFill, paintStroke, graphicFactory, false);
    }

    /**
     * @param paintFill      the initial {@code Paint} used to fill this polygon (may be null).
     * @param paintStroke    the initial {@code Paint} used to stroke this polygon (may be null).
     * @param graphicFactory the GraphicFactory
     * @param keepAligned    if set to true it will keep the bitmap aligned with the map,
     *                       to avoid a moving effect of a bitmap shader.
     */
    public Polygon(Paint paintFill, Paint paintStroke, GraphicFactory graphicFactory, boolean keepAligned) {
        super();
        this.keepAligned = keepAligned;
        this.paintFill = paintFill;
        this.paintStroke = paintStroke;
        this.graphicFactory = graphicFactory;
    }

    public synchronized void addPoint(LatLong point) {
        this.latLongs.add(point);
        updatePoints();
    }

    public synchronized void addPoints(List<LatLong> points) {
        this.latLongs.addAll(points);
        updatePoints();
    }

    public synchronized void clear() {
        this.latLongs.clear();
        updatePoints();
    }

    public synchronized boolean contains(LatLong tapLatLong) {
        return LatLongUtils.contains(latLongs, tapLatLong);
    }

    @Override
    public synchronized void draw(BoundingBox boundingBox, byte zoomLevel, Canvas canvas, Point topLeftPoint) {
        if (this.latLongs.size() < 2 || (this.paintStroke == null && this.paintFill == null)) {
            return;
        }

        if (this.boundingBox != null && !this.boundingBox.intersects(boundingBox)) {
            return;
        }

        Iterator<LatLong> iterator = this.latLongs.iterator();

        Path path = this.graphicFactory.createPath();
        LatLong latLong = iterator.next();
        long mapSize = MercatorProjection.getMapSize(zoomLevel, displayModel.getTileSize());
        float x = (float) (MercatorProjection.longitudeToPixelX(latLong.longitude, mapSize) - topLeftPoint.x);
        float y = (float) (MercatorProjection.latitudeToPixelY(latLong.latitude, mapSize) - topLeftPoint.y);
        path.moveTo(x, y);

        while (iterator.hasNext()) {
            latLong = iterator.next();
            x = (float) (MercatorProjection.longitudeToPixelX(latLong.longitude, mapSize) - topLeftPoint.x);
            y = (float) (MercatorProjection.latitudeToPixelY(latLong.latitude, mapSize) - topLeftPoint.y);
            path.lineTo(x, y);
        }


        if (this.paintStroke != null) {
            if (this.keepAligned) {
                this.paintStroke.setBitmapShaderShift(topLeftPoint);
            }
            canvas.drawPath(path, this.paintStroke);
        }
        if (this.paintFill != null) {
            if (this.keepAligned) {
                this.paintFill.setBitmapShaderShift(topLeftPoint);
            }

            canvas.drawPath(path, this.paintFill);
        }
    }

    /**
     * @return a thread-safe list of LatLongs in this polygon.
     */
    public List<LatLong> getLatLongs() {
        return this.latLongs;
    }

    /**
     * @return the {@code Paint} used to fill this polygon (may be null).
     */
    public synchronized Paint getPaintFill() {
        return this.paintFill;
    }

    /**
     * @return the {@code Paint} used to stroke this polygon (may be null).
     */
    public synchronized Paint getPaintStroke() {
        return this.paintStroke;
    }

    /**
     * @return true if it keeps the bitmap aligned with the map, to avoid a
     * moving effect of a bitmap shader, false otherwise.
     */
    public boolean isKeepAligned() {
        return keepAligned;
    }

    /**
     * @param paintFill the new {@code Paint} used to fill this polygon (may be null).
     */
    public synchronized void setPaintFill(Paint paintFill) {
        this.paintFill = paintFill;
    }

    /**
     * @param paintStroke the new {@code Paint} used to stroke this polygon (may be null).
     */
    public synchronized void setPaintStroke(Paint paintStroke) {
        this.paintStroke = paintStroke;
    }

    public synchronized void setPoints(List<LatLong> points) {
        this.latLongs.clear();
        this.latLongs.addAll(points);
        updatePoints();
    }

    private void updatePoints() {
        this.boundingBox = this.latLongs.isEmpty() ? null : new BoundingBox(this.latLongs);
    }
}
