/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/Debug.h>
#include <GTLCore/Image.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/TimeMachine.h>
#include <GTLCore/Type.h>
#include <GTLCore/Value.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>

// OpenShiva Headers
#include <OpenShiva/Kernel.h>
#include <OpenShiva/Version.h>

// ShivAnimator Headers
#include "Parser.h"
#include "ProgressBar.h"

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName() << " - " << OpenShiva::LibraryName() << " - " << OpenShiva::LibraryVersionString() << std::endl;
  std::cout << OpenShiva::LibraryCopyright() << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion() << std::endl;
}

void printHelp()
{
  std::cout << "Usage : shivanimator [option] fileName.shivanimation output" << std::endl;
  std::cout << "output must be a pattern of file that will be used to save the animation" << std::endl;
  std::cout << "For instance: output%04d.png will save the animation in output0000.png, output0001.png..." << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String input = "";
  GTLCore::String output = "";
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(input.isEmpty()) {
      input = argv[ai];
    } else {
      output = argv[ai];
    }
  }
  if(input.isEmpty() or output.isEmpty())
  {
    printHelp();
    return EXIT_SUCCESS;
  }
  
  int width = 800;
  int height = 600;
  
  shivanimator::Parser parser;
  if(parser.parse(input))
  {
    OpenShiva::Kernel* kernel = parser.kernel();
    GTL_ASSERT(kernel);
    GTL_DEBUG("Animate from " << parser.start() << " to " << parser.end() << " at " << parser.frameRate() << " images/s");
    double t = parser.start();
    double inct = 1.0 / parser.frameRate();
    int i = 0;
    GTLCore::PixelDescription pixel( GTLCore::Type::UnsignedInteger8, 4 );
    GTLCore::Image image( width, height, pixel );
    shivanimator::ProgressBar* bar = new shivanimator::ProgressBar(kernel->name(), (parser.end() - parser.start())/inct + 1);
    while(t < parser.end())
    {
      kernel->setParameters(parser.timeMachine()->parametersAt(t));
      kernel->setParameter(OpenShiva::Kernel::IMAGE_WIDTH, GTLCore::Value((float)width));
      kernel->setParameter(OpenShiva::Kernel::IMAGE_HEIGHT, GTLCore::Value((float)height));
      
      // Evaluate pixeles
      kernel->evaluatePixels( GTLCore::RegionI(0,0,width, height), std::list<const GTLCore::AbstractImage*>(), &image );

      // Generate output name
      char buffer[255];
      sprintf(buffer, output.c_str(), i);
      GTLCore::String outputFileName(buffer);
      
      // Encode image
      const GTLImageIO::ImageDC* encoder = GTLImageIO::ImageDCRegistry::instance()->encoder( outputFileName );
      if( not encoder )
      {
        std::cerr << "Can't find encoder for " << outputFileName << std::endl;
        return EXIT_FAILURE;
      }
      GTLCore::String errMsg;
      if( not encoder->encode( &image, image.boundingBox(), outputFileName, 0, &errMsg ) )
      {
        std::cerr << "Can't encode " << outputFileName << " : " << errMsg << std::endl;
        return EXIT_FAILURE;
      }
      ++i;
      bar->next();
      t += inct;
    }
    delete bar;
  } else {
    return EXIT_FAILURE;
  }
}
