use presentation;
use core:geometry;
use graphics;
use ui;

/**
 * Class representing the axis environment for a horizontal bar graph. This class is basically a
 * layout manager for the individual components of the graph.
 */
class BarGraph extends Element {
	// The axis owned by the graph.
	package GraphAxis axis;

	// Labels in use.
	package GraphLabel[] labels;

	// Data sets to plot.
	package GraphData[] data;

	// Animation state of the length of bars.
	package Float barLength;

	// Legend, if we have one.
	private Legend? legend;

	// Legend anchor pos.
	private Cardinal legendAnchor;

	// Create.
	init(Presentation p) {
		init() {
			axis(p.contentStyle);
			barLength = 1.0;
			legendAnchor = northEast;
		}
	}

	// Set the font used in the axis.
	void axisStyle(TextStyle style) {
		axis.textStyle(style);
	}

	// Set the label style.
	void labelStyle(TextStyle style) {
		axis.labelStyle(style);
	}

	// Set the ticks on the non-data axis.
	void ticks(Float min, Float step, Float max) {
		axis.ticks(min, step, max);
	}

	// Set labels on the y-axis.
	void labels(Str[] labels) {
		this.labels.clear();

		for (l in labels)
			this.labels << GraphLabel(Text(l, axis.textStyle.font));
	}

	// Indent a specific label to the right margin. Used to indicate sub-groups.
	void indentLabel(Nat id) {
		labels[id].indent = axis.marginBefore;
	}

	// Set the indentation color.
	void indentColor(Color color) {
		axis.indentColor = SolidBrush(color);
	}

	// Scale a specific label relative to the default size.
	void scaleLabel(Nat id, Float scale) {
		scaleLabel(id, scale, 1.0);
	}

	void scaleLabel(Nat id, Float heightScale, Float textScale) {
		labels[id].displayScale = heightScale;
		labels[id].textScale = textScale;
	}

	// Margin between individual bars.
	void barMargin(Float v) {
		axis.barMargin = v;
	}

	// Margin between groups of bars.
	void barGroupMargin(Float v) {
		axis.barGroupMargin = v;
	}

	// Add some data.
	void data(Float[] values, Color color) {
		data << GraphData(values, color);
	}

	// Generate labels for the bars by using a lambda function.
	void dataLabels(fn(Float)->Str fn) {
		for (d in data)
			d.labels(fn, axis.labelStyle.font);
	}

	// Set size of the arrow.
	void arrowSize(Float size) {
		axis.arrowSize = size;
	}

	// Create a legend.
	void legend(Cardinal anchor, Str[] labels) {
		legend(axis.textStyle, anchor, labels);
	}
	void legend(TextStyle style, Cardinal anchor, Str[] labels) {
		legendAnchor = anchor;

		Legend l(style);
		for (i, label in labels) {
			if (i < data.count)
				l.series(label, data[i].color);
		}

		l.pos = Rect(Point(), l.minSize);

		legend = l;
	}

	// Get the minimum size of the layout.
	Size minSize() {
		Size(0, 0);
	}

	// Animation handling.
	void draw(Graphics g, Nat step, Duration time) : override {
		for (label in labels)
			label.animationDefault = 0xFFFFFFFF;

		if (before(g, step, time))
			draw(g);
		after(g, step, time);
	}

	// Helper to draw the legend.
	protected void drawLegend(Graphics g, Rect graph) {
		if (legend) {
			// Adjust to position the legend at a reasonable location.
			graph.p0.x += axis.arrowSize;
			graph.p1.y -= axis.arrowSize;
			legend.pos = legendAnchor.align(graph, legend.pos.size);
			legend.draw(g);
		}
	}
}

// Helper for creating the bar graph.
XBarGraph barGraph(Presentation p) {
	// This was the previous default.
	XBarGraph(p);
}

/**
 * X-axis bar graph.
 */
class XBarGraph extends BarGraph {
	// Create.
	init(Presentation p) {
		init(p) {}
	}

	// Set margin on the left.
	void leftMargin(Float margin) {
		axis.marginBefore = margin;
	}

	// Set margin on the right.
	void rightMargin(Float margin) {
		axis.marginAfter = margin;
	}

	// Maximum height of a single bar.
	void barMaxHeight(Float v) {
		axis.barMaxSize = v;
	}

	// Draw this element.
	void draw(Graphics g) : override {
		Rect graph = pos;
		graph.p1.y -= axis.tickHeight() + axis.labelMargin;

		Float numBars = 0;
		for (label in labels)
			numBars += label.scale;

		Float leftMargin;
		Float height = graph.size.h / numBars;
		Float barHeight = (height - axis.barGroupMargin) / data.count.float;
		barHeight = min(barHeight, axis.barMaxSize);
		height = barHeight * data.count.float + axis.barGroupMargin;

		Float y;
		for (label in labels) {
			Float h = height * label.scale;
			label.totalSize = h;
			label.size = (h - axis.barGroupMargin) / data.count.float;
			label.offset = y + h / 2;
			y += h;

			leftMargin = max(leftMargin, label.text.size.w);
		}

		graph.p0.x += leftMargin + axis.labelMargin;

		// In case we made the graph smaller.
		graph.p1.y = y + graph.p0.y;

		// Compute offsets for each data set.
		for (i, d in data) {
			d.offset = i.float - (data.count - 1).float / 2;
		}

		// Draw data.
		for (d in data) {
			d.drawX(g, graph, labels, axis, barLength);
		}

		// Draw the axis itself.
		axis.drawX(g, graph, labels);

		// Draw legend if necessary.
		drawLegend(g, graph);
	}
}

XBarGraph xBarGraph(Presentation p) {
	XBarGraph(p);
}

/**
 * Y-axis bar graph.
 */
class YBarGraph extends BarGraph {
	// Create.
	init(Presentation p) {
		init(p) {}
	}

	// Set margin on the left.
	void bottomMargin(Float margin) {
		axis.marginBefore = margin;
	}

	// Set margin on the right.
	void topMargin(Float margin) {
		axis.marginAfter = margin;
	}

	// Maximum height of a single bar.
	void barMaxWidth(Float v) {
		axis.barMaxSize = v;
	}

	// Draw this element.
	void draw(Graphics g) : override {
		Rect graph = pos;
		graph.p0.x += axis.tickWidth() + axis.labelMargin;

		Float numBars = 0;
		for (label in labels)
			numBars += label.scale;

		Float bottomMargin;
		Float width = graph.size.w / numBars;
		Float barWidth = (width - axis.barGroupMargin) / data.count.float;
		barWidth = min(barWidth, axis.barMaxSize);
		width = barWidth * data.count.float + axis.barGroupMargin;

		Float x;
		for (label in labels) {
			Float w = width * label.scale;
			label.totalSize = w;
			label.size = (w - axis.barGroupMargin) / data.count.float;
			label.offset = x + w / 2;
			x += w;

			bottomMargin = max(bottomMargin, label.text.size.h);
		}

		graph.p1.y -= bottomMargin + axis.labelMargin;

		// In case we made the graph smaller.
		graph.p1.x = x + graph.p0.x;

		// Compute offsets for each data set.
		for (i, d in data) {
			d.offset = i.float - (data.count - 1).float / 2;
		}

		// Draw data.
		for (d in data) {
			d.drawY(g, graph, labels, axis, barLength);
		}

		// Draw the axis itself.
		axis.drawY(g, graph, labels);

		// Draw legend if necessary.
		drawLegend(g, graph);
	}
}

YBarGraph yBarGraph(Presentation p) {
	YBarGraph(p);
}

/**
 * Class responsible for drawing the axis itself.
 */
class GraphAxis on Render {
	// The style of the text on the axis.
	TextStyle textStyle;

	// Style of labels.
	TextStyle labelStyle;

	// Color of indentations.
	Brush indentColor;

	// Created tick labels.
	private Text[] ticks;

	// Minimum value.
	private Float min;

	// Maximum value.
	private Float max;

	// Step for the ticks.
	private Float step;

	// Text color. Copy so that we can freely modify its opacity!
	private SolidBrush textBrush;

	// Label color. Copy so that we can modify its opacity.
	package SolidBrush labelBrush;

	// Margin on the left side (pixels).
	Float marginBefore;

	// Margin on the right side (pixels).
	Float marginAfter;

	// Size of the markers.
	Float markSize;

	// Margin between the axis and the labels.
	Float labelMargin;

	// Margin between groups of bars.
	Float barGroupMargin;

	// Margin between individual bars.
	Float barMargin;

	// Maximum size of a single bar.
	Float barMaxSize;

	// The arrow we will draw.
	private Path arrow;

	// Size of the arrow.
	private Float arrowSz;

	// Create.
	init(TextStyle style) {
		var brush = brushColor(style.fill);
		init() {
			textStyle = style;
			labelStyle = style;
			indentColor = SolidBrush(white * 0.5);
			textBrush = SolidBrush(brush);
			labelBrush = SolidBrush(brush);
			min = 0;
			max = 1;
			step = 1;
			marginBefore = 0;
			marginAfter = 20;
			markSize = 4;
			labelMargin = 4;
			barGroupMargin = 10;
			barMargin = 5;
			barMaxSize = 40;
		}

		ticks(0, 1, 1);
		arrowSize(10);
	}

	void textStyle(TextStyle style) {
		textStyle = style;
		textBrush = SolidBrush(brushColor(style.fill));
	}

	void labelStyle(TextStyle style) {
		labelStyle = style;
		labelBrush = SolidBrush(brushColor(style.fill));
	}

	// Helper to find the color of a brush.
	private Color brushColor(Brush b) : static {
		Color c = black;
		if (b as SolidBrush) {
			c = b.color;
			c.a *= b.opacity;
		} else {
			print("WARNING: We ignore non-solid text styles for labels in a graph!");
		}
		c;
	}

	// Create ticks for the graph.
	void ticks(Float min, Float step, Float max) {
		ticks.clear();
		this.min = min;
		this.max = max;
		this.step = step;

		Int index = 0;
		do {
			Float at = min + index.float*step;
		} while (at <= max) {
			ticks << Text("${at}", textStyle.font);
			index++;
		}
	}

	// Compute the height of the tick labels.
	Float tickHeight() {
		Float h;
		for (l in ticks)
			h = max(h, l.size.h);
		h;
	}

	// Compute the width of the tick labels.
	Float tickWidth() {
		Float w;
		for (l in ticks)
			w = max(w, l.size.w);
		w;
	}

	// Draw the axis, assuming X-bars.
	void drawX(Graphics g, Rect graph, GraphLabel[] labels) {
		// The Y axis.
		if (marginBefore > 0.1) {
			g.line(graph.p0, Point(graph.p0.x, graph.p1.y), textStyle.fill);
		} else {
			g.line(graph.p0, Point(graph.p0.x, graph.p1.y + markSize / 2), textStyle.fill);
		}

		Point indentFrom(0, graph.p0.y);
		Float indentTo;

		for (l in labels) {
			if (l.scale > 0) {
				Size sz = l.text.size;
				Point align = graph.p0 + Point(0, l.offset);

				g.line(align - Point(markSize / 2, 0), align, textStyle.fill);

				textBrush.opacity = l.animationScale;

				align.x -= labelMargin;
				if (l.textScale == 1.0) {
					align -= Size(sz.w, sz.h / 2);
					g.draw(l.text, textBrush, align);
				} else {
					g.push();
					g.transform(scale(l.textScale) * translate(align));
					g.draw(l.text, textBrush, Point(-sz.w, -sz.h / 2));
					g.pop();
				}

				textBrush.opacity = 1;

				if (indentFrom.x != l.indent) {
					Float y = graph.p0.y + l.offset - l.totalSize/2;
					if (indentFrom.x > 0) {
						indentFrom.x += graph.p0.x;
						Rect bump = Rect(indentFrom, Point(graph.p0.x, y)).normalized;
						g.fill(bump, indentColor);
						g.draw(bump, textStyle.fill);
					} else {
						indentTo = indentFrom.y + l.totalSize;
					}

					indentFrom = Point(l.indent, y);
				}
			}
		}

		if (indentFrom.x > 0) {
			indentFrom.x += graph.p0.x;
			Rect bump = Rect(indentFrom, Point(graph.p0.x, indentTo)).normalized;
			g.fill(bump, indentColor);
			g.draw(bump, textStyle.fill);
		}

		// The X axis.
		g.line(Point(graph.p0.x, graph.p1.y), graph.p1, textStyle.fill);

		g.push();
		g.transform(translate(graph.p1));
		g.draw(arrow, textStyle.fill);
		g.pop();

		for (id, text in ticks) {
			Float x = min + step*id.float;
			Point p(xCoordToPx(graph, x), graph.p1.y);

			if (id > 0 | marginBefore > 0.1)
				g.line(p - Point(0, markSize/2), p + Point(0, markSize/2), textStyle.fill);

			Size sz = text.size;
			p -= Size(sz.w / 2, -labelMargin);
			g.draw(text, textStyle.fill, p);
		}
	}

	// Draw the axis, assuming X bars.
	void drawY(Graphics g, Rect graph, GraphLabel[] labels) {
		Point startPt(graph.p0.x, graph.p1.y);

		// The Y axis.
		if (marginBefore > 0.1)
			g.line(startPt, graph.p1, textStyle.fill);
		else
			g.line(startPt - Point(markSize / 2, 0), graph.p1, textStyle.fill);

		Point indentFrom(graph.p0.x, 0);
		Float indentTo;

		for (l in labels) {
			if (l.scale > 0) {
				Size sz = l.text.size;
				Point align = startPt + Point(l.offset, 0);

				g.line(align + Point(0, markSize / 2), align, textStyle.fill);

				textBrush.opacity = l.animationScale;

				align.y += labelMargin;
				if (l.textScale == 1.0) {
					align -= Size(sz.w / 2, 0);
					g.draw(l.text, textBrush, align);
				} else {
					g.push();
					g.transform(scale(l.textScale) * translate(align));
					g.draw(l.text, textBrush, Point(-sz.w / 2, 0));
					g.pop();
				}

				textBrush.opacity = 1;

				if (indentFrom.y != l.indent) {
					Float x = graph.p0.x + l.offset - l.totalSize/2;
					if (indentFrom.y > 0) {
						indentFrom.y = graph.p1.y - indentFrom.y;
						Rect bump = Rect(indentFrom, Point(x, graph.p1.y)).normalized;
						g.fill(bump, indentColor);
						g.draw(bump, textStyle.fill);
					} else {
						indentTo = indentFrom.x + l.totalSize;
					}

					indentFrom = Point(x, l.indent);
				}
			}
		}

		if (indentFrom.y > 0) {
			indentFrom.y = graph.p1.y - indentFrom.y;
			Rect bump = Rect(indentFrom, Point(indentTo, graph.p1.y)).normalized;
			g.fill(bump, indentColor);
			g.draw(bump, textStyle.fill);
		}

		// The X axis.
		g.line(graph.p0, startPt, textStyle.fill);

		g.push();
		g.transform(rotate(-90 deg) * translate(graph.p0));
		g.draw(arrow, textStyle.fill);
		g.pop();

		for (id, text in ticks) {
			Float y = min + step*id.float;
			Point p(graph.p0.x, yCoordToPx(graph, y));

			if (id > 0 | marginBefore > 0.1)
				g.line(p - Point(markSize/2, 0), p + Point(markSize/2, 0), textStyle.fill);

			Size sz = text.size;
			p -= Size(labelMargin + sz.w, sz.h / 2);
			g.draw(text, textStyle.fill, p);
		}
	}

	// Compute the pixel value of a coordinate (x coordinate)
	Float xCoordToPx(Rect graph, Float y) {
		Float l = graph.p0.x + marginBefore;
		Float r = graph.p1.x - marginAfter;
		Float step = (r - l) / (max - min);
		l + step*(y - min);
	}

	// Compute the pixel value of a coordinate (y coordinate)
	Float yCoordToPx(Rect graph, Float y) {
		Float t = graph.p0.y + marginAfter;
		Float b = graph.p1.y - marginBefore;
		Float step = (t - b) / (max - min);
		b + step*(y - min);
	}

	// Set the arrow size.
	assign arrowSize(Float size) {
		arrowSz = size;
		arrow.clear();

		Float height = size * 2 / 3;
		arrow.start(Point(-size, -height));
		arrow.line(Point(0, 0));
		arrow.line(Point(-size, height));
	}

	Float arrowSize() { arrowSz; }
}

/**
 * Class representing a label in a graph.
 */
class GraphLabel on Render {
	// The title of this label.
	Text text;

	// X/Y-offset of this label.
	Float offset;

	// Label indentation (i.e. start bars at this x-offset).
	Float indent;

	// Size of each bar in this label.
	Float size;

	// Total height of all bars in this label.
	Float totalSize;

	// Scale of this label (used to show/hide parts of the graph). Not shown if zero.
	Float scale() {
		animationScale * displayScale;
	}

	// Scaling applied by animations.
	Float animationScale;

	// Scaling applied by config.
	Float displayScale;

	// Text scale for this label (not the text inside the graph).
	Float textScale;

	// The lowest animation ID that has set the default value for 'scale'.
	Nat animationDefault;

	// Create.
	init(Text text) {
		init {
			text = text;
			animationScale = 1.0;
			displayScale = 1.0;
			textScale = 1.0;
		}
	}
}

/**
 * Class representing one set of data-points in a graph.
 */
class GraphData on Render {
	// A single value.
	value Value {
		// The value itself. Used for plotting.
		Float point;

		// Text used as a label (optional).
		Text? label;

		// Create.
		init(Float p) {
			init { point = p; }
		}
	}

	// Data.
	Value[] values;

	// Offset of this graph (relative to the "height").
	Float offset;

	// Color for this data set.
	SolidBrush color;

	// Create the data.
	init(Float[] values, Color color) {
		init() {
			color(color);
		}

		for (v in values)
			this.values << Value(v);
	}

	// Create data labels.
	void labels(fn(Float)->Str fn, Font font) {
		for (Nat i = 0; i < values.count; i++) {
			if (values[i].label) {
			} else {
				values[i].label = Text(fn.call(values[i].point), font);
			}
		}
	}

	// Draw this graph.
	void drawX(Graphics g, Rect graph, GraphLabel[] labels, GraphAxis axis, Float lengthScale) {
		for (i, label in labels) {
			if (i < values.count) {
				drawBarX(g, graph, label, values[i], axis, lengthScale);
			}
		}
	}

	// Draw a single bar.
	private void drawBarX(Graphics g, Rect graph, GraphLabel label, Value value, GraphAxis axis, Float lengthScale) {
		if (lengthScale <= 0.0)
			return;

		Float y = label.offset + graph.p0.y;
		y += offset * label.size;

		Float from = graph.p0.x + label.indent;
		Float to = axis.xCoordToPx(graph, value.point);
		if (lengthScale < 1.0)
			to = from + (to - from)*lengthScale;

		Float height = (label.size - axis.barMargin) / 2;
		if (height <= 0.0)
			return;

		Rect bar(Point(from, y - height), Point(to, y + height));
		g.draw(bar, axis.labelStyle.fill);
		g.fill(bar, color);

		if (text = value.label) {
			Float scale = label.textScale;

			Size sz = text.size;
			Point anchor(to + axis.labelMargin, y - sz.h * scale / 2);
			if (anchor.x + sz.w * scale > graph.p1.x)
				anchor.x = to - axis.labelMargin - sz.w * scale;

			axis.labelBrush.opacity = lengthScale;

			if (scale == 1.0) {
				g.draw(text, axis.labelBrush, anchor);
			} else {
				g.push();
				g.transform(scale(scale) * translate(anchor));
				g.draw(text, axis.labelBrush, Point());
				g.pop();
			}

			axis.labelBrush.opacity = 1.0;
		}
	}

	// Draw this graph.
	void drawY(Graphics g, Rect graph, GraphLabel[] labels, GraphAxis axis, Float lengthScale) {
		for (i, label in labels) {
			if (i < values.count) {
				drawBarY(g, graph, label, values[i], axis, lengthScale);
			}
		}
	}

	// Draw a single bar.
	private void drawBarY(Graphics g, Rect graph, GraphLabel label, Value value, GraphAxis axis, Float lengthScale) {
		if (lengthScale <= 0.0)
			return;

		Float x = label.offset + graph.p0.x;
		x += offset * label.size;

		Float from = graph.p1.y - label.indent;
		Float to = axis.yCoordToPx(graph, value.point);
		if (lengthScale < 1.0)
			to = from + (to - from)*lengthScale;

		Float width = (label.size - axis.barMargin) / 2;
		if (width <= 0.0)
			return;

		Rect bar(Point(x - width, from), Point(x + width, to));
		g.draw(bar, axis.labelStyle.fill);
		g.fill(bar, color);

		if (text = value.label) {
			Float scale = label.textScale;

			Size sz = text.size;
			Point anchor(x - sz.w*scale/2, to - axis.labelMargin - sz.h*scale);
			if (anchor.y < graph.p0.y)
				anchor.y = to + axis.labelMargin;

			axis.labelBrush.opacity = lengthScale;

			if (scale == 1.0) {
				g.draw(text, axis.labelBrush, anchor);
			} else {
				g.push();
				g.transform(scale(scale) * translate(anchor));
				g.draw(text, axis.labelBrush, Point());
				g.pop();
			}

			axis.labelBrush.opacity = 1.0;
		}
	}
}


/**
 * Element for the legend. Usable both inside the BarGraph itself, and as a separate element for
 * convenience.
 */
class Legend extends Element {
	// Text style.
	TextStyle style;

	// Text elements in the legend.
	private Text[] labels;

	// Colors for each label.
	private Brush[] colors;

	// Size of the squares.
	private Float sqSize;

	// Margin.
	private Size margin;

	// Fill background.
	Brush fill;

	// Border color.
	Brush border;

	// Create.
	init(TextStyle style) {
		init {
			style = style;
			fill = SolidBrush(white);
			border = style.fill;
			margin(8, 6);
		}
	}

	// Set margin.
	void margin(Float w, Float h) {
		margin = Size(w, h);
	}

	// Set fill color.
	void fill(Color c) {
		fill = SolidBrush(c);
	}

	// Set border color.
	void border(Color c) {
		border = SolidBrush(c);
	}

	// Add a series.
	void series(Str label, Color color) { series(label, SolidBrush(color)); }
	void series(Str label, Brush brush) {
		var text = Text(label, style.font);
		labels << text;
		colors << brush;

		sqSize = sqSize.max(text.lineInfo[0].baseline);
	}

	// Minimum size.
	Size minSize() {
		Size sz;
		for (l in labels) {
			sz.w = max(sz.w, l.size.w);
			sz.h += l.size.h;
		}

		sz.w += sqSize * 1.5;
		return sz + margin*2;
	}

	// Draw.
	void draw(Graphics g) : override {
		Rect p = pos;
		g.fill(p, fill);
		g.draw(p, border);

		Point at = p.p0 + margin;
		for (i, l in labels) {
			Float h = l.size.h;
			Rect box(at + Point(0, (h - sqSize) / 2), Size(sqSize));

			g.fill(box, colors[i]);
			g.draw(box, border);

			g.draw(l, style.fill, at + Point(sqSize * 1.5, 0));

			at.y += h;
		}
	}
}



/**
 * Grow the bars in the table.
 */
class GrowBarsAnimation extends Animation {
	init(Nat step) {
		init(step) {}
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (element as BarGraph) {
			if (cStep < step) {
				element.barLength = 0;
			} else if (cStep == step) {
				element.barLength = smoothVal(time);
			} else {
				element.barLength = 1;
			}
		}
		true;
	}

	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}


/**
 * Animation that shows a particular sub-graph at the given time.
 */
class ShowLabelAnimation extends Animation {
	// The label id we shall work with.
	private Nat labelId;

	init(Nat step, Nat id) {
		init(step) {
			labelId = id;
		}
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (element as BarGraph) {
			var label = element.labels[labelId];

			if (cStep <= step) {
				if (label.animationDefault > step) {
					label.animationScale = 0;
					label.animationDefault = step;
				}
			} else {
				label.animationScale = 1;
			}


			if (cStep == step) {
				label.animationScale = smoothVal(time);
			}
		}

		true;
	}
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}

/**
 * Animation that shows a particular sub-graph at the given time.
 */
class HideLabelAnimation extends Animation {
	// The label id we shall work with.
	private Nat labelId;

	init(Nat step, Nat id) {
		init(step) {
			labelId = id;
		}
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (element as BarGraph) {
			var label = element.labels[labelId];

			if (cStep <= step) {
				if (label.animationDefault > step) {
					label.animationScale = 1;
					label.animationDefault = step;
				}
			} else {
				label.animationScale = 0;
			}

			if (step == cStep) {
				label.animationScale = 1.0 - smoothVal(time);
			}
		}

		true;
	}
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}
