// Copyright (C) 2006-2009 Kent-Andre Mardal and Simula Research Laboratory
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.

#include <sstream>

#include "Polygon.h"
#include "tools.h"
#include "symbol_factory.h"

//using namespace std;

using std::string;
using GiNaC::ex;
using GiNaC::lst;
using GiNaC::exvector;
using GiNaC::ex_to;
using GiNaC::is_a;
using GiNaC::numeric;

namespace SyFi
{

	//---        Polygon ------------------------------------
	//

	Polygon::Polygon(const string & subscript_, unsigned int no_vert):
	subscript(subscript_),
		p(no_vert)
	{
	}

	Polygon::Polygon(const Polygon& polygon) :
	subscript(polygon.str()),
		p(polygon.no_vertices())
	{
		for (unsigned int i=0; i<p.size(); i++)
		{
			p[i] = polygon.vertex(i);
		}
	}

	unsigned int Polygon::no_vertices() const
	{
		return p.size();
	}

	ex Polygon::vertex (unsigned int i) const
	{
		if ( i < 0 || i > p.size()-1 )
		{
			throw std::out_of_range("Vertex index is out of range!");
		}
		return p[i];
	}

	Line Polygon::line(unsigned int i) const
	{
		throw std::logic_error("line not implemented for this polygon subclass");
	}

	Triangle Polygon::triangle(unsigned int i) const
	{
		throw std::logic_error("triangle not implemented for this polygon subclass");
	}

	Rectangle Polygon::rectangle(unsigned int i) const
	{
		throw std::logic_error("rectangle not implemented for this polygon subclass");
	}

	//---        Line ------------------------------------
	//

	Line::Line(ex x0, ex x1, const string & subscript_):
	Polygon(subscript_, 2)
	{
		/* Lines on the form
		 * x = x_0 + a_1 t
		 * y = y_0 + a_2 t
		 * z = z_0 + a_3 t
		 */

		p[0] = x0;
		p[1] = x1;

		//update internal structure
		if ( nsd == 1 )
		{
			a_ = x1-x0;
			b_ = x0;
		}
		else if (  (GiNaC::is_a<lst>(x0) && GiNaC::is_a<lst>(x1)) && (x0.nops() == x1.nops()) )
		{
			// TODO: support matrix in addition to lst?
			lst aa;
			lst bb;

			// compute aa and bb
			for (unsigned int i=0; i<= x0.nops()-1; i++)
			{
				bb.append(x0.op(i));
				aa.append(x1.op(i) - x0.op(i));
			}

			a_ = aa;
			b_ = bb;

		}
		else
		{
			throw(std::logic_error("The points x0 and x1 needs to be of type lst if nsd > 1."));
		}
	}

	Line::Line(const Line& line): Polygon(line), a_(line.a_), b_(line.b_)
	{
	}

	unsigned int Line::no_space_dim() const { return 1; }

	ex Line::a() const
	{
		return a_;
	}

	ex Line::b() const
	{
		return b_;
	}

	ex Line::repr(Repr_format format) const
	{
		return repr(GiNaC::symbol("t"), format);
	}

	ex Line::repr(ex t, Repr_format format) const
	{
		/* NOTE: use the same symbols in this
		 * description as in the actual symbols
		 * below and in the documentation.
		 *
		 * Lines on the form
		 * x = x_0 + a_1 t,
		 * y = y_0 + a_2 t,
		 * z = z_0 + a_3 t.
		 * for t in [0,1]
		 */

		lst r;

		if ( format == SUBS_PERFORMED )
		{
			if (!GiNaC::is_a<lst>(a_))
			{
				r = lst( x == b_ +  a_*t);
			}
			else
			{
				if ( a_.nops() == 1)
				{
					r = lst( x == b_.op(0) +  a_.op(0)*t);
				}				 // 2D
				else if ( a_.nops() == 2)
				{
					r = lst(  x == b_.op(0) +  a_.op(0)*t,
						y == b_.op(1) +  a_.op(1)*t);
					// 3D
				}
				else if ( a_.nops() == 3)
				{
					r = lst(  x == b_.op(0) +  a_.op(0)*t,
						y == b_.op(1) +  a_.op(1)*t,
						z == b_.op(2) +  a_.op(2)*t);
				}
			}
			r.append(lst(t,0,1));
		}
		else if ( format == SUBS_NOT_PERFORMED )
		{

			// NOTE: decide how the constant should be !!
			GiNaC::symbol a1("A"+subscript);
			GiNaC::symbol a2("C"+subscript);
			GiNaC::symbol a3("E"+subscript);
			GiNaC::symbol b1("B"+subscript);
			GiNaC::symbol b2("D"+subscript);
			GiNaC::symbol b3("F"+subscript);

			// 2D
			if ( a_.nops() == 2)
			{
				r = lst(  x == b1 +  a1*t,
					y == b2 +  a2*t);

				r.append( a1 == a_.op(0));
				r.append( b1 == b_.op(0));
				r.append( a2 == a_.op(1));
				r.append( b2 == b_.op(1));
				// 3D
			}
			else if ( a_.nops() == 3)
			{
				r = lst(  x == b1 +  a1*t,
					y == b2 +  a2*t,
					z == b3 +  a3*t);

				r.append( a1 == a_.op(0));
				r.append( b1 == b_.op(0));
				r.append( a2 == a_.op(1));
				r.append( b2 == b_.op(1));
				r.append( a3 == a_.op(2));
				r.append( b3 == b_.op(2));
			}
			r.append(lst(t,0,1));
		}
		return r;
	}

	const string Line::str() const
	{
		std::ostringstream s;
		//    s <<"Line("<<p[0]<<","<<p[1]<<")";
		// FIXME: would like to use the above code, but this causes a strange crash in Python
		s <<"Line";
		return s.str();
	}

	ex Line::integrate(ex f, Repr_format format)
	{
		ex ret;

		if ( format == SUBS_PERFORMED)
		{
			GiNaC::symbol t("t");
			ex t_repr = repr(t);
			lst sub;
			int counter=0;
			// perform substitution
			if ( p[0].nops() == 3)
			{
				sub = lst(t_repr.op(0), t_repr.op(1), t_repr.op(2));
				counter = 3;
			}
			else if ( p[0].nops() == 2)
			{
				sub = lst(t_repr.op(0), t_repr.op(1));
				counter = 2;
			}
			else if ( p[0].nops() == 1)
			{
				sub = lst(t_repr.op(0));
				counter = 1;
			}
			else if ( p[0].nops() == 0)
			{
				sub = lst(t_repr.op(0));
				counter = 1;
			}

			// compute D
			ex D;
			if ( p[0].nops() == 3)
			{
				D = pow(t_repr.op(0).rhs().coeff(t), 2) +
					pow(t_repr.op(1).rhs().coeff(t), 2) +
					pow(t_repr.op(2).rhs().coeff(t), 2);
			}
			else if ( p[0].nops() == 2)
			{
				D = pow(t_repr.op(0).rhs().coeff(t), 2) +
					pow(t_repr.op(1).rhs().coeff(t), 2);
			}
			else if ( p[0].nops() == 1)
			{
				D = pow(t_repr.op(0).rhs().coeff(t), 2);
			}
			else if ( p[0].nops() == 0)
			{
				D = pow(t_repr.op(0).rhs().coeff(t), 2);
			}

			D = sqrt(D);

			ex intf = f.subs(sub)*D;

			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			ret  = eval_integ(intf);

		}
		else if ( format == SUBS_NOT_PERFORMED )
		{
			GiNaC::symbol t("t");
			ex t_repr = repr(t);
			lst sub;
			GiNaC::symbol a("a"), b("b"), c("c"), D("D");
			int counter=0;
			// perform substitution

			if ( p[0].nops() == 3)
			{
				x ==  p[0].op(0) + a*t,
				//      sub = lst(t_repr.op(0), t_repr.op(1), t_repr.op(2));
					sub = lst( x ==  p[0].op(0) + a*t,
					y ==  p[0].op(1) + b*t,
					z ==  p[0].op(2) + c*t);
				counter = 3;
			}
			else if ( p[0].nops() == 2)
			{
				//      sub = lst(t_repr.op(0), t_repr.op(1));
				sub = lst( x ==  p[0].op(0) + a*t,
					y ==  p[0].op(1) + b*t);
				counter = 2;
			}

			// compute D
			ex DD;
			if ( p[0].nops() == 3)
			{
				DD = pow(t_repr.op(0).rhs().coeff(t), 2) +
					pow(t_repr.op(1).rhs().coeff(t), 2) +
					pow(t_repr.op(2).rhs().coeff(t), 2);
			}
			else if ( p[0].nops() == 2)
			{
				DD = pow(t_repr.op(0).rhs().coeff(t), 2) +
					pow(t_repr.op(1).rhs().coeff(t), 2);
			}

			DD = sqrt(DD);

			ex intf = f.subs(sub);

			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			intf = eval_integ(intf);

			ret = intf*D;

		}
		else
		{
			throw std::runtime_error("Invalid format!");
		}

		return ret;
	}

	Line* Line::copy() const
	{
		return new Line(*this);
	}

	//---        ReferenceLine --------------------------------
	//

	ReferenceLine::ReferenceLine(const string & subscript_):
	Line(lst(0,0,0), lst(1,0,0), subscript_)
	{
	}

	ReferenceLine::ReferenceLine(const ReferenceLine& line): Line(line) { }

	ex ReferenceLine::repr(ex t, Repr_format format) const
	{
		lst r;
		r = lst( x == t, y == 0, z == 0, t, 0, 1);
		return r;
	}

	const string ReferenceLine::str() const
	{
		std::ostringstream s;
		//    s <<"ReferenceLine("<<p[0]<<","<<p[1]<<")";
		s <<"ReferenceLine";
		return s.str();
	}

	ex ReferenceLine::integrate(ex f, Repr_format formt)
	{
		ex intf = GiNaC::integral(x,0,1,f);
		intf =  eval_integ(intf);
		return intf;
	}

	ReferenceLine* ReferenceLine::copy() const
	{
		return new ReferenceLine(*this);
	}

	//---        Triangle ------------------------------------

	Triangle::Triangle(const string & subscript_):
	Polygon(subscript_, 3)
	{
		// only used for the implementation of ReferenceTriangle
	}

	Triangle::Triangle(ex x0, ex x1, ex x2, const string & subscript_):
	Polygon(subscript_, 3)
	{
		p[0] = x0;
		p[1] = x1;
		p[2] = x2;
	}

	Triangle::Triangle(const Triangle& triangle): Polygon(triangle) { }

	unsigned int Triangle::no_space_dim() const { return 2; }

	Line Triangle::line(unsigned int i) const
	{

		if      ( i == 0) return Line(p[1],p[2], istr(subscript,i));
		else if ( i == 1) return Line(p[0],p[2], istr(subscript,i));
		else if ( i == 2) return Line(p[0],p[1], istr(subscript,i));

		throw std::out_of_range("Line index is out of range!");
	}

	ex Triangle::repr(Repr_format format) const
	{
		GiNaC::symbol r("r"), s("s");
		GiNaC::symbol G("G"), H("H");
		ex l1_repr = Line(vertex(0), vertex(1)).repr(r);
		ex l2_repr = Line(vertex(0), vertex(2)).repr(s);
		lst ret;
		//2D
		if ( p[0].nops() == 2)
		{
			ret = lst( x == l1_repr.op(0).rhs().coeff(r,0)
				+  l1_repr.op(0).rhs().coeff(r,1)*r
				+  l2_repr.op(0).rhs().coeff(s,1)*s,
				y == l1_repr.op(1).rhs().coeff(r,0)
				+  l1_repr.op(1).rhs().coeff(r,1)*r
				+  l2_repr.op(1).rhs().coeff(s,1)*s);

		}
		else if ( p[0].nops() == 3)
		{

			ret = lst( x == l1_repr.op(0).rhs().coeff(r,0)
				+ l1_repr.op(0).rhs().coeff(r,1)*r
				+ l2_repr.op(0).rhs().coeff(s,1)*s,
				y == l1_repr.op(1).rhs().coeff(r,0)
				+ l1_repr.op(1).rhs().coeff(r,1)*r
				+ l2_repr.op(1).rhs().coeff(s,1)*s,
				z == l1_repr.op(2).rhs().coeff(r,0)
				+ l1_repr.op(2).rhs().coeff(r,1)*r
				+ l2_repr.op(2).rhs().coeff(s,1)*s);

		}

		ret.append(lst(r, 0, 1));
		ret.append(lst(s, 0, 1 - r));

		return ret;

	}

	const string Triangle::str() const
	{
		std::ostringstream s;
		//    s <<"Triangle("<<p[0]<<","<<p[1]<<","<<p[2]<<")";
		s <<"Triangle";
		return s.str();
	}

	ex Triangle::integrate(ex func, Repr_format format)
	{
		ex ret;

		if ( format == SUBS_PERFORMED )
		{
			ex t_repr = repr();
			lst sub;
			int counter=0;

			// perform substitution
			if ( p[0].nops() == 3)
			{
				sub = lst(t_repr.op(0), t_repr.op(1), t_repr.op(2));
				counter = 3;
			}
			else if ( p[0].nops() == 2)
			{
				sub = lst(t_repr.op(0), t_repr.op(1));
				counter = 2;
			}
			ex intf = func.subs(sub);

			// compute D
			ex D;
			if ( p[0].nops() == 3)
			{
				ex r = t_repr.op(3).op(0);
				ex s = t_repr.op(4).op(0);
				ex a = t_repr.op(0).rhs().coeff(r,1);
				ex b = t_repr.op(0).rhs().coeff(s,1);
				ex c = t_repr.op(1).rhs().coeff(r,1);
				ex d = t_repr.op(1).rhs().coeff(s,1);
				ex e = t_repr.op(2).rhs().coeff(r,1);
				ex f = t_repr.op(2).rhs().coeff(s,1);
				D = pow(c*f-d*e,2) + pow(a*f - b*e,2) + pow(a*d - b*c,2);
				D = sqrt(D);
			}
			else if ( p[0].nops() == 2)
			{
				ex r = t_repr.op(2).op(0);
				ex s = t_repr.op(3).op(0);
				ex a = t_repr.op(0).rhs().coeff(r,1);
				ex b = t_repr.op(0).rhs().coeff(s,1);
				ex c = t_repr.op(1).rhs().coeff(r,1);
				ex d = t_repr.op(1).rhs().coeff(s,1);
				D = abs(a*d-b*c);
			}

			intf = intf*D;

			counter++;
			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			intf = eval_integ(intf);

			counter--;
			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			ret = eval_integ(intf);

		}
		else if ( format == SUBS_NOT_PERFORMED )
		{
			ex t_repr = repr();
			lst sub;
			int counter=0;

			GiNaC::symbol a("a"), b("b"), c("c"), d("d"), e("e"), f("f"), r("r"), s("s"), D("D");

			// perform substitution
			if ( p[0].nops() == 3)
			{
				sub = lst(x == p[0].op(0) + a*r + b*s,
					y == p[0].op(1) + c*r + d*s,
					z == p[0].op(2) + e*r + f*s);
				counter = 3;
			}
			else if ( p[0].nops() == 2)
			{
				sub = lst(t_repr.op(0), t_repr.op(1));
				sub = lst(x == p[0].op(0) + a*r + b*s,
					y == p[0].op(1) + c*r + d*s);
				counter = 2;
			}

			ex intf = func.subs(sub);

			counter++;
			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			intf = eval_integ(intf);

			counter--;
			intf = GiNaC::integral(t_repr.op(counter).op(0), t_repr.op(counter).op(1), t_repr.op(counter).op(2), intf);
			intf = eval_integ(intf);

			ret = intf*D;

		}
		else
		{
			throw std::runtime_error("Invalid format!");
		}

		return ret;
	}

	Triangle* Triangle::copy() const
	{
		return new Triangle(*this);
	}

	//---        ReferenceTriangle ----------------------------

	ReferenceTriangle::ReferenceTriangle(const string & subscript_):
	Triangle(subscript_)
	{
		ex x0;
		ex x1;
		ex x2;
		if ( nsd ==  3)
		{
			x0 = lst(0,0,0);
			x1 = lst(1,0,0);
			x2 = lst(0,1,0);
		}
		else if ( nsd == 2 )
		{
			x0 = lst(0,0);
			x1 = lst(1,0);
			x2 = lst(0,1);
		}
		p[0] = x0;
		p[1] = x1;
		p[2] = x2;
	}

	ReferenceTriangle::ReferenceTriangle(const ReferenceTriangle& triangle): Triangle(triangle) { }

	const string ReferenceTriangle::str() const
	{
		std::ostringstream s;
		//    s <<"ReferenceTriangle("<<p[0]<<","<<p[1]<<","<<p[2]<<")";
		s <<"ReferenceTriangle";
		return s.str();
	}

	ex ReferenceTriangle::integrate(ex f, Repr_format format)
	{
		ex ret = GiNaC::eval_integ(
			GiNaC::integral(y,0,1,
			GiNaC::eval_integ(
			GiNaC::integral(x,0,1-y,f))));
		return ret;
	}

	// ---------- Rectangle --------------------------------------

	Rectangle::Rectangle(const string & subscript_):
	Polygon(subscript_,4)
	{
		// only used for the implementation of ReferenceRectangle
	}

	Rectangle::Rectangle(ex p0, ex p1, const string & subscript_):
	Polygon(subscript_,4)
	{
		ex x0;
		ex x1;
		ex x2;
		ex x3;

		if (p0.nops() == 2 && p1.nops() == 2)
		{
			// Follows the UFC numbering convention for a reference quadrilateral:
			x0 = lst(p0.op(0), p0.op(1));
			x1 = lst(p1.op(0), p0.op(1));
			x2 = lst(p1.op(0), p1.op(1));
			x3 = lst(p0.op(0), p1.op(1));

		}
		else if (p0.nops() == 3 && p1.nops() == 3)
		{
			if ( p0.op(0) == p1.op(0) )
			{

				x0 = lst(p0.op(0), p0.op(1), p0.op(2));
				x1 = lst(p0.op(0), p1.op(1), p0.op(2));
				x2 = lst(p0.op(0), p1.op(1), p1.op(2));
				x3 = lst(p0.op(0), p0.op(1), p1.op(2));

			}
			else if ( p0.op(1) == p1.op(1) )
			{

				x0 = lst(p0.op(0), p0.op(1), p0.op(2));
				x1 = lst(p1.op(0), p0.op(1), p0.op(2));
				x2 = lst(p1.op(0), p0.op(1), p1.op(2));
				x3 = lst(p0.op(0), p0.op(1), p1.op(2));

			}
			else if ( p0.op(2) == p1.op(2) )
			{

				x0 = lst(p0.op(0), p0.op(1), p0.op(2));
				x1 = lst(p1.op(0), p0.op(1), p0.op(2));
				x2 = lst(p1.op(0), p1.op(1), p0.op(2));
				x3 = lst(p0.op(0), p1.op(1), p0.op(2));
			}
		}
		else
		{
			throw(std::invalid_argument("The points p0 and p1 must be of dimention either 2 or 3."));
		}

		p[0] = x0;
		p[1] = x1;
		p[2] = x2;
		p[3] = x3;
	}

	Rectangle::Rectangle(ex p0, ex p1, ex p2, ex p3, const string & subscript_):
	Polygon(subscript_,4)
	{
		p[0] = p0;
		p[1] = p1;
		p[2] = p2;
		p[3] = p3;
	}

	Rectangle::Rectangle(const Rectangle& rectangle): Polygon(rectangle) { }

	Rectangle* Rectangle::copy() const
	{
		return new Rectangle(*this);
	}

	unsigned int Rectangle::no_space_dim() const
	{
		return 2;
	}

	Line Rectangle::line(unsigned int i) const
	{
		if      ( i == 0) return Line(p[0],p[1], istr(subscript,i));
		else if ( i == 1) return Line(p[1],p[2], istr(subscript,i));
		else if ( i == 2) return Line(p[2],p[3], istr(subscript,i));
		else if ( i == 3) return Line(p[3],p[0], istr(subscript,i));

		throw std::out_of_range("Line index is out of range!");
	}

	ex Rectangle::repr(Repr_format format) const
	{
		lst ret;
		GiNaC::symbol r("r"), s("s"), t("t");
		if ( p[0].nops() == 2 )
		{
			ret.append( x == p[0].op(0) + r*( p[2].op(0) - p[0].op(0)));
			ret.append( y == p[0].op(1) + s*( p[2].op(1) - p[0].op(1)));
			ret.append( lst(r,0,1) );
			ret.append( lst(s,0,1) );
		}
		else if ( p[0].nops() == 3 )
		{
			ret.append( x == p[0].op(0) + r*( p[2].op(0) - p[0].op(0)));
			ret.append( y == p[0].op(1) + s*( p[2].op(1) - p[0].op(1)));
			ret.append( z == p[0].op(2) + t*( p[2].op(2) - p[0].op(2)));
			ret.append( lst(r,0,1) );
			ret.append( lst(s,0,1) );
			ret.append( lst(t,0,1) );
		}
		return ret;
	}

	const string Rectangle::str() const
	{
		std::ostringstream s;
		//    s <<"Rectangle("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<")";
		s <<"Rectangle";
		return s.str();
	}

	ex Rectangle::integrate(ex func, Repr_format format)
	{

		unsigned int counter = 0;
		ex s_repr = repr();

		lst sub;
		// perform substitution
		if ( p[0].nops() == 3)
		{
			sub = lst(s_repr.op(0), s_repr.op(1), s_repr.op(2));
			counter = 3;
		}
		else if ( p[0].nops() == 2)
		{
			sub = lst(s_repr.op(0), s_repr.op(1));
			counter = 2;
		}

		ex D;
		if      ( p[0].nops() == 2)
		{
			D =   ( p[2].op(0) - p[0].op(0))
				*( p[2].op(1) - p[0].op(1));
		}
		else if ( p[0].nops() == 3 )
		{

			if      ( p[2].op(0) == p[0].op(0) )
			{
				D =   ( p[2].op(1) - p[0].op(1))
					*( p[2].op(2) - p[0].op(2));
			}
			else if ( p[2].op(1) == p[0].op(1) )
			{
				D =   ( p[2].op(0) - p[0].op(0))
					*( p[2].op(2) - p[0].op(2));
			}
			else if ( p[2].op(2) == p[0].op(2) )
			{
				D =   ( p[2].op(0) - p[0].op(0))
					*( p[2].op(1) - p[0].op(1));
			}
		}

		ex intf = func.subs(sub);

		intf = intf*D;

		intf = GiNaC::integral(s_repr.op(counter).op(0), s_repr.op(counter).op(1), s_repr.op(counter).op(2), intf);
		intf = eval_integ(intf);

		counter++;
		intf = GiNaC::integral(s_repr.op(counter).op(0), s_repr.op(counter).op(1), s_repr.op(counter).op(2), intf);
		intf = eval_integ(intf);

		counter++;
		if ( counter <  s_repr.nops() )
		{
			intf = GiNaC::integral(s_repr.op(counter).op(0), s_repr.op(counter).op(1), s_repr.op(counter).op(2), intf);
			intf = eval_integ(intf);
		}

		return intf;
	}

	// ---------- ReferenceRectangle --------------------------------------
	//

	ReferenceRectangle::ReferenceRectangle(const string & subscript_):
	Rectangle(subscript_)
	{
		p[0] = lst(0,0);
		p[1] = lst(1,0);
		p[2] = lst(1,1);
		p[3] = lst(0,1);
	}

	ReferenceRectangle::ReferenceRectangle(const ReferenceRectangle& rectangle): Rectangle(rectangle) { }

	ReferenceRectangle* ReferenceRectangle::copy() const
	{
		return new ReferenceRectangle(*this);
	}

	const string ReferenceRectangle::str() const
	{
		std::ostringstream s;
		//    s <<"ReferenceRectangle("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<")";
		s <<"ReferenceRectangle";
		return s.str();
	}

	ReferenceTriangle* ReferenceTriangle::copy() const
	{
		return new ReferenceTriangle(*this);
	}

	//---        Tetrahedron ------------------------------------

	Tetrahedron::Tetrahedron(ex x0, ex x1, ex x2, ex x3, const string & subscript_):
	Polygon(subscript_,4)
	{
		p[0] = x0;
		p[1] = x1;
		p[2] = x2;
		p[3] = x3;
	}

	Tetrahedron::Tetrahedron(const Tetrahedron& tetrahedron): Polygon(tetrahedron) { }

	unsigned int Tetrahedron::no_space_dim() const
	{
		return 3;
	}

	Line Tetrahedron::line(unsigned int i) const
	{
		int i0, i1;
		switch(i)
		{
			case 0:  i0 = 0; i1 = 1;  break;
			case 1:  i0 = 0; i1 = 2;  break;
			case 2:  i0 = 0; i1 = 3;  break;
			case 3:  i0 = 1; i1 = 2;  break;
			case 4:  i0 = 1; i1 = 3;  break;
			case 5:  i0 = 2; i1 = 3;  break;
			default:
				throw std::out_of_range("Line index is out of range!");
		}
		return Line(p[i0], p[i1], istr(subscript,i));
	}

	Triangle Tetrahedron::triangle(unsigned int i) const
	{

		if ( i == 0 )
		{
			return Triangle(p[1], p[2], p[3], istr(subscript,i));
		}
		else if ( i == 1)
		{
			return Triangle(p[0], p[2], p[3], istr(subscript,i));
		}
		else if ( i == 2)
		{
			return Triangle(p[0], p[1], p[3], istr(subscript,i));
		}
		else if ( i == 3)
		{
			return Triangle(p[0], p[1], p[2], istr(subscript,i));
		}

		throw std::out_of_range("Face index is out of range!");

	}

	ex Tetrahedron::repr(Repr_format format) const
	{
		GiNaC::symbol r("r"), s("s"), t("t");
		ex l1_repr = Line(vertex(0), vertex(1)).repr(r);
		ex l2_repr = Line(vertex(0), vertex(2)).repr(s);
		ex l3_repr = Line(vertex(0), vertex(3)).repr(t);
		lst ret;

		ret = lst(
			x == l1_repr.op(0).rhs().coeff(r,0)   + l1_repr.op(0).rhs().coeff(r,1)*r
			+    l2_repr.op(0).rhs().coeff(s,1)*s + l3_repr.op(0).rhs().coeff(t,1)*t,
			y == l1_repr.op(1).rhs().coeff(r,0)   + l1_repr.op(1).rhs().coeff(r,1)*r
			+    l2_repr.op(1).rhs().coeff(s,1)*s + l3_repr.op(1).rhs().coeff(t,1)*t,
			z == l1_repr.op(2).rhs().coeff(r,0)   + l1_repr.op(2).rhs().coeff(r,1)*r
			+    l2_repr.op(2).rhs().coeff(s,1)*s + l3_repr.op(2).rhs().coeff(t,1)*t);

		ret.append(lst(r, 0, 1));
		ret.append(lst(s, 0, 1 - r));
		ret.append(lst(t, 0, 1 - r - s));

		return ret;
	}

	const string Tetrahedron::str() const
	{
		std::ostringstream s;
		//    s <<"Tetrahedron("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<")";
		s <<"Tetrahedron";
		return s.str();
	}

	ex Tetrahedron::integrate(ex func, Repr_format format)
	{
		ex ret;

		if ( format == SUBS_PERFORMED )
		{
			ex t_repr = repr();
			//      std::cout <<"t "<<t_repr<<std::endl;

			//perform substitution
			lst sub = lst(t_repr.op(0), t_repr.op(1), t_repr.op(2));
			ex intf = func.subs(sub);

			// compute D
			ex D;
			ex r = t_repr.op(3).op(0);
			ex s = t_repr.op(4).op(0);
			ex t = t_repr.op(5).op(0);
			ex a = t_repr.op(0).rhs().coeff(r,1);
			ex b = t_repr.op(0).rhs().coeff(s,1);
			ex c = t_repr.op(0).rhs().coeff(t,1);
			ex d = t_repr.op(1).rhs().coeff(r,1);
			ex e = t_repr.op(1).rhs().coeff(s,1);
			ex f = t_repr.op(1).rhs().coeff(t,1);
			ex g = t_repr.op(2).rhs().coeff(r,1);
			ex h = t_repr.op(2).rhs().coeff(s,1);
			ex k = t_repr.op(2).rhs().coeff(t,1);

			D = a*(e*k-f*h) - b*(d*k-f*g) + c*(d*h - g*e);

			intf = intf*D;

			intf = GiNaC::integral(t_repr.op(5).op(0), t_repr.op(5).op(1), t_repr.op(5).op(2), intf);
			intf = GiNaC::eval_integ(intf);

			intf = GiNaC::integral(t_repr.op(4).op(0), t_repr.op(4).op(1), t_repr.op(4).op(2), intf);
			intf = GiNaC::eval_integ(intf);

			intf = GiNaC::integral(t_repr.op(3).op(0), t_repr.op(3).op(1), t_repr.op(3).op(2), intf);
			ret = GiNaC::eval_integ(intf);

		}
		else if ( format == SUBS_NOT_PERFORMED )
		{
			ex t_repr = repr();

			GiNaC::symbol a("a"), b("b"), c("c"), d("d"), e("e"), f("f"), g("g"), h("h"), k("k"), D("D");
			ex r = t_repr.op(3).op(0);
			ex s = t_repr.op(4).op(0);
			ex t = t_repr.op(5).op(0);

			//perform substitution
			//    lst sub = lst(t_repr.op(0), t_repr.op(1), t_repr.op(2));
			ex sub = lst(
				x == p[0].op(0) + a*r + b*s + c*t,
				y == p[0].op(1) + d*r + e*s + f*t,
				z == p[0].op(2) + g*r + h*s + k*t);

			ex intf = func.subs(sub);

			intf = GiNaC::integral(t_repr.op(5).op(0), t_repr.op(5).op(1), t_repr.op(5).op(2), intf);
			intf = GiNaC::eval_integ(intf);

			intf = GiNaC::integral(t_repr.op(4).op(0), t_repr.op(4).op(1), t_repr.op(4).op(2), intf);
			intf = GiNaC::eval_integ(intf);

			intf = GiNaC::integral(t_repr.op(3).op(0), t_repr.op(3).op(1), t_repr.op(3).op(2), intf);
			intf = GiNaC::eval_integ(intf);

			ret = intf*D;

		}
		else
		{
			throw std::runtime_error("Invalid format.");
		}

		return ret;
	}

	Tetrahedron* Tetrahedron::copy() const
	{
		return new Tetrahedron(*this);
	}

	//---        ReferenceTetrahedron ---------------------------
	//

	ReferenceTetrahedron::ReferenceTetrahedron(const string & subscript_):
	Tetrahedron(lst(0, 0, 0), lst(1, 0, 0), lst(0, 1, 0), lst(0, 0, 1), subscript_)
	{
	}

	ReferenceTetrahedron::ReferenceTetrahedron(const ReferenceTetrahedron& tetrahedron): Tetrahedron(tetrahedron) { }

	const string ReferenceTetrahedron::str() const
	{
		std::ostringstream s;
		//    s <<"ReferenceTetrahedron("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<")";
		s <<"ReferenceTetrahedron";
		return s.str();
	}

	ex ReferenceTetrahedron::integrate(ex f, Repr_format format)
	{

		ex intf = GiNaC::integral(x,0,1-y-z,f);
		intf = GiNaC::eval_integ(intf);

		intf = GiNaC::integral(y,0,1-z, intf);
		intf = GiNaC::eval_integ(intf);

		intf = GiNaC::integral(z,0,1, intf);
		intf = GiNaC::eval_integ(intf);

		return intf;
	}

	ReferenceTetrahedron* ReferenceTetrahedron::copy() const
	{
		return new ReferenceTetrahedron(*this);
	}

	// ---------- Box --------------------------------------
	//

	Box::Box(ex p0, ex p1, const string & subscript_):
	Polygon(subscript_, 8)
	{
		// Numbered by the UFC convention:
		p[0] = lst(p0.op(0), p0.op(1), p0.op(2));
		p[1] = lst(p1.op(0), p0.op(1), p0.op(2));
		p[2] = lst(p1.op(0), p1.op(1), p0.op(2));
		p[3] = lst(p0.op(0), p1.op(1), p0.op(2));

		p[4] = lst(p0.op(0), p0.op(1), p1.op(2));
		p[5] = lst(p1.op(0), p0.op(1), p1.op(2));
		p[6] = lst(p1.op(0), p1.op(1), p1.op(2));
		p[7] = lst(p0.op(0), p1.op(1), p1.op(2));
	}

	Box::Box(ex p0, ex p1, ex p2, ex p3, ex p4, ex p5, ex p6, ex p7, const string & subscript_):
	Polygon(subscript_, 8)
	{
		p[0] = p0;
		p[1] = p1;
		p[2] = p2;
		p[3] = p3;

		p[4] = p4;
		p[5] = p5;
		p[6] = p6;
		p[7] = p7;
	}

	Box::Box(const Box& box):
	Polygon(box)
	{
	}

	unsigned int Box::no_space_dim() const
	{
		return 3;
	}

	// Numbered by the UFC convention:
	Line Box::line(unsigned int i) const
	{
		int i0, i1;
		switch(i)
		{
			case  0:  i0 = 6; i1 = 7; break;
			case  1:  i0 = 5; i1 = 6; break;
			case  2:  i0 = 4; i1 = 7; break;
			case  3:  i0 = 4; i1 = 5; break;
			case  4:  i0 = 3; i1 = 7; break;
			case  5:  i0 = 2; i1 = 6; break;
			case  6:  i0 = 2; i1 = 3; break;
			case  7:  i0 = 1; i1 = 5; break;
			case  8:  i0 = 1; i1 = 2; break;
			case  9:  i0 = 0; i1 = 4; break;
			case 10:  i0 = 0; i1 = 3; break;
			case 11:  i0 = 0; i1 = 1; break;
			default:
				throw std::out_of_range("Line index is out of range!");
		}
		return Line(p[i0], p[i1], istr(subscript,i));
	}

	// Numbered by the UFC convention.
	Rectangle Box::rectangle(unsigned int i) const
	{
		switch(i)
		{
			case 0: return Rectangle(p[4], p[6], istr(subscript,i));
			case 1: return Rectangle(p[2], p[7], istr(subscript,i));
			case 2: return Rectangle(p[1], p[6], istr(subscript,i));
			case 3: return Rectangle(p[0], p[7], istr(subscript,i));
			case 4: return Rectangle(p[0], p[5], istr(subscript,i));
			case 5: return Rectangle(p[0], p[2], istr(subscript,i));
		}
		throw std::out_of_range("Rectangle index is out of range!");
	}

	ex Box::repr(Repr_format format) const
	{
		lst ret;
		GiNaC::symbol r("r"), s("s"), t("t");
		ret.append( x == p[0].op(0) + r * (p[6].op(0) - p[0].op(0)) );
		ret.append( y == p[0].op(1) + s * (p[6].op(1) - p[0].op(1)) );
		ret.append( z == p[0].op(2) + t * (p[6].op(2) - p[0].op(2)) );
		ret.append( lst(r,0,1) );
		ret.append( lst(s,0,1) );
		ret.append( lst(t,0,1) );
		return ret;
	}

	const string Box::str() const
	{
		std::ostringstream s;
		//    s <<"Box("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<","<<p[4]<<","<<p[5]<<","<<p[6]<<","<<p[7]<<")";
		s <<"Box";
		return s.str();
	}

	ex Box::integrate(ex func, Repr_format format)
	{

		ex b_repr = repr();

		lst sub;
		sub = lst(b_repr.op(0), b_repr.op(1), b_repr.op(2));
		ex intf = func.subs(sub);

		ex D =   ( p[6].op(0) - p[0].op(0) )
			* ( p[6].op(1) - p[0].op(1) )
			* ( p[6].op(2) - p[0].op(2) );

		intf = intf*D;

		intf = GiNaC::integral(b_repr.op(3).op(0), b_repr.op(3).op(1), b_repr.op(3).op(2), intf);
		intf = eval_integ(intf);

		intf = GiNaC::integral(b_repr.op(4).op(0), b_repr.op(4).op(1), b_repr.op(4).op(2), intf);
		intf = eval_integ(intf);

		intf = GiNaC::integral(b_repr.op(5).op(0), b_repr.op(5).op(1), b_repr.op(5).op(2), intf);
		intf = eval_integ(intf);

		return intf;
	}

	Box* Box::copy() const
	{
		return new Box(*this);
	}

	// ---------- ReferenceBox --------------------------------------

	ReferenceBox::ReferenceBox(const string & subscript_):
	Box(lst(0,0,0), lst(1,1,1), subscript_)
	{
	}

	ReferenceBox::ReferenceBox(const ReferenceBox& box):
	Box(box)
	{
	}

	const string ReferenceBox::str() const
	{
		std::ostringstream s;
		//    s <<"ReferenceBox("<<p[0]<<","<<p[1]<<","<<p[2]<<","<<p[3]<<","<<p[4]<<","<<p[5]<<","<<p[6]<<","<<p[7]<<")";
		s <<"ReferenceBox";
		return s.str();
	}

	ReferenceBox* ReferenceBox::copy() const
	{
		return new ReferenceBox(*this);
	}

	// ---------- General Simplex --------------------------------------

	Simplex::Simplex(GiNaC::lst vertices, const string & subscript_):
	Polygon(subscript_,vertices.nops())
	{
		//FIXME does this work in 1D  ? Need to check!
		unsigned int space_dim = vertices.op(0).nops();
		for (unsigned int i=0; i<vertices.nops(); i++)
		{
			if (space_dim != vertices.op(i).nops())
			{
				p.clear();
				throw std::logic_error("Vertex dimension must be the same for all points!");
			}
			p[i] = vertices.op(i);
		}
	}

	Simplex::Simplex(const Simplex& simplex):
	Polygon(simplex)
	{
	}

	unsigned int Simplex::no_space_dim() const
	{
		return p[0].nops()-1;
	}

	ex Simplex::repr(Repr_format format) const
	{
		unsigned int nsd = vertex(0).nops();
		unsigned int no_lines = no_vertices()-1;

		ex r = get_symbolic_vector(nsd, "r");
		ex x = get_symbolic_vector(nsd, "x");
		ex ri;
		lst lines;
		for (unsigned int i=0; i< no_vertices()-1; i++)
		{
			ri = r.op(i);
			lst line_i_repr;
			for (unsigned int d=0; d< nsd; d++)
			{
				line_i_repr.append(x.op(d) == (vertex(i+1).op(d) - vertex(0).op(d))*ri + vertex(0).op(d));
			}
			line_i_repr.append(lst(ri, 0, 1));

			lines.append(line_i_repr);
		}

		lst ret;
		for (unsigned int i=0; i < nsd; i++)
		{
			ri = r.op(i);
			GiNaC::ex xi_expr;
			GiNaC::ex rhs  = lines.op(0).op(i).rhs().coeff(ri,0);
			for (unsigned int l=0; l < no_lines; l++)
			{
				//        xi_expr2 == xi_expr.lhs() == xi_expr.rhs()  + lines.op(l).op(i).rhs().coeff(ri,1)*ri;
				rhs += lines.op(l).op(i).rhs().coeff(ri,1)*ri;

			}
			xi_expr = x.op(i) == rhs;
			ret.append(xi_expr);
		}

		GiNaC::ex limit=1;
		for (unsigned int i=0; i< no_lines; i++)
		{
			ri = r.op(i);
			ret.append(lst(ri, 0, limit));
			limit -= ri;
		}

		return ret;
	}

	const string Simplex::str() const
	{
		std::ostringstream s;
		/*
		s <<"Simplex(";
		for (int i=0; i<p.size()-1; i++) {
		  s << p[i]<<",";
		}
		s << p[p.size()-1]<<")";
		*/
		s <<"Simplex";
		return s.str();
	}

	ex Simplex::integrate(ex func, Repr_format format)
	{
		ex ret;

		unsigned int nsd = vertex(0).nops();

		ex s_repr = repr();
		lst subs;
		for (unsigned int i=0; i< nsd; i++)
		{
			subs.append(s_repr.op(i));
		}

		ex intf = func.subs(subs);

		for (unsigned int i=s_repr.nops()-1; i>=nsd; i--)
		{
			intf = GiNaC::integral(s_repr.op(i).op(0), s_repr.op(i).op(1), s_repr.op(i).op(2), intf);
			intf = GiNaC::eval_integ(intf);
		}

		return intf;
	}

	Simplex Simplex::sub_simplex(unsigned int i)
	{
		if ( i < 0 || i >= p.size())
		{
			throw std::out_of_range("Can only create subsimplices between 0 and the number of vertices-1.");
		}
		lst v;
		for (unsigned int k=0; k< p.size(); k++)
		{
			if ( k != i)
			{
				v.append(p[k]);
			}
		}
		return Simplex(v, istr(subscript, i));
	}

	Simplex* Simplex::copy() const
	{
		return new Simplex(*this);
	}

	//--  tools for Polygons ------------------------------------------------------

	lst bezier_ordinates(Tetrahedron& tetrahedra, unsigned int d)
	{

		//FIXME: ugly conversion to matrix

		lst ret;
		ex V1 = tetrahedra.vertex(0);
		ex V2 = tetrahedra.vertex(1);
		ex V3 = tetrahedra.vertex(2);
		ex V4 = tetrahedra.vertex(3);

		lst V1l = ex_to<lst>(V1);
		lst V2l = ex_to<lst>(V2);
		lst V3l = ex_to<lst>(V3);
		lst V4l = ex_to<lst>(V4);

		ex V1m  = lst_to_matrix2(V1l);
		ex V2m  = lst_to_matrix2(V2l);
		ex V3m  = lst_to_matrix2(V3l);
		ex V4m  = lst_to_matrix2(V4l);

		int l;
		for (unsigned int i=0; i<= d; i++)
		{
			for (unsigned int j=0; j<= d; j++)
			{
				for (unsigned int k=0; k<= d; k++)
				{
					if ( d - i - j -k  >= 0 )
					{
						l= d - i - j -k;
						ex sum = (l*V1m + k*V2m + j*V3m + i*V4m)/d;
						ret.append(matrix_to_lst2(sum.evalm()));
					}
				}
			}
		}
		// FIXME how should these be sorted ?????
		//  ret = ret.sort();
		return ret;
	}

	lst interior_coordinates(Tetrahedron& tetrahedra, unsigned int d)
	{

		//FIXME: ugly conversion to matrix
		d = d+4;

		lst ret;
		ex V1 = tetrahedra.vertex(0);
		ex V2 = tetrahedra.vertex(1);
		ex V3 = tetrahedra.vertex(2);
		ex V4 = tetrahedra.vertex(3);

		lst V1l = ex_to<lst>(V1);
		lst V2l = ex_to<lst>(V2);
		lst V3l = ex_to<lst>(V3);
		lst V4l = ex_to<lst>(V4);

		ex V1m  = lst_to_matrix2(V1l);
		ex V2m  = lst_to_matrix2(V2l);
		ex V3m  = lst_to_matrix2(V3l);
		ex V4m  = lst_to_matrix2(V4l);

		int l;
		for (unsigned int i=1; i< d; i++)
		{
			for (unsigned int j=1; j< d; j++)
			{
				for (unsigned int k=1; k< d; k++)
				{
					if ( int(d) - int(i) - int(j) - int(k)  >= 1 )
					{
						l= int(d) - int(i) - int(j) - int(k);
						ex sum = (l*V1m + k*V2m + j*V3m + i*V4m)/d;
						ret.append(matrix_to_lst2(sum.evalm()));
					}
				}
			}
		}
		// FIXME how should these be sorted ?????
		//  ret = ret.sort();
		return ret;
	}

	lst bezier_ordinates(Triangle& triangle, unsigned int d)
	{

		//FIXME: ugly conversion to matrix

		lst ret;
		ex V1 = triangle.vertex(0);
		ex V2 = triangle.vertex(1);
		ex V3 = triangle.vertex(2);

		lst V1l = ex_to<lst>(V1);
		lst V2l = ex_to<lst>(V2);
		lst V3l = ex_to<lst>(V3);

		ex V1m  = lst_to_matrix2(V1l);
		ex V2m  = lst_to_matrix2(V2l);
		ex V3m  = lst_to_matrix2(V3l);

		int k;
		for (unsigned int i=0; i <= d; i++)
		{
			for (unsigned int j=0; j <= d; j++)
			{
				if ( int(d) - int(i) - int(j) >= 0  )
				{
					k = d - i - j;
					ex sum = (k*V1m + j*V2m + i*V3m)/d;
					ret.append(matrix_to_lst2(sum.evalm()));
				}
			}
		}
		// FIXME how should these be sorted ?????
		// ret = ret.sort();
		return ret;
	}

	lst interior_coordinates(Triangle& triangle, unsigned int d)
	{

		//FIXME: ugly conversion to matrix
		//
		d=d+3;

		lst ret;
		ex V1 = triangle.vertex(0);
		ex V2 = triangle.vertex(1);
		ex V3 = triangle.vertex(2);

		lst V1l = ex_to<lst>(V1);
		lst V2l = ex_to<lst>(V2);
		lst V3l = ex_to<lst>(V3);

		ex V1m  = lst_to_matrix2(V1l);
		ex V2m  = lst_to_matrix2(V2l);
		ex V3m  = lst_to_matrix2(V3l);

		int k;
		for (unsigned int i=1; i < d; i++)
		{
			for (unsigned int j=1; j < d; j++)
			{
				if ( int(d) - int(i) - int(j) >= 1  )
				{
					k = d - i - j;
					ex sum = (k*V1m + j*V2m + i*V3m)/d;
					ret.append(matrix_to_lst2(sum.evalm()));
				}
			}
		}
		// FIXME how should these be sorted ?????
		// ret = ret.sort();
		return ret;
	}

	lst bezier_ordinates(Line& line, unsigned int d)
	{

		lst ret;
		ex V1 = line.vertex(0);
		ex V2 = line.vertex(1);

		if (!GiNaC::is_a<lst>(V1))
		{
			int k;
			for (unsigned int i=0; i <= d; i++)
			{
				k = d - i;
				ex sum = (k*V1 + i*V2)/d;
				ret.append(sum);
			}
		}
		else
		{

			//FIXME: ugly conversion to matrix

			lst V1l = ex_to<lst>(V1);
			lst V2l = ex_to<lst>(V2);

			ex V1m  = lst_to_matrix2(V1l);
			ex V2m  = lst_to_matrix2(V2l);

			int k;
			for (unsigned int i=0; i <= d; i++)
			{
				k = d - i;
				ex sum = (k*V1m + i*V2m)/d;
				ret.append(matrix_to_lst2(sum.evalm()));
			}
			// FIXME how should these be sorted ?????
			// ret = ret.sort();
		}
		return ret;
	}

	lst interior_coordinates(Line& line, unsigned int d)
	{

		//FIXME: ugly conversion to matrix
		d = d+2;

		lst ret;
		ex V1 = line.vertex(0);
		ex V2 = line.vertex(1);

		lst V1l = ex_to<lst>(V1);
		lst V2l = ex_to<lst>(V2);

		ex V1m  = lst_to_matrix2(V1l);
		ex V2m  = lst_to_matrix2(V2l);

		int k;
		for (unsigned int i=1; i < d; i++)
		{
			k = d - i;
			ex sum = (k*V1m + i*V2m)/d;
			ret.append(matrix_to_lst2(sum.evalm()));
		}
		// FIXME how should these be sorted ?????
		// ret = ret.sort();
		return ret;
	}

	// FIXME barycenter_line, barycenter_triangle, barycenter_tetrahedron
	// all needs to determine which equations to use in a proper way

	ex barycenter_line(ex p0, ex p1)
	{
		ex sol;

		// 1D
		if (!GiNaC::is_a<lst>(p0))
		{
			GiNaC::symbol b0("b0"), b1("b1");
			ex eq1 = x == b0*p0 + b1*p1;
			ex eq2 = 1 == b0 + b1;
			sol = lsolve(lst(eq1, eq2), lst(b0, b1));
		}
		else if (p0.nops() == 1 && p1.nops() == 1)
		{
			GiNaC::symbol b0("b0"), b1("b1");
			ex eq1 = x == b0*p0.op(0) + b1*p1.op(0);
			ex eq2 = 1 == b0 + b1;
			sol = lsolve(lst(eq1, eq2), lst(b0, b1));
			if ( sol == 0 )
			{
				ex eq1 = y == b0*p0.op(1) + b1*p1.op(1);
				sol = lsolve(lst(eq1, eq2), lst(b0, b1));
			}
			if ( sol == 0 )
			{
				ex eq1 = z == b0*p0.op(2) + b1*p1.op(2);
				sol = lsolve(lst(eq1, eq2), lst(b0, b1));
			}
		}
		//2D
		else if ( p0.nops() == 2 && p1.nops() == 2 )
		{
			GiNaC::symbol b0("b0"), b1("b1");
			ex eq1 = x == b0*p0.op(0) + b1*p1.op(0);
			ex eq3 = 1 == b0 + b1;
			sol = lsolve(lst(eq1, eq3), lst(b0, b1));
			if (sol.nops() == 0)
			{
				ex eq2 = y == b0*p0.op(1) + b1*p1.op(1);
				sol = lsolve(lst(eq2, eq3), lst(b0, b1));
			}
		}
		//3D
		else if ( p0.nops() == 3 && p1.nops() == 3 )
		{
			GiNaC::symbol b0("b0"), b1("b1");
			ex eq1 = x == b0*p0.op(0) + b1*p1.op(0);
			ex eq4 = 1 == b0 + b1;
			sol = lsolve(lst(eq1, eq4), lst(b0, b1));
			if (sol.nops() == 0)
			{
				ex eq2 = y == b0*p0.op(1) + b1*p1.op(1);
				sol = lsolve(lst(eq2, eq4), lst(b0, b1));
			}
			if (sol.nops() == 0)
			{
				ex eq3 = z == b0*p0.op(2) + b1*p1.op(2);
				sol = lsolve(lst(eq3, eq4), lst(b0, b1));
			}
		}
		else
		{
			throw std::runtime_error("Could not compute the barycentric coordinates. Check the coordinates.");
		}

		return sol;
	}

	ex barycenter_triangle(ex p0, ex p1, ex p2)
	{
		ex sol;

		// 2D
		if ( p0.nops() == 2 && p1.nops() == 2 && p2.nops() == 2)
		{
			GiNaC::symbol b0("b0"), b1("b1"), b2("b2");
			ex eq1 = x == b0*p0.op(0) + b1*p1.op(0) + b2*p2.op(0);
			ex eq2 = y == b0*p0.op(1) + b1*p1.op(1) + b2*p2.op(1);
			ex eq3 = 1 == b0 + b1 + b2;

			sol = lsolve(lst(eq1, eq2, eq3), lst(b0, b1, b2));
		}
		// 3D
		else if ( p0.nops() == 3 && p1.nops() == 3 && p2.nops() == 3)
		{
			lst n1(p1.op(0) - p0.op(0),  p1.op(1) - p0.op(1), p1.op(2) - p0.op(2));
			lst n2 = lst(p2.op(0) - p0.op(0),  p2.op(1) - p0.op(1), p2.op(2) - p0.op(2));
			lst n = cross(n1,n2);

			lst midpoint = lst((p0.op(0) + p1.op(0) + p2.op(0))/3,
				(p0.op(1) + p1.op(1) + p2.op(1))/3,
				(p0.op(2) + p1.op(2) + p2.op(2))/3);

			ex p3 = lst(midpoint.op(0) + n.op(0),
				midpoint.op(1) + n.op(1),
				midpoint.op(2) + n.op(2));

			ex s = barycenter_tetrahedron(p0, p1, p2, p3);
			lst solution;
			for (unsigned int i=0; i<s.nops(); i++)
			{
				ex d = s.op(i).subs(x == p3.op(0)).subs(y == p3.op(1)).subs(z == p3.op(2));
				d = d.rhs();
				if ( GiNaC::is_a<GiNaC::numeric>(d))
				{
								 // FIXME: bad test, should use the toleranse variable set by CLN or something
					if ( GiNaC::abs(GiNaC::ex_to<GiNaC::numeric>(d)) < 10e-8)
					{
						solution.append(s.op(i));
					}
				}
				else
				{
					if ( d.is_zero() )
					{
						solution.append(s.op(i));
					}
				}
			}
			sol = solution;
		}
		else
		{
			throw std::runtime_error("Could not compute the barycentric coordinates. Check the coordinates.");
		}

		return sol;
	}

	ex barycenter_tetrahedron(ex p0, ex p1, ex p2, ex p3)
	{
		GiNaC::symbol b0("b0"), b1("b1"), b2("b2"), b3("b3");

		// 3D
		ex eq1 = x == b0*p0.op(0) + b1*p1.op(0) + b2*p2.op(0) + b3*p3.op(0);
		ex eq2 = y == b0*p0.op(1) + b1*p1.op(1) + b2*p2.op(1) + b3*p3.op(1);
		ex eq3 = z == b0*p0.op(2) + b1*p1.op(2) + b2*p2.op(2) + b3*p3.op(2);
		ex eq4 = 1 == b0 + b1 + b2 +b3;

		ex sol = lsolve(lst(eq1, eq2, eq3, eq4), lst(b0, b1, b2, b3));

		return sol;

	}

	ex barycenter(Simplex& simplex)
	{
		if (nsd != simplex.no_vertices()-1)
		{
			throw std::runtime_error("Could not compute the barycentric coordinates. Not implemented yet for simplices with no_vertices != nsd +1.");
		}

		// put symbols in lst
		ex b = get_symbolic_vector(simplex.no_vertices(), "b");
		lst symbols;
		for (unsigned int i=0; i<b.nops(); i++)
		{
			symbols.append(b.op(i));
		}

		// put equations in lst
		lst eqs;
		for (unsigned int i=0; i<nsd; i++)
		{
			ex sum = 0;
			for (unsigned int k=0; k< simplex.no_vertices(); k++)
			{
				sum += b.op(k)*simplex.vertex(k).op(i);
			}
			ex eqi = p[i] == sum;
			eqs.append(eqi);
		}

		// last eq, sum = 1
		ex sum = 0;
		for (unsigned int i=0; i<symbols.nops(); i++)
		{
			sum += symbols.op(i);
		}
		ex last_eq = 1 == sum;
		eqs.append(last_eq);

		// solve equations
		ex sol = lsolve(eqs, symbols);
		return sol;
	}

	ex bernstein(unsigned int order, Polygon& p, const string & a)
	{

		if ( order < 0 )
		{
			throw(std::logic_error("Can not create polynomials of order less than 0!"));
		}

		ex ret;					 // ex to return
		int dof;				 // degrees of freedom
		ex A;					 // ex holding the coefficients a_0 .. a_dof
		lst basis;

		if ( p.str().find("Line") != string::npos )
		{
			ex bary = barycenter_line(p.vertex(0), p.vertex(1));
			ex b0= bary.op(0).rhs();
			ex b1= bary.op(1).rhs();
			dof = order+1;
			A = get_symbolic_matrix(1,dof, a);
			int o=0;
			for (GiNaC::const_iterator i = A.begin(); i != A.end(); ++i)
			{
				ex scale = GiNaC::binomial(order,o);
				ret += (*i)*scale*pow(b0,o)*pow(b1,order-o);
				basis.append(scale*pow(b0,o)*pow(b1,order-o));
				o++;
			}
		}
		else if ( p.str().find("Triangle") != string::npos )
		{

			dof = (order+1)*(order+2)/2;
			A = get_symbolic_matrix(1, dof , a);

			ex bary = barycenter_triangle(p.vertex(0), p.vertex(1), p.vertex(2));
			ex b0= bary.op(0).rhs();
			ex b1= bary.op(1).rhs();
			ex b2= bary.op(2).rhs();

			size_t i=0;
			for (unsigned int o1 = 0; o1 <= order; o1++)
			{
				for (unsigned int o2 = 0; o2 <= order; o2++)
				{
					for (unsigned int o3 = 0; o3 <= order; o3++)
					{
						if ( o1 + o2 + o3 == order )
						{
							ex scale = (GiNaC::factorial(order)/(GiNaC::factorial(o1)*GiNaC::factorial(o2)*GiNaC::factorial(o3)));
							ret += A.op(i)*scale*pow(b0,o1)*pow(b1,o2)*pow(b2,o3);

							basis.append(scale*pow(b0,o1)*pow(b1,o2)*pow(b2,o3));
							i++;
						}
					}
				}
			}
		}

		else if ( p.str().find("Tetrahedron") != string::npos )
		{

			dof = 0;
			for (unsigned int j=0; j<= order; j++)
			{
				dof += (j+1)*(j+2)/2;
			}
			A = get_symbolic_matrix(1, dof , a);

			ex bary = barycenter_tetrahedron(p.vertex(0), p.vertex(1), p.vertex(2), p.vertex(3));
			ex b0= bary.op(0).rhs();
			ex b1= bary.op(1).rhs();
			ex b2= bary.op(2).rhs();
			ex b3= bary.op(3).rhs();

			size_t i=0;
			for (unsigned int o1 = 0; o1 <= order; o1++)
			{
				for (unsigned int o2 = 0; o2 <= order; o2++)
				{
					for (unsigned int o3 = 0; o3 <= order; o3++)
					{
						for (unsigned int o4 = 0; o4 <= order; o4++)
						{
							if ( o1 + o2 + o3 + o4 == order )
							{
								ex scale = (GiNaC::factorial(order)/(GiNaC::factorial(o1)*GiNaC::factorial(o2)*GiNaC::factorial(o3)*GiNaC::factorial(o4)));
								ret += A.op(i)*scale*pow(b0,o1)*pow(b1,o2)*pow(b2,o3)*pow(b3,o4);
								basis.append(scale*pow(b0,o1)*pow(b1,o2)*pow(b2,o3)*pow(b3,o4));
								i++;
							}
						}
					}
				}
			}
		}

		else if (p.str() == "Simplex" || p.str() == "ReferenceSimplex")
		{

			throw std::runtime_error("Not implemented yet.");
			//      ex bary = barycenter(p);
		}
		return lst(ret,matrix_to_lst2(A),basis);
	}

	lst bernsteinv(unsigned int no_fields, unsigned int order, Polygon& p, const string & a)
	{

		if ( order < 0 )
		{
			throw(std::logic_error("Can not create polynomials of order less than 0!"));
		}

		lst ret1;				 // contains the polynom
		lst ret2;				 // contains the coefficients
		lst ret3;				 // constains the basis functions
		lst basis_tmp;
		for (unsigned int i=0; i< no_fields; i++)
		{
			lst basis;
			std::ostringstream s;
			s <<a<<""<<i<<"_";
			ex pol = bernstein(order, p, s.str());
			ret1.append(pol.op(0));
			ret2.append(pol.op(1));
			basis_tmp = ex_to<lst>(pol.op(2));
			for (lst::const_iterator basis_iterator = basis_tmp.begin();
				basis_iterator != basis_tmp.end(); ++basis_iterator)
			{
				lst tmp_lst;
				for (unsigned int d=1; d<=no_fields; d++) tmp_lst.append(0);
				tmp_lst.let_op(i) = (*basis_iterator);
				ret3.append(tmp_lst);
			}
		}
		return lst(ret1,ret2,ret3);

	}

	lst normal(Tetrahedron& tetrahedron, unsigned int i)
	{
		// Normal as defined by Maries note
		Triangle triangle = tetrahedron.triangle(i);
		lst      vertex_i   = ex_to<lst>(tetrahedron.vertex(i));
		lst      vertex_0   = ex_to<lst>(triangle.vertex(0));
		lst      vertex_1   = ex_to<lst>(triangle.vertex(1));
		lst      vertex_2   = ex_to<lst>(triangle.vertex(2));

		lst n1(vertex_1.op(0) - vertex_0.op(0),
			vertex_1.op(1) - vertex_0.op(1),
			vertex_1.op(2) - vertex_0.op(2));

		lst n2(vertex_2.op(0) - vertex_0.op(0),
			vertex_2.op(1) - vertex_0.op(1),
			vertex_2.op(2) - vertex_0.op(2));

		/*
		lst n3(vertex_0.op(0) - vertex_i.op(0),
			   vertex_0.op(1) - vertex_i.op(1),
			   vertex_0.op(2) - vertex_i.op(2));
		*/

		lst n4 = cross(n1,n2);
		/*
		ex nn = inner(n3, n4);
		int sign = 1;
		if ( is_a<numeric>(nn)) {
		  if ( nn > 0 ) {
			sign = 1;
		} else if ( nn < 0) {
		sign = -1;
		} else {
		sign = 0;
		}
		}
		*/

		ex norm = sqrt(pow(n4.op(0),2)
			+ pow(n4.op(1),2)
			+ pow(n4.op(2),2));

		n4.let_op(0) = n4.op(0)/norm;
		n4.let_op(1) = n4.op(1)/norm;
		n4.let_op(2) = n4.op(2)/norm;

		return n4;

	}

	lst normal(Triangle& triangle, unsigned int i)
	{
		Line line = triangle.line(i);
		lst      vertex_i   = ex_to<lst>(triangle.vertex(i));
		lst      vertex_0   = ex_to<lst>(line.vertex(0));
		lst      vertex_1   = ex_to<lst>(line.vertex(1));

		/*
		lst n1 = lst (- (vertex_1.op(1) - vertex_0.op(1)),  vertex_1.op(0) - vertex_0.op(0) );
		lst n2 = lst (vertex_0.op(0) - vertex_i.op(0),   vertex_0.op(1) - vertex_i.op(1));

		ex nn = inner(n1, n2);
		int sign = 1;
		/ *
			if ( is_a<numeric>(nn)) {
			  if ( nn > 0 ) {
				sign = 1;
			  } else if ( nn < 0) {
				sign = -1;
		} else {
		sign = 0;
		}
		}

		ex norm = sqrt(pow(n1.op(0),2) + pow(n1.op(1),2));
		n1.let_op(0) = sign*n1.op(0)/norm;
		n1.let_op(1) = sign*n1.op(1)/norm;
		*/

		// normal vector as Marie has defined them
		lst n1 = lst (  (vertex_1.op(1) - vertex_0.op(1)),
			-(vertex_1.op(0) - vertex_0.op(0)) );

		ex norm = sqrt(pow(n1.op(0),2) + pow(n1.op(1),2));
		n1.let_op(0) = n1.op(0)/norm;
		n1.let_op(1) = n1.op(1)/norm;

		return n1;
	}

	lst tangent(Triangle& triangle, unsigned int i)
	{
		/*
		Line line = triangle.line(i);
		//FIXME: 5 lines to compute the tangent vector, these should
		// be put somewhere else.
		GiNaC::symbol t("t");
		ex line_repr = line.repr(t);
		ex t1 = line_repr.op(0).rhs().coeff(t,1);
		ex t2 = line_repr.op(1).rhs().coeff(t,1);
		ex norm = sqrt(pow(t1,2) + pow(t2,2));
		lst tangent = lst(t1/norm,t2/norm);
		return tangent;
		*/
		/*
		ex t1, t2;
		if ( i == 0 ) {
		  t1 = triangle.vertex(2).op(0) - triangle.vertex(1).op(0);
		  t2 = triangle.vertex(2).op(1) - triangle.vertex(1).op(1);
		} else if ( i == 1 ) {
		t1 = triangle.vertex(0).op(0) - triangle.vertex(2).op(0);
		t2 = triangle.vertex(0).op(1) - triangle.vertex(2).op(1);
		} else if ( i == 2 ) {
		t1 = triangle.vertex(1).op(0) - triangle.vertex(0).op(0);
		t2 = triangle.vertex(1).op(1) - triangle.vertex(0).op(1);
		} else {
		throw(std::out_of_range("The side index is out of range!"));
		}
		*/
		Line line = triangle.line(i);
		ex t1 = line.vertex(1).op(0) - line.vertex(0).op(0);
		ex t2 = line.vertex(1).op(1) - line.vertex(0).op(1);

		ex norm = sqrt(pow(t1,2) + pow(t2,2));
		lst tangent = lst(t1/norm,t2/norm);
		return tangent;

	}

}								 //namespace SyFi
